﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{HidNpadIntegrate_Main.cpp,PageSampleHidNpadIntegrate}

    @brief
    パッド、振動、6 軸センサの機能を切り替えて動作させるサンプルプログラム
*/

/**
    @page PageSampleHidNpadIntegrate Npad の複数の機能を切り替えて実行
    @tableofcontents

    @brief
    Npad の複数の機能を切り替えて実行するためのサンプルプログラムの解説です。

    @image html Applications\HidNpadIntegrate\HidNpadIntegrate.png

    @section PageSampleHidNpadIntegrate_SectionBrief 概要
    Npad の複数の機能を切り替えて実行する方法について説明します。

    @section PageSampleHidNpadIntegrate_SectionFileStructure ファイル構成
    本サンプルプログラムは
    @link ../../../Samples/Sources/Applications/HidNpadIntegrate
    Samples/Sources/Applications/HidNpadIntegrate @endlink 以下にあります。

    @section PageSampleHidNpadIntegrate_SectionNecessaryEnvironment 必要な環境
    事前に実機とコントローラをペアリングするか、Joy-Con を接続してください。

    @section PageSampleHidNpadIntegrate_SectionHowToOperate 操作方法
    サンプルプログラムを実行して、コントローラのボタンを押してください。
    コントローラの([Plus + R] or [Plus + ZR])でシーンが切り替わります。
    また、([Minus + L] or [Minus + ZL])や左右スワイプ操作でも切り替えられます。

    Npad の接続状態を変更するには([ZR + R Stick] or [ZL + L Stick])を押してください。
    操作形態の変更画面に切り替わり、各種設定を行えます。
    再度([ZR + R Stick] or [ZL + L Stick])を押すことで、元のシーンに戻ります。

    他のシーンを表示中に、6 軸センサの計算処理を
    任意のタイミングで切り替えることができます。
    切り替えるには以下のキーを押してください。
      6 軸センサ：([Right + L Stick] or [A + R Stick])
    ただし、該当のシーンに切り替わった時には
    自動的に計算処理が On に切り替わります。

    サンプルプログラムを終了させるには([Plus + R Stick] or [Minus + L Stick])を押してください。
    もしくはNintendo Target Manager経由で終了させてください。

    @section PageSampleHidNpadIntegrate_SectionPrecaution 注意事項
    コントローラは十分に充電した状態でお使いください。

    @section PageSampleHidNpadIntegrate_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleHidNpadIntegrate_SectionDetail 解説
    サンプルプログラムは以下のシーンから構成されます。
    - パッドの入力状態の表示
    - 振動子の制御
    - 6 軸センサの状態の表示
    - Npad の接続状態を変更する画面

    シーンの切り替えについては操作方法を参照してください
*/

#include "HidNpadIntegrate_Gesture.h"
#include "HidNpadIntegrate_Main.h"
#include "HidNpadIntegrate_Pad.h"
#include "HidNpadIntegrate_PluginManager.h"
#include "HidNpadIntegrate_SettingScreen.h"

const nn::hid::NpadIdType NpadIds[] = {nn::hid::NpadId::No1,
                                 nn::hid::NpadId::No2,
                                 nn::hid::NpadId::No3,
                                 nn::hid::NpadId::No4,
                                 nn::hid::NpadId::Handheld,
                                 };
const int NpadIdCountMax = NN_ARRAY_SIZE(NpadIds);

const int FrameBufferWidth = 1280;
const int FrameBufferHeight = 720;

namespace
{

    FontSystem*     g_pFontSystem;
    GraphicsSystem* g_pGraphicsSystem;

    ApplicationHeap& GetApplicationHeap() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(ApplicationHeap, s_ApplicationHeap);
        return s_ApplicationHeap;
    }

    bool g_IsMainContinued;
    bool g_IsSettingScreenOn;

    const char ProgramName[] = "HidNpadIntegrate Sample";

    void InitializeGraphics() NN_NOEXCEPT
    {
        const size_t ApplicationHeapSize = 128 * 1024 * 1024;
        GetApplicationHeap().Initialize(ApplicationHeapSize);
        g_pGraphicsSystem = new ::GraphicsSystem();
        g_pGraphicsSystem->Initialize(
            &GetApplicationHeap(), FrameBufferWidth, FrameBufferHeight);

        EnableWindowMessage(g_pGraphicsSystem->GetNativeWindowHandle());

        g_pFontSystem = new FontSystem();
        g_pFontSystem->Initialize(&GetApplicationHeap(), g_pGraphicsSystem);
    }

    void FinalizeGraphics() NN_NOEXCEPT
    {
        g_pFontSystem->Finalize();
        delete g_pFontSystem;

        g_pGraphicsSystem->Finalize();
        delete g_pGraphicsSystem;

        GetApplicationHeap().Finalize();
    }

    void Initialize() NN_NOEXCEPT
    {
        nn::hid::InitializeNpad();
        nn::hid::SetSupportedNpadStyleSet(GetPluginManager().GetAvailableNpadStyleSet());
        nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);

        nn::hid::InitializeGesture();

        GetPluginManager().InitializePlatformPlugin();
        InitializeGraphics();

        PadDemoScene* pNpadDemo = new PadDemoScene();
        pNpadDemo->SetName("Pad Demo");
        pNpadDemo->SetRunnableAlways(false);
        pNpadDemo->SetStatusName("PAD");
        GetPluginManager().AddPluginToHead(pNpadDemo);

        GetPluginManager().InitializeScenePlugin(&GetApplicationHeap(), g_pGraphicsSystem);
        GetPluginManager().InitializeNpadPlugin();
    }

    void Finalize() NN_NOEXCEPT
    {
        GetPluginManager().FinalizeNpadPlugin();
        GetPluginManager().FinalizeScenePlugin(&GetApplicationHeap());
        FinalizeGraphics();
        GetPluginManager().FinalizePlatformPlugin();
    }

    void ChangeActiveScene() NN_NOEXCEPT
    {
        const int CurrentSceneNum = GetPluginManager().GetCurrentSceneNum();
        const int OldSceneNum     = GetPluginManager().GetOldSceneNum();

        GetPluginManager().GetScene(OldSceneNum)->SwitchScene();
        GetPluginManager().GetScene(CurrentSceneNum)->InitializeScene(&GetApplicationHeap(),
                                                                          g_pGraphicsSystem);
    }

    void ChangeSettingScreen() NN_NOEXCEPT
    {
        g_IsSettingScreenOn = !g_IsSettingScreenOn;
        if (g_IsSettingScreenOn)
        {
            GetPluginManager().StopScenePlugin();
            InitializeSettingScreen();
        }
        else
        {
            FinalizeSettingScreen();
            GetPluginManager().RestartScenePlugin();
        }
    }

    void CheckPushButton(const nn::hid::NpadButtonSet Buttons,
                            GestureManager& gestureManager) NN_NOEXCEPT
    {
        const int ChengeWaitTime = 500;

        // モード設定画面がOFFの場合のみ有効
        if (!g_IsSettingScreenOn)
        {
            // 次のシーンに進む
            if ((Buttons.Test<nn::hid::NpadButton::Plus>() &&
                 Buttons.Test<nn::hid::NpadButton::R>()) ||
                (Buttons.Test<nn::hid::NpadButton::Minus>() &&
                 Buttons.Test<nn::hid::NpadButton::L>()) ||
                (gestureManager.GetSwipeType() == SwipeType_Left))
            {
                GetPluginManager().NextScene();
                ChangeActiveScene();
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ChengeWaitTime));
            }
            // 前のシーンに戻る
            if ((Buttons.Test<nn::hid::NpadButton::Plus>() &&
                 Buttons.Test<nn::hid::NpadButton::ZR>()) ||
                (Buttons.Test<nn::hid::NpadButton::Minus>() &&
                 Buttons.Test<nn::hid::NpadButton::ZL>()) ||
                (gestureManager.GetSwipeType() == SwipeType_Right))
            {
                GetPluginManager().PreviousScene();
                ChangeActiveScene();
                ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ChengeWaitTime));
            }
            gestureManager.ClearSwipeType();

            // 計算処理のON/OFFを切り替えるかチェックする
            for(int i = 0; i < GetPluginManager().GetScenePluginCount(); i++)
            {
                GetPluginManager().GetScene(i)->ToggleProcess(Buttons,
                                                                  &GetApplicationHeap(),
                                                                  g_pGraphicsSystem);
            }
        }
        // サンプルアプリを終了する
        if ((Buttons.Test<nn::hid::NpadButton::Plus>() &&
             Buttons.Test<nn::hid::NpadButton::StickR>()) ||
            (Buttons.Test<nn::hid::NpadButton::Minus>() &&
             Buttons.Test<nn::hid::NpadButton::StickL>()))
        {
            g_IsMainContinued = false;
        }

        // モード設定画面のON/OFFを切り替える
        if ((Buttons.Test<nn::hid::NpadButton::ZR>() &&
             Buttons.Test<nn::hid::NpadButton::StickR>()) ||
            (Buttons.Test<nn::hid::NpadButton::ZL>() &&
             Buttons.Test<nn::hid::NpadButton::StickL>()))
        {
            ChangeSettingScreen();
            ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ChengeWaitTime));
        }
    }

    void CheckChangeDemo(const nn::hid::NpadIdType NpadId,
                            GestureManager& gestureManager) NN_NOEXCEPT
    {
        // ボタンの入力状態を取得
        NpadCommonState buttonState = GetNpadButtonSet(NpadId);
        CheckPushButton(buttonState.buttons, gestureManager);
    }

    void WriteCursor(
        nn::gfx::util::DebugFontTextWriter* pTextWriter,
        int32_t x, int32_t y) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pTextWriter);

        const float barOffsetX   = pTextWriter->CalculateStringWidth("|") / 2;
        const float barOffsetY   = pTextWriter->CalculateStringHeight("|") / 2;
        const float minusOffsetX = pTextWriter->CalculateStringWidth("-") / 2;
        const float minusOffsetY = pTextWriter->CalculateStringHeight("-") / 2;
        const float plusOffsetX  = pTextWriter->CalculateStringWidth("+") / 2;
        const float plusOffsetY  = pTextWriter->CalculateStringHeight("+") / 2;

        pTextWriter->SetCursor(x - barOffsetX - 45, y - barOffsetY);
        pTextWriter->Print("|");
        pTextWriter->SetCursor(x - barOffsetX + 45, y - barOffsetY);
        pTextWriter->Print("|");
        pTextWriter->SetCursor(x - minusOffsetX, y - minusOffsetY - 45);
        pTextWriter->Print("-");
        pTextWriter->SetCursor(x - minusOffsetX, y - minusOffsetY + 45);
        pTextWriter->Print("-");
        pTextWriter->SetCursor(x - plusOffsetX - 45, y - plusOffsetY - 45);
        pTextWriter->Print("+");
        pTextWriter->SetCursor(x - plusOffsetX - 45, y - plusOffsetY + 45);
        pTextWriter->Print("+");
        pTextWriter->SetCursor(x - plusOffsetX + 45, y - plusOffsetY - 45);
        pTextWriter->Print("+");
        pTextWriter->SetCursor(x - plusOffsetX + 45, y - plusOffsetY + 45);
        pTextWriter->Print("+");
    }

    void WritePoints(
        nn::gfx::util::DebugFontTextWriter* pTextWriter,
        const std::vector<nn::hid::GesturePoint>& points) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pTextWriter);

        pTextWriter->SetTextColor(Color::Yellow);

        for (size_t i = 0; i < points.size(); ++i)
        {
            WriteCursor(pTextWriter, points[i].x, points[i].y);
        }
    }

} // namespace

void WriteCommonGuide(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
{
    const float OffsetX = 405;
    const float OffsetY = 0;
    const int ActiveSceneNum = GetPluginManager().GetCurrentSceneNum();

    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetScale(1.5f, 1.5f);
    pTextWriter->SetCursor(10, 10);
    pTextWriter->Print("%s", GetPluginManager().GetScene(ActiveSceneNum)->GetName());

    pTextWriter->SetTextColor(Color::Aqua);
    pTextWriter->SetScale(0.9f, 0.9f);
    pTextWriter->SetCursor(OffsetX, OffsetY);
    pTextWriter->Print("Next Demo");
    pTextWriter->SetCursor(OffsetX, OffsetY + 15);
    pTextWriter->Print("Previous Demo");
    pTextWriter->SetCursor(OffsetX + 470, OffsetY);
    pTextWriter->Print("Mode Settings");
    pTextWriter->SetCursor(OffsetX + 470, OffsetY + 15);
    pTextWriter->Print("Finish Demo");

    pTextWriter->SetCursor(OffsetX + 150, OffsetY);
    pTextWriter->Print(": Minus + L ( Plus + R ) or Left Swipe");
    pTextWriter->SetCursor(OffsetX + 150, OffsetY + 15);
    pTextWriter->Print(": Minus + ZL ( Plus + ZR ) or Right Swipe");
    pTextWriter->SetCursor(OffsetX + 580, OffsetY);
    pTextWriter->Print(": ZL + L Stick ( ZR + R Stick )");
    pTextWriter->SetCursor(OffsetX + 580, OffsetY + 15);
    pTextWriter->Print(": Minus + L Stick ( Plus + R Stick )");

    int count = 0;

    for(int i = 0; i < GetPluginManager().GetScenePluginCount(); i++)
    {
        if(GetPluginManager().GetScene(i)->IsRunnableAlways())
        {
            pTextWriter->SetTextColor(Color::White);
            pTextWriter->SetCursor(OffsetX - 115, OffsetY + count * 20);
            pTextWriter->Print("%s", GetPluginManager().GetScene(i)->GetStatusName());
            pTextWriter->SetCursor(OffsetX - 80, OffsetY + count * 20);
            pTextWriter->Print(":");
            pTextWriter->SetCursor(OffsetX - 70, OffsetY + count * 20);
            pTextWriter->SetTextColor(GetPluginManager().GetScene(i)
                                              ->IsRunning() ? Color::Orange : Color::Lime);
            pTextWriter->Print(GetPluginManager().GetScene(i)
                                              ->IsRunning() ? "On" : "Off");

            pTextWriter->SetTextColor(Color::Aqua);
            pTextWriter->SetCursor(OffsetX + count * 470, OffsetY + 30);
            pTextWriter->Print("%s  On / Off", GetPluginManager().GetScene(i)
                                              ->GetStatusName());
            pTextWriter->SetCursor(OffsetX + 150 + 430 * count, OffsetY + 30);
            pTextWriter->Print(": %s", GetPluginManager().GetScene(i)->GetToggleOperation());

            count++;
        }
    }

}

void DrawRect(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                const float OffsetX, const float OffsetY,
                const int Width, const int Height) NN_NOEXCEPT
{
    const float Unit = 8.0f;
    pTextWriter->SetScale(1.0f, 1.0f);
    pTextWriter->SetTextColor(Color::Aqua);

    for (int y = 1; y <= Height; y += 2)
    {
        pTextWriter->SetCursor(OffsetX - 3, OffsetY + Unit * y);
        pTextWriter->Print("|");
        pTextWriter->SetCursor(OffsetX + 3 + Unit * Width, OffsetY + Unit * y);
        pTextWriter->Print("|");
    }

    for (int x = 0; x <= Width; x += 2)
    {
        pTextWriter->SetCursor(OffsetX + Unit * x, OffsetY);
        pTextWriter->Print("-");
        pTextWriter->SetCursor(OffsetX + Unit * x, OffsetY + Unit * Height);
        pTextWriter->Print("-");
    }
}

NpadCommonState GetNpadButtonSet(const nn::hid::NpadIdType NpadId) NN_NOEXCEPT
{
    //現在有効な操作形態(NpadStyleSet)を取得
    nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadId);
    NpadCommonState buttonState ={};

    NpadStylePluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
    if(pNpad != NULL)
    {
        // ボタンの入力状態を取得
        buttonState = pNpad->GetNpadButtonState(NpadId);
    }

    return buttonState;
}

extern "C" void nnMain()
{
    NN_LOG("%s Start\n", ProgramName);
    GestureManager gestureManager;

    Initialize();

    nn::gfx::util::DebugFontTextWriter& textWriter =
        g_pFontSystem->GetDebugFontTextWriter();

    const int FrameRate = 60;
    g_IsMainContinued = true;
    g_IsSettingScreenOn = false;
    while(g_IsMainContinued)
    {
        switch (GetWindowMessage(g_pGraphicsSystem->GetNativeWindowHandle()))
        {
        case WindowMessage_Close:
            g_IsMainContinued = false;
            break;

        default:
            break;
        }

        gestureManager.Update();

        // コントローラIDの数だけループ
        for(int i = 0 ; i < NpadIdCountMax ; i++)
        {
            // ボタンが押されたらデモを切り替える。
            CheckChangeDemo(NpadIds[i], gestureManager);
        }

        g_pGraphicsSystem->BeginDraw();

        if (g_IsSettingScreenOn)
        {
            // 設定画面メイン処理
            SettingScreenMain(textWriter);
        }
        else
        {
            const int ActiveSceneNum = GetPluginManager().GetCurrentSceneNum();
            GetPluginManager().GetScene(ActiveSceneNum)->RunScene(textWriter,
                                                             g_pGraphicsSystem);
        }

        WritePoints(&textWriter, gestureManager.GetPoints());

        // 描画を行う
        textWriter.Draw(&g_pGraphicsSystem->GetCommandBuffer());
        g_pGraphicsSystem->EndDraw();

        g_pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));
    }

    Finalize();

    NN_LOG("%s End\n", ProgramName);
}
