﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <sstream>

#include <nn/nn_Assert.h>

#include "HidNpadIntegrate_IrSensorImageTransferModeState.h"

namespace
{

    struct ImageProperties
    {
        int         width;
        int         height;
        int         size;
        int         workBufferSize;
        const char* name;
    };

    NN_STATIC_ASSERT(nn::irsensor::ImageTransferProcessorFormat_320x240 == 0);
    NN_STATIC_ASSERT(nn::irsensor::ImageTransferProcessorFormat_160x120 == 1);
    NN_STATIC_ASSERT(nn::irsensor::ImageTransferProcessorFormat_80x60 == 2);

    const ImageProperties Properties[] ={
        {
            nn::irsensor::IrCameraImageWidth,
            nn::irsensor::IrCameraImageHeight,
            nn::irsensor::ImageTransferProcessorImageSize320x240,
            nn::irsensor::ImageTransferProcessorWorkBufferSize320x240,
            "320x240",
        },
        {
            nn::irsensor::IrCameraImageWidth >> nn::irsensor::ImageTransferProcessorFormat_160x120,
            nn::irsensor::IrCameraImageHeight >> nn::irsensor::ImageTransferProcessorFormat_160x120,
            nn::irsensor::ImageTransferProcessorImageSize160x120,
            nn::irsensor::ImageTransferProcessorWorkBufferSize160x120,
            "160x120",
        },
        {
            nn::irsensor::IrCameraImageWidth >> nn::irsensor::ImageTransferProcessorFormat_80x60,
            nn::irsensor::IrCameraImageHeight >> nn::irsensor::ImageTransferProcessorFormat_80x60,
            nn::irsensor::ImageTransferProcessorImageSize80x60,
            nn::irsensor::ImageTransferProcessorWorkBufferSize80x60,
            "80x60",
        }
    };

} // namespace

class Format : public ReadWriteBase
{
    NN_DISALLOW_COPY(Format);
    NN_DISALLOW_MOVE(Format);

public:
    NN_IMPLICIT Format(
        nn::irsensor::ImageTransferProcessorConfig* pImageTransferConfig) NN_NOEXCEPT
        : m_pImageTransferConfig(pImageTransferConfig)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << Properties[m_pImageTransferConfig->format].name;
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        int8_t format =
            static_cast<int8_t>(m_pImageTransferConfig->format) + static_cast<int8_t>(delta);
        format = std::max(
            std::min(format,
                static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_80x60)),
            static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_320x240)
        );
        m_pImageTransferConfig->format =
            static_cast<nn::irsensor::ImageTransferProcessorFormat>(format);
    }

private:
    nn::irsensor::ImageTransferProcessorConfig* m_pImageTransferConfig;
};

ImageTransferModeState::ImageTransferModeState(IrSensorMode* pNextProcessor,
    int* pMenuSelection, nn::irsensor::IrCameraHandle irCameraHandle,
    void* pWorkMemory,
    GraphicsSystem *pGraphicsSystem) NN_NOEXCEPT
    : IrSensorModeState(pNextProcessor, pMenuSelection, irCameraHandle)
    , m_Format()
    , m_ImageTransferProcessorState()
    , m_ImageTransferProcessorConfig()
    , m_pImageTransferWorkBuffer(pWorkMemory)
    , m_pImageCopyWorkBuffer(static_cast<uint8_t*>(pWorkMemory) +
        nn::irsensor::ImageTransferProcessorWorkBufferSize320x240)
    , m_PreviousSamplingNumber(0)
    , m_pGraphicsSystem(pGraphicsSystem)
    ,m_pFormat(NULL)
{
    {
        nn::irsensor::GetImageTransferProcessorDefaultConfig(&m_ImageTransferProcessorConfig);

        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_Read);
        info.SetSize(nn::irsensor::ImageTransferProcessorImageSize320x240);
        pGraphicsSystem->AllocateBuffer(&m_Buffer, &info);

        for (int i = 0; i < ImageTransferProcessorFormatMax; i++)
        {
            CreateTexture(pGraphicsSystem, Properties[i].width, Properties[i].height,
                nn::gfx::ImageFormat_R8_Unorm, &m_Texture[i],
                &m_TextureView[i], &m_TextureDescriptorSlot[i]);
        }
        CreateSampler(pGraphicsSystem, &m_Sampler, &m_SamplerDescriptorSlot);
    }

    nn::irsensor::ImageTransferProcessorConfig* pImageTransferConfig =
        &m_ImageTransferProcessorConfig;
    AddCommonReadWriteMenu(&m_ReadWriteMenu,
        &pImageTransferConfig->irCameraConfig,
        nn::irsensor::ImageTransferProcessorExposureTimeMin,
        nn::irsensor::ImageTransferProcessorExposureTimeMax
    );

    m_pFormat = new Format(pImageTransferConfig);
    MenuItem formatMenuItem("Format",
        m_pFormat
    );
    m_ReadWriteMenu.push_back(formatMenuItem);

    AddCommonReadOnlyMenu(&m_ReadOnlyMenu, &m_ImageTransferProcessorState.samplingNumber,
        &m_ImageTransferProcessorState.ambientNoiseLevel);

}

ImageTransferModeState::~ImageTransferModeState() NN_NOEXCEPT
{
    m_pGraphicsSystem->FreeBuffer(&m_Buffer);

    for (int i = 0; i < ImageTransferProcessorFormatMax; i++)
    {
        m_pGraphicsSystem->FreeTexture(&m_Texture[i]);
        m_TextureView[i].Finalize(&m_pGraphicsSystem->GetDevice());
    }

    m_Sampler.Finalize(&m_pGraphicsSystem->GetDevice());

    delete m_pFormat;
}

void ImageTransferModeState::Start() NN_NOEXCEPT
{
    nn::irsensor::RunImageTransferProcessor(m_IrCameraHandle,
        m_ImageTransferProcessorConfig,
        static_cast<void*>(m_pImageTransferWorkBuffer),
        Properties[m_ImageTransferProcessorConfig.format].workBufferSize
    );
    m_Format = m_ImageTransferProcessorConfig.format;
    // バッファをクリアしておく
    uint8_t* sourcePixels = m_Buffer.Map<uint8_t>();
    memset(sourcePixels, 0, Properties[m_Format].size);
    m_Buffer.FlushMappedRange(0, nn::irsensor::ImageTransferProcessorImageSize320x240);
    m_Buffer.Unmap();
}

namespace
{

    void CopyBufferToImage(nn::gfx::Texture* pTexture, const nn::gfx::Buffer* pBuffer,
        int width, int height, nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pTexture);
        NN_ASSERT_NOT_NULL(pBuffer);
        NN_ASSERT_NOT_NULL(pCommandBuffer);

        nn::gfx::TextureCopyRegion dstRegion;
        dstRegion.SetDefault();
        dstRegion.SetDepth(1);

        dstRegion.SetWidth(width);
        dstRegion.SetHeight(height);
        pCommandBuffer->CopyBufferToImage(pTexture, dstRegion, pBuffer, 0);
    }

    void CopyRawImageToBuffer(const nn::gfx::Buffer* pBuffer,
        void* copyBuffer,
        nn::irsensor::ImageTransferProcessorFormat format) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pBuffer);

        size_t writeSize = Properties[format].size;

        uint8_t* sourcePixels = pBuffer->Map<uint8_t>();
        memcpy(sourcePixels, copyBuffer, Properties[format].size);
        pBuffer->FlushMappedRange(0, writeSize);
        pBuffer->Unmap();
    }

} // namespace

void ImageTransferModeState::CopyImageBuffer(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    CopyBufferToImage(&m_Texture[m_Format], &m_Buffer,
        Properties[m_Format].width, Properties[m_Format].height,
        pCommandBuffer);
}

void ImageTransferModeState::Update() NN_NOEXCEPT
{
    nn::irsensor::GetImageTransferProcessorState(&m_ImageTransferProcessorState,
        m_pImageCopyWorkBuffer, Properties[m_Format].size, m_IrCameraHandle);
    if (m_PreviousSamplingNumber < m_ImageTransferProcessorState.samplingNumber)
    {
        // データ更新があった場合はバッファコピー
        m_PreviousSamplingNumber = m_ImageTransferProcessorState.samplingNumber;
        CopyRawImageToBuffer(&m_Buffer, m_pImageCopyWorkBuffer, m_Format);
    }
}

void ImageTransferModeState::RenderImageTransferProcessorState(
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer,
    const int ScreenIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // DepthStencilTypeをDepthStencilType_DepthNoWriteTestへ設定
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer,
        nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType corner;
    nn::util::Vector3fType size;

    const float XStart = -0.8f;
    const float XEnd = -0.2f;
    float yStart;
    float yEnd;
    if (ScreenIndex == 0)
    {
        yStart = 0.0f;
        yEnd = 0.8f;
    }
    else
    {
        yStart = -0.95f;
        yEnd = -0.15f;
    }

    nn::util::VectorSet(&corner, XStart, yStart, 0.0f);
    nn::util::VectorSet(&size, XEnd - XStart, yEnd - yStart, 0.0f);

    pPrimitiveRenderer->SetColor(Color::White);
    pPrimitiveRenderer->DrawQuad(
        pCommandBuffer,
        corner,
        size,
        m_TextureDescriptorSlot[m_Format],
        m_SamplerDescriptorSlot
    );
}

void ImageTransferModeState::Render(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer, int index) NN_NOEXCEPT
{
    CopyImageBuffer(pCommandBuffer);
    RenderImageTransferProcessorState(pPrimitiveRenderer, pCommandBuffer, index);
}
