﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <sstream>

#include <nn/nn_Assert.h>

#include "HidNpadIntegrate_IrSensorHandAnalysisModeState.h"

class HandAnalysisCurrentMode : public ReadWriteBase
{
    NN_DISALLOW_COPY(HandAnalysisCurrentMode);
    NN_DISALLOW_MOVE(HandAnalysisCurrentMode);

public:
    NN_IMPLICIT HandAnalysisCurrentMode(
        nn::irsensor::HandAnalysisConfig* pHandAnalysisConfig) NN_NOEXCEPT
        : m_pHandAnalysisConfig(pHandAnalysisConfig)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_pHandAnalysisConfig->mode == nn::irsensor::HandAnalysisMode_Silhouette)
        {
            sstr << "Silhouette";
        }
        else if (m_pHandAnalysisConfig->mode == nn::irsensor::HandAnalysisMode_Image)
        {
            sstr << "Image";
        }
        else if (m_pHandAnalysisConfig->mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage)
        {
            sstr << "Silhouette and Image";
        }
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        int8_t mode = static_cast<int8_t>(m_pHandAnalysisConfig->mode) + delta;
        mode = std::min(
                std::max(mode, static_cast<int8_t>(nn::irsensor::HandAnalysisMode_Silhouette)
            ),
            static_cast<int8_t>(nn::irsensor::HandAnalysisMode_SilhouetteAndImage)
        );
        m_pHandAnalysisConfig->mode = static_cast<nn::irsensor::HandAnalysisMode>(mode);
    }

private:
    nn::irsensor::HandAnalysisConfig* m_pHandAnalysisConfig;
};

class ShapeArea : public ReadWriteBase
{
    NN_DISALLOW_COPY(ShapeArea);
    NN_DISALLOW_MOVE(ShapeArea);

public:
    NN_IMPLICIT ShapeArea(const PolygonProperties* pPolygonProperties) NN_NOEXCEPT
        : m_pPolygonProperties(pPolygonProperties)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << m_pPolygonProperties->signedArea;
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
    }

private:
    const PolygonProperties* m_pPolygonProperties;
};

class FingerShape : public ReadWriteBase
{
    NN_DISALLOW_COPY(FingerShape);
    NN_DISALLOW_MOVE(FingerShape);

public:
    NN_IMPLICIT FingerShape(
        nn::irsensor::HandAnalysisConfig* pHandAnalysisConfig,
        nn::irsensor::HandAnalysisSilhouetteState* pHandAnalysisSilhouetteState) NN_NOEXCEPT
        : m_pHandAnalysisConfig(pHandAnalysisConfig)
        , m_pHandAnalysisSilhouetteState(pHandAnalysisSilhouetteState)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_pHandAnalysisSilhouetteState->handCount > 0)
        {
            const nn::irsensor::Hand& hand = m_pHandAnalysisSilhouetteState->hands[0];
            int numFinger = 0;

            for (int i = 0; i < nn::irsensor::HandFinger_Count; ++i)
            {
                if (hand.fingers[i].isValid)
                {
                    numFinger++;
                }
            }

            switch (numFinger)
            {
            case 0:
            case 1:
                sstr << "Rock";
                break;
            case 2:
            case 3:
                sstr << "Scissor";
                break;
            case 5:
            case 6:
                sstr << "Paper";
                break;
            default:
                sstr << "Unknown";
                break;
            }
        }
        else
        {
            sstr << "Unknown";
        }
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
    }

private:
    nn::irsensor::HandAnalysisConfig* m_pHandAnalysisConfig;
    nn::irsensor::HandAnalysisSilhouetteState* m_pHandAnalysisSilhouetteState;
};

HandAnalysisModeState::HandAnalysisModeState(IrSensorMode* pNextProcessor, int* pMenuSelection,
    nn::irsensor::IrCameraHandle irCameraHandle, GraphicsSystem* pGraphicsSystem) NN_NOEXCEPT
    : IrSensorModeState(pNextProcessor, pMenuSelection, irCameraHandle)
    , m_NextHandAnalysisConfig()
    , m_CurrentHandAnalysisConfig()
    , m_HandAnalysisSilhouetteState()
    , m_HandAnalysisImageState()
    , m_PolygonProperties()
    , m_pGraphicsSystem(pGraphicsSystem)
    , m_pHandAnalysisCurrentMode(NULL)
    , m_pShapeArea(NULL)
    , m_pFingerShape(NULL)

{
    {
        m_NextHandAnalysisConfig.mode = nn::irsensor::HandAnalysisMode_SilhouetteAndImage;
        m_CurrentHandAnalysisConfig = m_NextHandAnalysisConfig;

        CreateTexture(pGraphicsSystem, nn::irsensor::IrHandAnalysisImageWidth,
            nn::irsensor::IrHandAnalysisImageHeight, nn::gfx::ImageFormat_R16_Unorm,
            &m_Texture, &m_TextureView, &m_TextureDescriptorSlot);
        CreateSampler(pGraphicsSystem, &m_Sampler, &m_SamplerDescriptorSlot);

        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_Read);
        info.SetSize(ImageSize);
        pGraphicsSystem->AllocateBuffer(&m_Buffer, &info);
    }

    nn::irsensor::HandAnalysisConfig* pHandAnalysisConfig = &m_NextHandAnalysisConfig;
    m_pHandAnalysisCurrentMode = new HandAnalysisCurrentMode(pHandAnalysisConfig);
    MenuItem handAnalysisCurrentModeMenuItem("Mode",
        m_pHandAnalysisCurrentMode
    );
    m_ReadWriteMenu.push_back(handAnalysisCurrentModeMenuItem);

    AddCommonReadOnlyMenu(&m_ReadOnlyMenu, &m_HandAnalysisImageState.samplingNumber,
        &m_HandAnalysisImageState.ambientNoiseLevel);
    AddStatisticsMenu(&m_ReadOnlyMenu, m_Statistics.GetPtrPacketDropPercentage());
    const PolygonProperties* pPolygonProperties = &m_PolygonProperties;
    nn::irsensor::HandAnalysisSilhouetteState* pHandAnalysisSilhouetteState =
        &m_HandAnalysisSilhouetteState;
    m_pShapeArea = new ShapeArea(pPolygonProperties);
    MenuItem shapeAreaMenuItem("Shape Area",
        m_pShapeArea
    );
    m_ReadOnlyMenu.push_back(shapeAreaMenuItem);

    m_pFingerShape = new FingerShape(pHandAnalysisConfig, pHandAnalysisSilhouetteState);
    MenuItem fingerShapeMenuItem("Shape",
        m_pFingerShape
    );
    m_ReadOnlyMenu.push_back(fingerShapeMenuItem);
}

HandAnalysisModeState::~HandAnalysisModeState() NN_NOEXCEPT
{
    m_pGraphicsSystem->FreeBuffer(&m_Buffer);

    m_pGraphicsSystem->FreeTexture(&m_Texture);
    m_TextureView.Finalize(&m_pGraphicsSystem->GetDevice());

    m_Sampler.Finalize(&m_pGraphicsSystem->GetDevice());

    delete m_pHandAnalysisCurrentMode;
    delete m_pShapeArea;
    delete m_pFingerShape;
}

void HandAnalysisModeState::Start() NN_NOEXCEPT
{
    nn::irsensor::RunHandAnalysis(m_IrCameraHandle, m_NextHandAnalysisConfig);
    m_CurrentHandAnalysisConfig = m_NextHandAnalysisConfig;
}

void HandAnalysisModeState::Update() NN_NOEXCEPT
{
    if ((m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_Silhouette) ||
        (m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage))
    {
        int count = 0;
        nn::irsensor::GetHandAnalysisSilhouetteState(&m_HandAnalysisSilhouetteState,
            &count, 1, 0, m_IrCameraHandle);

        if(count > 0)
        {
            m_Statistics.Update(&m_HandAnalysisSilhouetteState.samplingNumber,
                IrSensorModeStatistics::ExpectedHandAnalysisFramerate);
        }
    }
    if ((m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_Image) ||
        (m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage))
    {
        int count = 0;
        nn::irsensor::GetHandAnalysisImageState(&m_HandAnalysisImageState,
            &count, 1, 0, m_IrCameraHandle);

        if(count > 0)
        {
            m_Statistics.Update(&m_HandAnalysisImageState.samplingNumber,
                IrSensorModeStatistics::ExpectedHandAnalysisFramerate);

            uint16_t* pSource = static_cast<uint16_t*>(m_HandAnalysisImageState.image);
            uint16_t* pDest = m_Buffer.Map< uint16_t >();
            memcpy(pDest, pSource, ImageSize);
            for (size_t i = 0; i < ImageSize; i++)
            {
                const float Max = static_cast<float>(0xffff);
                const float Value = static_cast<float>(pDest[i]) / Max;
                const float ScaledValue = Value * 100.0f > 1.0f ? 1.0f : Value * 100.0f;

                pDest[i] = static_cast<uint16_t>(ScaledValue * Max);
            }
            m_Buffer.FlushMappedRange(0, ImageSize);
            m_Buffer.Unmap();
        }
    }
}

void HandAnalysisModeState::RenderImageState(
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer,
    const int ScreenIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // DepthStencilTypeをDepthStencilType_DepthNoWriteTestへ設定
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer,
        nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType corner;
    nn::util::Vector3fType size;

    const float XStart = -0.2f;
    const float XEnd = -0.8f;
    float yStart;
    float yEnd;
    if (ScreenIndex == 0)
    {
        yStart = 0.8f;
        yEnd = 0.0f;
    }
    else
    {
        yStart = -0.15f;
        yEnd = -0.95f;
    }

    nn::util::VectorSet(&corner, XStart, yStart, 0.0f);
    nn::util::VectorSet(&size, XEnd - XStart, yEnd - yStart, 0.0f);

    pPrimitiveRenderer->SetColor(Color::White);
    pPrimitiveRenderer->DrawQuad(
        pCommandBuffer,
        corner,
        size,
        m_TextureDescriptorSlot,
        m_SamplerDescriptorSlot);
}

namespace
{

    void WriteSilhouette(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::irsensor::HandAnalysisSilhouetteState* pSilhouetteState,
        PolygonProperties* pPolygonProperties,
        const float XStart, const float XEnd,
        const float YStart, const float YEnd,
        const float XScale, const float YScale) NN_NOEXCEPT
    {
        nn::util::Vector3fType begin;
        nn::util::Vector3fType end;

        for (int shapeIndex = 0; shapeIndex < pSilhouetteState->shapeCount; shapeIndex++)
        {
            std::vector<nn::util::Float2> silhouette;
            const nn::irsensor::Shape& Shape = pSilhouetteState->shapes[shapeIndex];

            nn::util::Float2 p1, p2;
            p2 = pSilhouetteState->points[Shape.firstPointIndex];
            pPrimitiveRenderer->SetColor(Color::Lime);
            for (int i = 0; i < Shape.pointCount - 1; ++i)
            {
                p1 = p2;
                p2 = pSilhouetteState->points[(Shape.firstPointIndex + i + 1) %
                    NN_ARRAY_SIZE(pSilhouetteState->points)];

                nn::util::VectorSet(&begin,
                    XStart + (XEnd - XStart) * (p1.x * XScale + 1.0f) / 2.0f,
                    YStart + (YEnd - YStart) * (p1.y * YScale + 1.0f) / 2.0f, 0.0f);
                nn::util::VectorSet(&end,
                    XStart + (XEnd - XStart) * (p2.x * XScale + 1.0f) / 2.0f,
                    YStart + (YEnd - YStart) * (p2.y * YScale + 1.0f) / 2.0f, 0.0f);

                pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);

                silhouette.push_back(p1);
            }
            silhouette.push_back(p2);

            std::vector<size_t> convexHullIndices;
            // Convex Hull を計算する
            ExtractConvexHull(convexHullIndices, silhouette);
            if (convexHullIndices.size() > 0)
            {
                p2 = pSilhouetteState->points[Shape.firstPointIndex + convexHullIndices[0]];
                pPrimitiveRenderer->SetColor(Color::Yellow);
                for (size_t i = 0; i < convexHullIndices.size() - 1; ++i)
                {
                    p1 = p2;
                    p2 = pSilhouetteState->points[Shape.firstPointIndex +
                        convexHullIndices[i + 1]];

                    nn::util::VectorSet(&begin,
                        XStart + (XEnd - XStart) * (p1.x * XScale + 1.0f) / 2.0f,
                        YStart + (YEnd - YStart) * (p1.y * YScale + 1.0f) / 2.0f, 0.0f);
                    nn::util::VectorSet(&end,
                        XStart + (XEnd - XStart) * (p2.x * XScale + 1.0f) / 2.0f,
                        YStart + (YEnd - YStart) * (p2.y * YScale + 1.0f) / 2.0f, 0.0f);

                    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
                }
            }

            ComputePolygonProperties(pPolygonProperties, &silhouette[0], silhouette.size());
        }
    }

    void WriteFingertip(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::irsensor::HandAnalysisSilhouetteState* pSilhouetteState,
        const float XStart, const float XEnd,
        const float YStart, const float YEnd,
        const float XScale, const float YScale) NN_NOEXCEPT
    {
        nn::util::Vector3fType begin;

        for (int handIndex = 0; handIndex < pSilhouetteState->handCount; handIndex++)
        {
            const nn::irsensor::Hand& hand = pSilhouetteState->hands[handIndex];

            // 検出物の中央を描画
            nn::util::VectorSet(&begin,
                XStart + (XEnd - XStart) * (hand.palm.center.x * XScale + 1.0f) / 2.0f,
                YStart + (YEnd - YStart) * (hand.palm.center.y * YScale + 1.0f) / 2.0f, 0.0f);
            pPrimitiveRenderer->SetColor(Color::Lime);
            pPrimitiveRenderer->DrawSphere(
                pCommandBuffer,
                nns::gfx::PrimitiveRenderer::Surface_Solid,
                nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
                begin,
                8.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));

            for (int i = 0; i < nn::irsensor::HandFinger_Count; ++i)
            {
                if (hand.fingers[i].isValid)
                {
                    nn::util::VectorSet(&begin,
                        XStart + (XEnd - XStart) * (hand.fingers[i].tip.x * XScale + 1.0f) / 2.0f,
                        YStart + (YEnd - YStart) * (hand.fingers[i].tip.y * YScale + 1.0f) / 2.0f, 0.0f);
                    pPrimitiveRenderer->SetColor(Color::Red);
                    pPrimitiveRenderer->DrawSphere(
                        pCommandBuffer,
                        nns::gfx::PrimitiveRenderer::Surface_Solid,
                        nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
                        begin,
                        8.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));

                    nn::util::VectorSet(&begin,
                        XStart + (XEnd - XStart) * (hand.fingers[i].root.x * XScale + 1.0f) / 2.0f,
                        YStart + (YEnd - YStart) * (hand.fingers[i].root.y * YScale + 1.0f) / 2.0f, 0.0f);
                    pPrimitiveRenderer->SetColor(Color::Blue);
                    pPrimitiveRenderer->DrawSphere(
                        pCommandBuffer,
                        nns::gfx::PrimitiveRenderer::Surface_Solid,
                        nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
                        begin,
                        8.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));
                }
            }
        }
    }

    void WriteHandAnalysisState(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::irsensor::HandAnalysisSilhouetteState* pSilhouetteState,
        PolygonProperties* pPolygonProperties,
        const int ScreenIndex) NN_NOEXCEPT
    {
        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Matrix4x4fType projectionMatrix;
        nn::util::Matrix4x3f modelMatrix;

        nn::util::MatrixIdentity(&viewMatrix);
        nn::util::MatrixIdentity(&projectionMatrix);
        nn::util::MatrixIdentity(&modelMatrix);
        pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
        pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
        pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

        // DepthStencilTypeをDepthStencilType_DepthNoWriteTestへ設定
        pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer,
            nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

        pPrimitiveRenderer->SetLineWidth(10.0f);

        const float XStart = -0.2f;
        const float XEnd = -0.8f;
        float yStart;
        float yEnd;
        if (ScreenIndex == 0)
        {
            yStart = 0.0f;
            yEnd = 0.8f;
        }
        else
        {
            yStart = -0.95f;
            yEnd = -0.15f;
        }

        // 注：座標はY軸を基準に調整されます。
        // 画像の縦横比が4/3であるため、X座標の値は画像スペースに収まるよう
        // 3/4倍にする必要があるためです。
        const float XScale = 0.75f;
        const float YScale = 1.0f;

        // シルエットを描画
        WriteSilhouette(pPrimitiveRenderer, pCommandBuffer, pSilhouetteState, pPolygonProperties,
        XStart, XEnd, yStart, yEnd, XScale, YScale);

        // 指先を描画
        WriteFingertip(pPrimitiveRenderer, pCommandBuffer, pSilhouetteState,
        XStart, XEnd, yStart, yEnd, XScale, YScale);
    }

} // namespace

void HandAnalysisModeState::Render(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer, int index) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    // 読み込んだカメラ画像 nn::gfx::buffer を、テクスチャにタイリングします
    if ((m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_Image) ||
        (m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage))
    {
        nn::gfx::TextureCopyRegion dstRegion;
        dstRegion.SetDefault();
        dstRegion.SetDepth(1);

        dstRegion.SetWidth(nn::irsensor::IrHandAnalysisImageWidth);
        dstRegion.SetHeight(nn::irsensor::IrHandAnalysisImageHeight);
        pCommandBuffer->CopyBufferToImage(&m_Texture, dstRegion, &m_Buffer, 0);

        RenderImageState(pPrimitiveRenderer, pCommandBuffer, index);
    }
    if ((m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_Silhouette) ||
        (m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage))
    {
        WriteHandAnalysisState(pPrimitiveRenderer, pCommandBuffer,
            &m_HandAnalysisSilhouetteState, &m_PolygonProperties, index);
    }
}
