﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "./HidControllerSequence_Thread.h"

ThreadManager g_ThreadManager;

ThreadState::ThreadState() : m_pThreadStack(nullptr)
{
}

ThreadState::ThreadState(const char* name, nn::os::ThreadFunction func) : ThreadState()
{
    SetName(name);
    SetFunc(func);
}

void ThreadState::SetName(const char* name)
{
    m_ThreadName = name;
}

void ThreadState::SetFunc(nn::os::ThreadFunction func)
{
    m_ThreadFunction = func;
}

nn::Result ThreadState::Create(int priority)
{
    nn::Result result;
    if (m_pThreadStack != nullptr)
    {
        delete[] m_pThreadStack;
        m_pThreadStack = nullptr;
    }
    m_pThreadStack = new NN_OS_ALIGNAS_THREAD_STACK char[ThreadStackSize];
    result = nn::os::CreateThread(&m_ThreadType, m_ThreadFunction, this, m_pThreadStack, ThreadStackSize, priority);
    nn::os::SetThreadName(&m_ThreadType, m_ThreadName.c_str());
    return result;
}

bool ThreadState::Destroy()
{
    if (m_ThreadType._state != nn::os::ThreadType::State::State_NotInitialized)
    {
        nn::os::DestroyThread(&m_ThreadType);
        return true;
    }
    return false;
}

bool ThreadState::Start()
{
    if (m_ThreadType._state == nn::os::ThreadType::State::State_Initialized)
    {
        nn::os::StartThread(&m_ThreadType);
        return true;
    }
    return false;
}

nn::os::ThreadType* ThreadState::GetThreadType()
{
    return &m_ThreadType;
}

// --------------------------------------------------------------------------------------

nn::os::ThreadType* ThreadManager::pPrevThread = nullptr;
nn::os::ThreadType* ThreadManager::pNextThread = nullptr;

ThreadManager::ThreadManager() : m_Mutex(true, 9)
{
    m_StateList.resize(0);
    pPrevThread = pNextThread = nullptr;
}

ThreadManager::~ThreadManager()
{
    DestroyThread();
}

size_t ThreadManager::Add(ThreadState* state)
{
    m_StateList.push_back(state);
    return m_StateList.size();
}

void ThreadManager::CreateThread(int priority)
{
    for (
        std::vector<ThreadState*>::iterator it = m_StateList.begin();
        it != m_StateList.end();
        ++it)
    {
        (*it)->Create(priority);
    }
}

void ThreadManager::DestroyThread()
{
    for (
        std::vector<ThreadState*>::iterator it = m_StateList.begin();
        it != m_StateList.end();
        ++it)
    {
        (*it)->Destroy();
    }
}

bool ThreadManager::GetThreadStateFromIndex(uint32_t index, ThreadState** outState)
{
    if (index < m_StateList.size() && outState != nullptr)
    {
        *outState = m_StateList.at(index);
        return true;
    }
    return false;
}

bool ThreadManager::GetThreadState(const nn::os::ThreadType* inThread, ThreadState** outState)
{
    if (inThread != nullptr)
    {
        uint64_t addr = (uint64_t)inThread;
        for (
            std::vector<ThreadState*>::iterator it = m_StateList.begin();
            it != m_StateList.end();
            ++it)
        {
            if ((uint64_t)((*it)->GetThreadType()) == addr)
            {
                *outState = (*it);
                return true;
            }
        }
    }
    return false;
}

void ThreadManager::SetEndThread(nn::os::ThreadType* stopThread)
{
    m_Mutex.Lock();
    {
        pPrevThread = stopThread;
    }
    m_Mutex.Unlock();
}

void ThreadManager::SetNextThread(nn::os::ThreadType* nextThread)
{
    m_Mutex.Lock();
    {
        pNextThread = nextThread;
    }
    m_Mutex.Unlock();
}
