﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "./HidControllerSequence_Dialog.h"

void DialogInfo::SetDefault()
{
    this->Size = NN_UTIL_FLOAT_2_INITIALIZER(480.f, 192.f);
    this->Pos = NN_UTIL_FLOAT_2_INITIALIZER((1280 - Size.x) / 2.f, (720 - Size.y) / 2.f);
    this->Title = "Title";
    this->Message = "Message";
    this->Answer.clear();
    this->Answer.push_back("Cancel");
    this->Answer.push_back("OK");
    this->CancelNum = 0;
    this->ControllerCheck.Set();
}

int32_t ShowDialog(const DialogInfo& info)
{
    return ShowDialog(info, ApplicationState::IsHandheldMode() ? nn::hid::NpadId::Handheld : nn::hid::NpadId::No1, false);
}

int32_t ShowDialog(const DialogInfo& info, const nn::hid::NpadIdType& npadId, bool single)
{
    // 呼び出したNpadId
    nn::hid::NpadIdType callNpadId = npadId;
    // グラフィックス 関連の機能を取得
    nns::gfx::PrimitiveRenderer::Renderer*  pRenderer   = &g_pGraphicsSystem->GetPrimitiveRenderer();
    nn::gfx::util::DebugFontTextWriter*     pTextWriter = &g_pGraphicsSystem->GetDebugFont();
    nn::gfx::CommandBuffer*                 pCommand    = &g_pGraphicsSystem->GetCommandBuffer();
    // 戻り値
    int32_t result = 0;
    // 項目数
    int32_t itemsCount = static_cast<int32_t>(std::min(static_cast<size_t>(8), info.Answer.size()));
    //============================================================
    // 描画座標
    //============================================================
    // 余白
    const float margin = 16.f;
    // タイトル座標と描画範囲 ------------------------------------
    pTextWriter->SetScale(1.f, 1.f);
    nn::util::Float2 titleSize = NN_UTIL_FLOAT_2_INITIALIZER(pTextWriter->CalculateStringWidth(info.Title.c_str()), pTextWriter->CalculateStringHeight(info.Title.c_str()) + (margin * 2.f));
    nn::util::Float2 titlePos = NN_UTIL_FLOAT_2_INITIALIZER((info.Size.x - titleSize.x) / 2.f + info.Pos.x, info.Pos.y + margin);
    if (info.Title.length() == 0)
    {
        titleSize = NN_UTIL_FLOAT_2_INITIALIZER(0.f, 0.f);
    }
    // タイトルとメッセージの境目
    nn::util::Float2 headerLine = NN_UTIL_FLOAT_2_INITIALIZER(info.Pos.x, info.Pos.y + titleSize.y);
    // 本文の座標
    nn::util::Float2 messagePos = NN_UTIL_FLOAT_2_INITIALIZER(info.Pos.x, headerLine.y);
    pTextWriter->SetScale(1.2f, 1.2f);
    nn::util::Float2 messageSize = NN_UTIL_FLOAT_2_INITIALIZER(info.Size.x, pTextWriter->CalculateStringHeight(info.Message.c_str()) + (margin * 2.f));
    pTextWriter->SetScale(1.f, 1.f);
    if (info.Message.length() == 0)
    {
        messageSize = NN_UTIL_FLOAT_2_INITIALIZER(0.f, 0.f);
    }
    if (titleSize.y + messageSize.y + 48.f < info.Size.y)
    {
        messageSize.y = info.Size.y - titleSize.y - 48.f;
    }
    // 選択項目の上の線
    nn::util::Float2 bottomLine = NN_UTIL_FLOAT_2_INITIALIZER(info.Pos.x, info.Pos.y + titleSize.y + messageSize.y);
    // 選択肢の描画間隔
    float buttonDivSize = info.Size.x / static_cast<float>(itemsCount);
    nn::util::Float2 answerItemBasePos = NN_UTIL_FLOAT_2_INITIALIZER(info.Pos.x, bottomLine.y + 16.f);

    g_pGraphicsSystem->StopScanScreen();

    //============================================================
    // メインループ
    //============================================================
    // 画面のコピーを中断して遷移直前の画面を記憶する

    bool exit = false;
    uint64_t animationFrameCount = 0;

    while (!exit)
    {
        //============================================================
        // コントローラー入力の更新
        //============================================================
        nn::os::LockMutex(&g_Mutex);
        {
            g_pControllerManager->Update();
        }
        nn::os::UnlockMutex(&g_Mutex);
        // コントローラーの接続確認
        CheckPatternSet check = info.ControllerCheck;
        if (check.IsAllOn())
        {
            check.Reset();
            if (ApplicationState::GetPlayerCount() > 1)
            {
                check.Set<CheckPattern::PlayerCountShort>();
            }
            else
            {
                check.Set<CheckPattern::Single>();
                check.Set<CheckPattern::ChangeMasterController>();
                check.Set<CheckPattern::CallInterval>();
                check.Set<CheckPattern::Resume>();
            }
        }

        bool isCallControllerSupport = CheckConnectController(check, &callNpadId);
        if (isCallControllerSupport)
        {
            nn::hid::ControllerSupportResultInfo resultInfo;
            nn::Result appletResult = CallControllerSupportApplet(&resultInfo, static_cast<int8_t>(ApplicationState::GetPlayerCount()));
            if(ApplicationState::GetPlayerCount() > 1)
            {
                if (
                    appletResult.IsFailure() ||
                    resultInfo.playerCount < static_cast<int8_t>(std::min(ApplicationState::GetPlayerCount(), ApplicationState::MaxPlayerCount))
                    )
                {
                    result = -1;
                    break;
                }
            }
            else
            {
                callNpadId = resultInfo.selectedId;
                ApplicationState::EnableHandheldMode(resultInfo.selectedId == nn::hid::NpadId::Handheld);
            }
        }

        if (ApplicationState::GetPlayerCount() == 1)
        {
            callNpadId = ApplicationState::IsHandheldMode() ? nn::hid::NpadId::Handheld : nn::hid::NpadId::No1;
        }

        if (!isCallControllerSupport)
        {
            for (
                std::vector<nns::hid::Controller*>::iterator it = g_pControllerManager->GetControllerList().begin();
                it != g_pControllerManager->GetControllerList().end();
                ++it)
            {
                const nn::hid::NpadIdType id = ApplicationState::NpadIds[it - g_pControllerManager->GetControllerList().begin()];

                if (
                    (!single || id == callNpadId) &&
                    (*it)->IsConnected()
                    )
                {
                    if ((*it)->GetButtonsDown().Test<nns::hid::Button::B>())
                    {
                        result = info.CancelNum;
                        exit = true;
                        break;
                    }
                    else if ((*it)->GetButtonsDown().Test<nns::hid::Button::A>())
                    {
                        exit = true;
                        break;

                    }
                    else if (
                        (*it)->GetButtonsDown().Test<nns::hid::Button::LsLeft>() ||
                        (*it)->GetButtonsDown().Test<nns::hid::Button::RsLeft>() ||
                        (*it)->GetButtonsDown().Test<nns::hid::Button::Left>())
                    {
                        result = (result + itemsCount - 1) % itemsCount;
                        break;

                    }
                    else if (
                        (*it)->GetButtonsDown().Test<nns::hid::Button::LsRight>() ||
                        (*it)->GetButtonsDown().Test<nns::hid::Button::RsRight>() ||
                        (*it)->GetButtonsDown().Test<nns::hid::Button::Right>())
                    {
                        result = (result + 1) % itemsCount;
                        break;

                    }
                }
            }
        }

        //============================================================
        // テキストの処理
        //============================================================

        nn::os::LockMutex(&g_Mutex);
        {
            pTextWriter->SetScale(1.f, 1.f);
            pTextWriter->SetTextColor(nn::util::Color4u8::Blue());
            pTextWriter->SetCursor(titlePos.x, titlePos.y);
            {
                size_t offset = 0;
                size_t next = 0;
                do {
                    next = info.Title.find_first_of('\n', offset);
                    if (next == info.Title.npos) {
                        const float width = pTextWriter->CalculateStringWidth(info.Title.substr(offset).c_str());
                        pTextWriter->SetCursorX(info.Pos.x + (info.Size.x - width) / 2.f);
                        pTextWriter->Print("%s\n", info.Title.substr(offset, next - offset).c_str());
                    }
                    else
                    {
                        const float width = pTextWriter->CalculateStringWidth(info.Title.substr(offset, next - offset).c_str());
                        pTextWriter->SetCursorX(info.Pos.x + (info.Size.x - width) / 2.f);
                        pTextWriter->Print("%s\n", info.Title.substr(offset, next - offset).c_str());
                        offset = next + 1;
                    }
                } while (next != info.Title.npos);
            }

            pTextWriter->SetScale(1.2f, 1.2f);
            pTextWriter->SetTextColor(nn::util::Color4u8::Black());
            pTextWriter->SetCursor(messagePos.x, messagePos.y);
            {
                size_t offset = 0;
                size_t next = 0;
                do{
                    next = info.Message.find_first_of('\n', offset);
                    if (next == info.Message.npos) {
                        const float width = pTextWriter->CalculateStringWidth(info.Message.substr(offset).c_str());
                        pTextWriter->SetCursorX(info.Pos.x + (info.Size.x - width) / 2.f);
                        pTextWriter->Print("%s\n", info.Message.substr(offset, next - offset).c_str());
                    }
                    else
                    {
                        const float width = pTextWriter->CalculateStringWidth(info.Message.substr(offset, next - offset).c_str());
                        pTextWriter->SetCursorX(info.Pos.x + (info.Size.x - width) / 2.f);
                        pTextWriter->Print("%s\n", info.Message.substr(offset, next - offset).c_str());
                        offset = next + 1;
                    }
                }while (next != info.Title.npos);
            }
            pTextWriter->SetScale(1.f, 1.f);

            for (int i = 0; i < itemsCount; ++i)
            {
                pTextWriter->SetTextColor(
                    i == result ? nn::util::Color4u8::White() : nn::util::Color4u8::Blue());
                pTextWriter->SetCursor(
                    answerItemBasePos.x + buttonDivSize * i +
                    (buttonDivSize - pTextWriter->CalculateStringWidth(info.Answer.at(i).c_str())) / 2.f,
                    answerItemBasePos.y);
                pTextWriter->Print("%s\n", info.Answer.at(i).c_str());
            }
        }

        nn::os::UnlockMutex(&g_Mutex);

        //============================================================
        // 描画処理
        //============================================================

        nn::os::LockMutex(&g_Mutex);
        {
            g_pGraphicsSystem->BeginDraw();
            pRenderer->SetColor(nn::util::Color4u8::White());
            // 背景のぼかし
            pRenderer->Draw2DRect(pCommand,
                0, 720, 1280, -720, g_ScanScreenDescriptor, g_SamplerDescriptor);
            // ダイアログの背景
            pRenderer->SetColor(nn::util::Color4u8::White());
            pRenderer->Draw2DRect(pCommand, info.Pos.x, info.Pos.y, info.Size.x, titleSize.y + messageSize.y + 48.f);
            // 分割線
            pRenderer->SetColor(nn::util::Color4u8::Gray());
            pRenderer->Draw2DLine(pCommand, headerLine.x, headerLine.y, headerLine.x + info.Size.x, headerLine.y);
            pRenderer->Draw2DLine(pCommand, bottomLine.x, bottomLine.y, bottomLine.x + info.Size.x, bottomLine.y);
            // 選択項目
            uint8_t col = static_cast<uint8_t>((nn::util::SinEst(nn::util::DegreeToRadian(static_cast<float>(animationFrameCount % 90) * 4.f)) + 1.f) * 36.f);
            pRenderer->SetColor(nn::util::Color4u8(col + 63, static_cast<uint8_t>(std::min(col * 1.25, 192.0) + 63), 255, 255));
            pRenderer->Draw2DRect(pCommand,
                info.Pos.x + buttonDivSize * result, bottomLine.y,
                buttonDivSize, 48.f);

            pRenderer->SetColor(nn::util::Color4u8::Gray());
            for (int i = 1; i < itemsCount; ++i)
            {
                pRenderer->Draw2DLine(pCommand,
                    info.Pos.x + buttonDivSize * i, bottomLine.y,
                    info.Pos.x + buttonDivSize * i, bottomLine.y + 48.f);
            }

            pTextWriter->Draw(pCommand);

            g_pGraphicsSystem->EndDraw();
        }
        nn::os::UnlockMutex(&g_Mutex);

        ++animationFrameCount;
        g_pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / g_FrameRate));
    }

    // 元の画面に戻る前にコントローラー情報を更新する
    nn::os::LockMutex(&g_Mutex);
    {
        g_pControllerManager->Update();
    }
    nn::os::UnlockMutex(&g_Mutex);

    g_pGraphicsSystem->StartScanScreen();
    return result;

} // NOLINT(impl/function_size)
