﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "./HidControllerSequence_ControllerSupportApplet.h"

const nn::util::Color4u8 g_ControllerColorList[4] =
{
    nn::util::Color4u8::Red(),
    nn::util::Color4u8::Green(),
    nn::util::Color4u8::Blue(),
    nn::util::Color4u8::Yellow()
};

const char* g_ControllerTextList[4] =
{
    "Red", "Green", "Blue", "Yellow"
};

bool CheckConnectController(const CheckPatternSet& check, const nn::hid::NpadIdType* npadId)
{
    static nn::oe::OperationMode    prevOperationMode = nn::oe::GetOperationMode(); //!< 直前の動作モード
    static nn::os::TimerEventType   timerEventType;                                 //!< コントローラーサポートアプレットの呼び出し遅延用のタイマーイベント
    static bool                     isWaitTimeExtended = false;                     //!< true の場合イベント呼び出しを延期します
                                                                                    //!< 動作モードが携帯モードから据置モードに変化した場合 true
    bool isChangeToConsoleMode                  = (nn::oe::GetOperationMode() != prevOperationMode) && (prevOperationMode == nn::oe::OperationMode_Handheld);

    if (timerEventType._state == nn::os::TimerEventType::State_NotInitialized)
    {
        // タイマーイベントが初期化されていない場合
        nn::os::InitializeTimerEvent(&timerEventType, nn::os::EventClearMode_AutoClear);
    }

    //==============================================
    // コントローラーの接続確認
    //==============================================
    bool isHandheldConnected                    = false;                            //!< 本体装着コントローラーが接続済みか
    bool isWirelessCountollerConnected          = false;                            //!< 無線コントローラー(NpadId::No1)が接続済みか
    const bool isTargetControllerSpecified      = (npadId != nullptr);              //!< 接続を確認するコントローラーが指定されているか否か
    bool isTargetControllerConnected            = false;                            //!< 接続を確認するコントローラーが接続済みか
    bool isConnectNewController                 = false;                            //!< 新しいコントローラーを検出した場合 true
    size_t connectedWirelessCountollerCount     = 0;                                //!< 無線コントローラーの接続台数
    nns::hid::ButtonSet disableMasterControllerButtonSet;                           //!< マスターのコントローラー以外からのボタン入力
    disableMasterControllerButtonSet.Reset();

    for (
        std::vector<nns::hid::Controller*>::iterator it = g_pControllerManager->GetControllerList().begin();
        it != g_pControllerManager->GetControllerList().end();
        ++it
        )
    {
        const size_t controllerIndex = (it - g_pControllerManager->GetControllerList().begin());
        const nn::hid::NpadIdType id = ApplicationState::NpadIds[controllerIndex];

        if (
            (*it)->IsConnected() &&
            !(isChangeToConsoleMode && (id == nn::hid::NpadId::Handheld))
            )
        {
            isHandheldConnected                 |= (id == nn::hid::NpadId::Handheld);                           // 本体装着コントローラーが接続されているか
            isWirelessCountollerConnected       |= (id == nn::hid::NpadId::No1);                                // 無線の1Pコントローラーが接続されているか
            isTargetControllerConnected         |= (isTargetControllerSpecified && (id == (*npadId)));          // 確認対象のコントローラーが接続されているか

            if (
                !(isTargetControllerSpecified && (id == (*npadId))) &&
                (
                (ApplicationState::IsHandheldMode() && (id != nn::hid::NpadId::Handheld)) ||
                (!ApplicationState::IsHandheldMode() && (id != nn::hid::NpadId::No1)))
                )
            {
                // マスターコントローラーでないコントローラーのボタン入力を取得
                disableMasterControllerButtonSet |= (*it)->GetButtons();
                if (ApplicationState::GetPlayerCount() > 1 && id == nn::hid::NpadId::Handheld)
                {
                    // 本体装着コンではメニューボタンを押してもコントローラーサポートアプレットが開かないようにする
                    disableMasterControllerButtonSet.Reset<nns::hid::Button::Plus>();
                    disableMasterControllerButtonSet.Reset<nns::hid::Button::Minus>();
                }
            }
            if (
                (id != nn::hid::NpadId::Handheld) &&
                (controllerIndex < ApplicationState::GetPlayerCount())
                )
            {
                ++connectedWirelessCountollerCount;
            }
        }

        bool isLeftConnected = static_cast<nns::hid::GamePadNx*>((*it))->GetAttributesSet().Test<nns::hid::Attribute::IsLeftConnected>();
        bool isRightConnected = static_cast<nns::hid::GamePadNx*>((*it))->GetAttributesSet().Test<nns::hid::Attribute::IsRightConnected>();

        if((*it)->IsConnected() || isLeftConnected || isRightConnected)
        {
            if (id != nn::hid::NpadId::Handheld)
            {
                if (ApplicationState::IsHandheldMode() || id != nn::hid::NpadId::No1)
                {
                    isConnectNewController |= (isLeftConnected || (*it)->IsConnected()) && !ApplicationState::InitialConnectedController[controllerIndex].Test<nn::hid::NpadJoyAttribute::IsLeftConnected>();
                    isConnectNewController |= (isRightConnected || (*it)->IsConnected()) && !ApplicationState::InitialConnectedController[controllerIndex].Test<nn::hid::NpadJoyAttribute::IsRightConnected>();
                }
            }
        }
        else
        {
            ApplicationState::InitialConnectedController[controllerIndex].Reset();
        }
    }

    bool isConnectedMasterController = (ApplicationState::IsHandheldMode() && isHandheldConnected) || (!ApplicationState::IsHandheldMode() && isWirelessCountollerConnected);   // マスターのコントローラーが接続されているか
    bool isConnectedSinglePlayController = isHandheldConnected || isWirelessCountollerConnected;                                            // マスターになりうるコントローラーのいずれかが接続されている場合

    //==============================================
    // コントローラーサポートアプレットの呼び出し確認
    //==============================================

    bool callApplet = false;

    if ((timerEventType._timerState != nn::os::TimerEventType::TimerState_Stop) &&
        ((isConnectedMasterController) || !check.Test<CheckPattern::CallInterval>()))
    {
        NN_LOG("## Stop TimerEvent\n");
        isWaitTimeExtended = false;
        nn::os::StopTimerEvent(&timerEventType);
        nn::os::ClearTimerEvent(&timerEventType);
    }

    if (check.Test<CheckPattern::Single>())
    {
        /*
         *  一人プレイ
         */
        nn::oe::Message message;

        if (ApplicationState::IsHandheldMode() && !isHandheldConnected && isWirelessCountollerConnected) { ApplicationState::EnableHandheldMode(false); }
        if (!ApplicationState::IsHandheldMode() && isHandheldConnected && !isWirelessCountollerConnected) { ApplicationState::EnableHandheldMode(true); }

        if (check.Test<CheckPattern::Resume>())
        {
            if (
                nn::oe::TryPopNotificationMessage(&message) &&
                (message == nn::oe::MessageResume) &&
                !isConnectedSinglePlayController
                )
            {
                // ゲーム復帰直後にコントローラーが消失していた場合 (即時実行)
                NN_LOG("## Call Controller Support Applet (Resume)\n");
                callApplet = true;
            }
        }

        if (check.Test<CheckPattern::ChangeMasterController>() && isConnectedSinglePlayController)
        {
            if (
                (timerEventType._timerState == nn::os::TimerEventType::TimerState_Stop && isConnectNewController) ||
                disableMasterControllerButtonSet.IsAnyOn())
            {
                if (!(!isConnectedMasterController && isConnectedSinglePlayController))
                {
                    // 異なるコントローラーの接続を検知した場合 (即時実行)
                    NN_LOG("## Call Controller Support Applet (ChangeMasterController%s)\n",
                        isConnectNewController ? " [ConnectNewController]" : ""
                    );
                    callApplet = true;
                }
            }
        }
        if (check.Test<CheckPattern::CallInterval>() && (!isConnectedSinglePlayController && (nn::os::TryWaitTimerEvent(&timerEventType))))
        {
            // コントローラーが消失した状態で、5秒が経過した場合
            if (!isWaitTimeExtended)
            {
                // コントローラーを消失し待ち時間が経過しても新たなコントローラーが発見されなかった場合
                NN_LOG("## Call Controller Support Applet (TimerEvent)\n");
                callApplet = true;
            }
            else
            {
                // 待ち時間が延長された場合
                NN_LOG("## Wait Time Extended( + 5s )\n");
                isWaitTimeExtended = false;
            }
        }

        if (!check.Test<CheckPattern::CallInterval>() && !isConnectedSinglePlayController)
        {
            NN_LOG("## Call Controller Support Applet (MasterControllerLost)\n");
            callApplet = true;
        }

        if (callApplet)
        {
            NN_LOG("\n\n\n");
            nn::os::StopTimerEvent(&timerEventType);
            nn::os::ClearTimerEvent(&timerEventType);
            isWaitTimeExtended = false;
        }
        else
        {
            if (check.Test<CheckPattern::CallInterval>() && !isConnectedMasterController)
            {
                if (
                    timerEventType._timerState == nn::os::TimerEventType::TimerState_Stop)
                {
                    /*
                    *  コントローラーが消失した場合に5秒間待ち時間を設けます
                    */
                    NN_LOG("## Start TimerEvent\n");
                    nn::os::StartOneShotTimerEvent(&timerEventType, nn::TimeSpan::FromSeconds(5));
                    isWaitTimeExtended = false;
                }
                if (isChangeToConsoleMode)
                {
                    /*
                     *  コントローラーが喪失している最中に
                     *  動作モードが携帯モードから据置モードに変化した場合、追加で5秒の待ち時間を設けます
                     */
                    NN_LOG("## Extension (Change Operation Mode)\n");
                    isWaitTimeExtended = true;
                }
            }
        }
    }
    else
    {
        /*
        *  複数人プレイ時
        */
        if (
            (check.Test<CheckPattern::PlayerCountShort>() && (connectedWirelessCountollerCount < ApplicationState::GetPlayerCount())) ||
            (check.Test<CheckPattern::NoneController>() && (connectedWirelessCountollerCount == 0)) ||
            (check.Test<CheckPattern::LostController>() && !isTargetControllerConnected)
            )
        {
            callApplet = true;
        }
    }
    prevOperationMode = nn::oe::GetOperationMode();
    return callApplet;
} // NOLINT(impl/function_size)

nn::Result CallControllerSupportApplet(nn::hid::ControllerSupportResultInfo* resultInfo, int playerCount)
{
    nn::Result result;
    nn::hid::ControllerSupportArg arg;
    nn::hid::ControllerSupportResultInfo info;

    if (playerCount == 1)
    {
        //  一人プレイ
        arg.SetDefault();
        arg.enableSingleMode = true;
        arg.enableIdentificationColor = true;
        arg.enableExplainText = true;
    }
    else
    {
        //  マルチプレイ
        arg.SetDefault();
        arg.playerCountMin = arg.playerCountMax = static_cast<int8_t>(playerCount);
        arg.enableIdentificationColor = true;
        arg.enableExplainText = true;
    }

    for (size_t i = 0; i < NN_ARRAY_SIZE(g_ControllerColorList); ++i)
    {
        arg.identificationColor[i] = g_ControllerColorList[i];
        nn::hid::SetExplainText(&arg, g_ControllerTextList[i], ApplicationState::NpadIds[i]);
    }

    /*
     * コントローラーサポートアプレットでは、縦持ちと下記のスタイルの混在をサポートしていません。
     * 下記のスタイルが一つでも有効な場合は
     * コントローラーサポートアプレットの呼び出し前にコントローラーの持ち方を横持ちに変更する必要があります。
     * （縦持ちにする場合は下記のスタイルを無効にしてください。）
     *
     * nn::hid::NpadStyleHandheld
     * nn::hid::NpadStyleFullKey
     * nn::hid::NpadStyleJoyDual
     */

    // nn::hid::SetNpadJoyHoldType(nn::hid::NpadJoyHoldType_Horizontal);

    // コントローラーサポートアプレットの呼び出しを実行します
    result = nn::hid::ShowControllerSupport(&info, arg);

    if (result.IsSuccess())
    {
        nn::os::LockMutex(&g_Mutex);
        {
            ApplicationState::EnableHandheldMode(info.selectedId == nn::hid::NpadId::Handheld);
        }
        nn::os::UnlockMutex(&g_Mutex);
    }

    if (resultInfo != nullptr)
    {
        *resultInfo = info;
    }

    return result;
}
