﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{HidConnectionTrigger.cpp,PageSampleHidConnectionTrigger}
 *
 * @brief
 *  コントローラーを起こす機能を示すのサンプルプログラム
 */

/**
 * @page PageSampleHidConnectionTrigger コントローラーへの接続機能
 * @tableofcontents
 *
 * @brief
 *  コントローラーを起こして接続する機能のサンプルプログラムの解説です。
 *
 * @section PageSampleConnectionTrigger_SectionBrief 概要
 *  ここでは、コントローラーを起こして接続する機能のサンプルプログラムの説明をします。
 *
 *  コントローラーを起こす機能の使い方については、Hid の関数リファレンスも併せて参照して下さい。
 *
 * @section PageSampleConnectionTrigger_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidConnectionTrigger Samples/Sources/Applications/HidConnectionTrigger @endlink 以下にあります。
 *
 * @section PageSampleConnectionTrigger_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleConnectionTrigger_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleConnectionTrigger_SectionPrecaution 注意事項
 *  とくになし
 *
 * @section PageSampleConnectionTrigger_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleConnectionTrigger_SectionDetail 解説
 *
 * @subsection PageSampleConnectionTrigger_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  HidConnectionTrigger.cpp
 *  @includelineno HidConnectionTrigger.cpp
 *
 * @subsection PageSampleConnectionTrigger_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nn::hid::GetRegisteredDevices() で登録されているデバイスをすべて列挙する
 *  - 登録されているデバイスに対して接続トリガーを送る
 *  - 接続トリガーへの処理が成功したか、タイムアウトして失敗したかを待ち受ける
 *
 *  本サンプルプログラムを起動すると、登録済みのコントローラーに対して順に接続を試みます。
 *  都度接続を試みるたびに、接続済みのコントローラーすべてに対して切断を行います。
 *
 */

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/system/hid_BluetoothConnection.h>
#include <nn/hid/system/hid_UniquePad.h>

namespace
{
    const ::nn::hid::NpadIdType NpadIds[] = { ::nn::hid::NpadId::No1,
                                             ::nn::hid::NpadId::No2,
                                             ::nn::hid::NpadId::No3,
                                             ::nn::hid::NpadId::No4,
                                             ::nn::hid::NpadId::No5,
                                             ::nn::hid::NpadId::No6,
                                             ::nn::hid::NpadId::No7,
                                             ::nn::hid::NpadId::No8,
                                             ::nn::hid::NpadId::Handheld };

    void PrintRegisteredDevices(nn::hid::system::RegisteredDevice* pDevices, int count);
    bool GetUniquePadId(nn::hid::system::UniquePadId* pOutUniquePadId, nn::bluetooth::Address& address);
    void Disconnect(nn::hid::system::UniquePadId& uniquePadId);
    void DisconnectAll();
}

extern "C" void nninitStartup()
{
    // 本サンプルはアプレット向け desc を利用しており、アプレット向けのリソース制限が適用されます。
    // ここでは、デフォルトの nninitStartup() のデフォルトメモリアロケータのサイズが
    // アプレットで利用できるサイズ上限を超えているため、自前で nninitStartup() を用意しています。
}

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::hid::InitializeNpad();
    nn::hid::SetSupportedNpadIdType(NpadIds, sizeof(NpadIds) / sizeof(NpadIds[0]));

    // Set supported NpadStyle
    ::nn::hid::SetSupportedNpadStyleSet(::nn::hid::NpadStyleFullKey::Mask |
                                        ::nn::hid::NpadStyleJoyDual::Mask |
                                        ::nn::hid::NpadStyleJoyLeft::Mask |
                                        ::nn::hid::NpadStyleJoyRight::Mask);

    ::nn::os::MultiWaitType multiWait;
    ::nn::os::SystemEventType timeoutEvent;
    ::nn::os::MultiWaitHolderType timeoutEventHolder;
    ::nn::os::SystemEventType connectionEvent;
    ::nn::os::MultiWaitHolderType connectionEventHolder;
    ::nn::os::TimerEventType timerEvent;
    ::nn::os::MultiWaitHolderType timerEventHolder;

    ::nn::os::InitializeMultiWait(&multiWait);

    ::nn::hid::system::BindConnectionTriggerTimeoutEvent(&timeoutEvent,
                                                    ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&timeoutEventHolder, &timeoutEvent);
    ::nn::os::LinkMultiWaitHolder(&multiWait, &timeoutEventHolder);

    ::nn::hid::system::BindUniquePadConnectionEvent(&connectionEvent,
                                                    ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&connectionEventHolder, &connectionEvent);
    ::nn::os::LinkMultiWaitHolder(&multiWait, &connectionEventHolder);

    ::nn::os::InitializeTimerEvent(&timerEvent,
                                   ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&timerEventHolder, &timerEvent);
    ::nn::os::LinkMultiWaitHolder(&multiWait, &timerEventHolder);

    bool runs = true;
    int deviceIndex = 0;

    ::nn::hid::system::RegisteredDevice registeredDevices[::nn::hid::system::RegisteredDeviceCountMax];
    auto registeredDeviceCount = ::nn::hid::system::GetConnectableRegisteredDevices(registeredDevices, NN_ARRAY_SIZE(registeredDevices));

    // デバイス一覧を出力
    PrintRegisteredDevices(registeredDevices, registeredDeviceCount);

    // 一発目の接続イベントに向けたタイマーをセット
    ::nn::os::StartOneShotTimerEvent(&timerEvent, ::nn::TimeSpan::FromSeconds(1));


    while(NN_STATIC_CONDITION(runs))
    {
        auto pHolder = nn::os::WaitAny(&multiWait);

        if(pHolder == &timerEventHolder)
        {
            ::nn::os::ClearTimerEvent(&timerEvent);

            // 全切断
            DisconnectAll();

            // Index のインクリメント
            deviceIndex++;
            if (deviceIndex == registeredDeviceCount)
            {
                deviceIndex = 0;
            }

            // 接続を試みる
            NN_LOG("Send Trigger to %d\n", deviceIndex);
            auto connectionResult = ::nn::hid::system::SendConnectionTrigger(registeredDevices[deviceIndex].address);
            if (connectionResult.IsFailure())
            {
                NN_LOG("[hid] Send Trigger Failed : %d\n", connectionResult.GetDescription());
                ::nn::os::StartOneShotTimerEvent(&timerEvent, ::nn::TimeSpan::FromSeconds(1));
            }
        }
        if(pHolder == &timeoutEventHolder)
        {
            ::nn::os::ClearSystemEvent(&timeoutEvent);
            NN_LOG("[%d] Connection Timeout\n", deviceIndex);

            ::nn::os::StartOneShotTimerEvent(&timerEvent, ::nn::TimeSpan::FromSeconds(1));
        }
        if(pHolder == &connectionEventHolder)
        {
            ::nn::os::ClearSystemEvent(&connectionEvent);
            ::nn::hid::system::UniquePadId uniquePadId;
            if (GetUniquePadId(&uniquePadId, registeredDevices[deviceIndex].address) == true)
            {
                NN_LOG("[%d] Connection Trigger Succeeded\n", deviceIndex);

                ::nn::os::StartOneShotTimerEvent(&timerEvent, ::nn::TimeSpan::FromSeconds(3));
            }
            else
            {
                // 接続トリガーをかけているデバイスはまだ接続されていない
            }
        }
    }
}

namespace
{
void PrintRegisteredDevices(nn::hid::system::RegisteredDevice* pDevices, int count)
{
    NN_LOG("%d Devices Registered\n", count);
    for (int i = 0; i < count; i++)
    {
        NN_LOG("[%02d] %02x-%02x-%02x-%02x-%02x-%02x Color:%02x-%02x-%02x/%02x-%02x-%02x ", i,
                                                       pDevices[i].address.address[0],
                                                       pDevices[i].address.address[1],
                                                       pDevices[i].address.address[2],
                                                       pDevices[i].address.address[3],
                                                       pDevices[i].address.address[4],
                                                       pDevices[i].address.address[5],
                                                       pDevices[i].mainColor.v[0],
                                                       pDevices[i].mainColor.v[1],
                                                       pDevices[i].mainColor.v[2],
                                                       pDevices[i].subColor.v[0],
                                                       pDevices[i].subColor.v[1],
                                                       pDevices[i].subColor.v[2]);
        if (pDevices[i].deviceType.Test<nn::hid::system::DeviceType::JoyConLeft>())
        {
            NN_LOG("Device: Joy-Con(L)\n");
        }
        else if (pDevices[i].deviceType.Test<nn::hid::system::DeviceType::JoyConRight>())
        {
            NN_LOG("Device: Joy-Con(R)\n");
        }
        else if (pDevices[i].deviceType.Test<nn::hid::system::DeviceType::SwitchProController>())
        {
            NN_LOG("Device: SwitchProController\n");
        }
        else
        {
            NN_LOG("Device: Unknown %d != %d\n", pDevices[i].deviceType._storage[0], nn::hid::system::DeviceType::JoyConLeft::Mask._storage[0]);
        }
    }
}

bool GetUniquePadId(nn::hid::system::UniquePadId* pOutUniquePadId, nn::bluetooth::Address& address)
{
    ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax];


    auto uniquePadIdCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                              sizeof(uniquePadIds) / sizeof(uniquePadIds[0]));
    for (int i = 0; i < uniquePadIdCount; i++)
    {
        ::nn::bluetooth::Address gotAddress;
        if (::nn::hid::system::GetUniquePadBluetoothAddress(&gotAddress, uniquePadIds[i]).IsSuccess())
        {
            if (gotAddress == address)
            {
                *pOutUniquePadId = uniquePadIds[i];
                return true;
            }
        }
    }
    return false;
}

void Disconnect(nn::hid::system::UniquePadId& uniquePadId)
{
    ::nn::hid::system::DisconnectUniquePad(uniquePadId);
}

void DisconnectAll()
{
    ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax];


    auto uniquePadIdCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                              sizeof(uniquePadIds) / sizeof(uniquePadIds[0]));
    for (int i = 0; i < uniquePadIdCount; i++)
    {
        Disconnect(uniquePadIds[i]);
    }
}

}
