﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{HidAnalogStickManualCalibration.cpp,PageSampleAnalogStickManualCalibration}
 *
 * @brief
 *  アナログスティックマニュアルキャリブレーションのサンプルプログラム
 */

/**
 * @page PageSampleAnalogStickManualCalibration アナログスティックマニュアルキャリブレーション
 * @tableofcontents
 *
 * @brief
 *  アナログスティックマニュアルキャリブレーションのサンプルプログラムの解説です。
 *
 * @section PageSampleAnalogStickManualCalibration_SectionBrief 概要
 *  ここでは、アナログスティックのマニュアルキャリブレーションのサンプルプログラムの説明をします。
 *
 *  アナログスティックマニュアルキャリブレーションの使い方については、Hid の関数リファレンスも併せて参照して下さい。
 *
 * @section PageSampleAnalogStickManualCalibration_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidAnalogStickManualCalibration Samples/Sources/Applications/HidAnalogStickManualCalibration @endlink 以下にあります。
 *
 * @section PageSampleAnalogStickManualCalibration_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleAnalogStickManualCalibration_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleAnalogStickManualCalibration_SectionPrecaution 注意事項
 *  とくになし
 *
 * @section PageSampleAnalogStickManualCalibration_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAnalogStickManualCalibration_SectionDetail 解説
 *
 * @subsection PageSampleAnalogStickManualCalibration_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  HidAnalogStickManualCalibration.cpp
 *  @includelineno HidAnalogStickManualCalibration.cpp
 *
 * @subsection PageSampleAnalogStickManualCalibration_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - UniquePad でデバイスを列挙する
 *  - キャリブレーションするデバイスを選択する
 *  - キャリブレーション処理を開始する
 *  - キャリブレーション中の状態をプリントする
 *
 *  本サンプルプログラムを起動してコントローラーを接続すると、キャリブレーションを行うスティックの選択が開始されます。
 *  アナログスティックを押してリリースされたデバイスが、キャリブレーションの対象デバイスとなります。
 *  再度アナログスティックを押してリリースすると、キャリブレーションが開始します。
 *  キャリブレーション中にキャンセルをする場合は、アナログスティックのボタンを押してください
 *
 */

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/system/hid_AnalogStickManualCalibration.h>
#include <nn/hid/system/hid_UniquePad.h>

namespace
{
    const ::nn::hid::NpadIdType NpadIds[] = {::nn::hid::NpadId::No1,
                                             ::nn::hid::NpadId::No2,
                                             ::nn::hid::NpadId::No3,
                                             ::nn::hid::NpadId::No4,
                                             ::nn::hid::NpadId::No5,
                                             ::nn::hid::NpadId::No6,
                                             ::nn::hid::NpadId::No7,
                                             ::nn::hid::NpadId::No8,
                                             ::nn::hid::NpadId::Handheld};


    ::nn::hid::system::UniquePadId g_UniquePadIds[::nn::hid::system::UniquePadIdCountMax];
    int g_UniquePadIdCount = 0;

    const ::nn::hid::system::AnalogStickPosition AnalogSticks[] = { ::nn::hid::system::AnalogStickPosition_Left,
                                                                    ::nn::hid::system::AnalogStickPosition_Right };
    const int AnalogStickCountMax = sizeof(AnalogSticks) / sizeof(AnalogSticks[0]);

    // キャリブレーションを行うデバイスが選択されていない
    bool g_IsDeviceSelected = false;
    // キャリブレーション処理を行っている UniquePadId
    ::nn::hid::system::UniquePadId g_UniquePadOnCalibration;
    // キャリブレーション処理を行う対象のアナログスティック
    ::nn::hid::system::AnalogStickPosition g_AnalogStickPosition;
    // 直前のボタンの押下状態(デバイス選択用)
    bool g_IsButtonPressed[::nn::hid::system::UniquePadIdCountMax][AnalogStickCountMax];
    // 直前のスティックのボタンの押下状態
    bool g_IsButtonPressedForSelectedDevice;
    // キャリブレーション処理が進行中かどうか
    bool g_IsOnAnalogStickCalibration = false;

    void PrintDevices(const ::nn::hid::system::UniquePadId* pIds,
               const int& count)
    {
        for(int i = 0; i < count; i++)
        {
            int uniquePadControllerNumber;
            ::nn::hid::system::UniquePadInterface uniquePadInterface;
            ::nn::hid::system::UniquePadType uniquePadType;
            ::nn::hid::system::UniquePadSerialNumber uniquePadSerialNumber;

            auto resultControllerNumber = ::nn::hid::system::GetUniquePadControllerNumber(&uniquePadControllerNumber,
                                                                                          pIds[i]);
            auto resultInterface = ::nn::hid::system::GetUniquePadInterface(&uniquePadInterface,
                                                                            pIds[i]);
            uniquePadType = ::nn::hid::system::GetUniquePadType(pIds[i]);

            auto resultSerialNumber = ::nn::hid::system::GetUniquePadSerialNumber(&uniquePadSerialNumber, pIds[i]);

            if(resultControllerNumber.IsSuccess() &&
               resultInterface.IsSuccess() &&
               resultSerialNumber.IsSuccess())
            {
                NN_LOG("Id=0x%x, ControllerNumber=%d, Interface=%x, Type=%x\n",
                    pIds[i]._storage,
                    resultControllerNumber.IsSuccess() ? uniquePadControllerNumber : -1,
                    resultInterface.IsSuccess() ? uniquePadInterface : -1,
                    uniquePadType);
                NN_LOG("SerialNumber: ");
                for (auto& c : uniquePadSerialNumber._storage)
                {
                    NN_LOG("%c", c);
                }
                NN_LOG("\n");
            }
        }

        NN_LOG("Press your stick to select analog stick to calibrate\n");
    }

    void PrintAnalogStickState(::nn::hid::AnalogStickState& state)
    {
        NN_LOG("[id:%d] value: x:%05d y:%05d\n", g_UniquePadOnCalibration, state.x, state.y);
    }

    void PrintDirection(::nn::hid::system::AnalogStickManualCalibrationStage stage)
    {
        switch (stage)
        {
        case ::nn::hid::system::AnalogStickManualCalibrationStage_ReleaseFromRight:
            NN_LOG("Right");
            break;
        case ::nn::hid::system::AnalogStickManualCalibrationStage_ReleaseFromBottom:
            NN_LOG("Bottom");
            break;
        case ::nn::hid::system::AnalogStickManualCalibrationStage_ReleaseFromLeft:
            NN_LOG("Left");
            break;
        case ::nn::hid::system::AnalogStickManualCalibrationStage_ReleaseFromTop:
            NN_LOG("Top");
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void PrintAnalogStickCalibrationState(::nn::hid::system::AnalogStickManualCalibrationStage stage,
                                          ::nn::hid::AnalogStickState& state,
                                          bool inReleasePosition,
                                          bool inCircuitPosition)
    {
        switch (stage)
        {
        case ::nn::hid::system::AnalogStickManualCalibrationStage_ReleaseFromRight:
        case ::nn::hid::system::AnalogStickManualCalibrationStage_ReleaseFromBottom:
        case ::nn::hid::system::AnalogStickManualCalibrationStage_ReleaseFromLeft:
        case ::nn::hid::system::AnalogStickManualCalibrationStage_ReleaseFromTop:
            if (!inReleasePosition)
            {
                NN_LOG("Put the stick to ");
                PrintDirection(stage);
                NN_LOG(" end ->");
            }
            else
            {
                NN_LOG("Release your finger from Stick -> ");
            }
            PrintAnalogStickState(state);
            break;
        case ::nn::hid::system::AnalogStickManualCalibrationStage_Rotate:
            NN_LOG("Rotate the stick around the edge [%c]-> ", (inCircuitPosition ? 'o' : 'x' ));
            PrintAnalogStickState(state);
            break;
        case ::nn::hid::system::AnalogStickManualCalibrationStage_Update:
            NN_LOG("Updateing Calibration Value\n");
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    // デバイスの接続状態に更新があったことをハンドリングする
    void HandleUniquePadConnection()
    {
        // UniquePadId の更新
        g_UniquePadIdCount = ::nn::hid::system::ListUniquePads(g_UniquePadIds,
                                                               sizeof(g_UniquePadIds) / sizeof(g_UniquePadIds[0]));
        PrintDevices(g_UniquePadIds, g_UniquePadIdCount);
    }

    // デバイスの選択処理。スティックの押し込みボタンが選択されたデバイスについてキャリブレーションが行われる
    void ProceedDeviceSelection()
    {
        for (int i = 0; i < g_UniquePadIdCount; i++)
        {
            for (int j = 0; j < AnalogStickCountMax; j++)
            {
                // スティックのボタンがリリースされたデバイスについて、キャリブレーションを開始する
                auto isButtonPressed = ::nn::hid::system::IsAnalogStickButtonPressed(g_UniquePadIds[i], AnalogSticks[j]);
//                NN_LOG("Index : %d(%s)\n", j, isButtonPressed ? "on" : "off");
                if(g_IsButtonPressed[i][j] && !isButtonPressed)
                {
                    g_IsDeviceSelected = true;
                    g_UniquePadOnCalibration = g_UniquePadIds[i];
                    g_AnalogStickPosition = AnalogSticks[j];
                    g_IsButtonPressed[i][j] = false;
                    return;
                }
                g_IsButtonPressed[i][j] = isButtonPressed;
            }
        }
    }

    // アナログスティックのキャリブレーション開始待ち
    void WaitForAnalogStickCalibrationToStart()
    {
        // デバイスの状態をチェック
        int uniquePadControllerNumber;
        if (::nn::hid::system::GetUniquePadControllerNumber(&uniquePadControllerNumber,
            g_UniquePadOnCalibration).IsFailure())
        {
            // 対象のデバイスが切断された
            NN_LOG("Device Detached\n");
            g_IsDeviceSelected = false;
            g_IsOnAnalogStickCalibration = false;
            return;
        }

        // アナログスティックのボタンの押下状態
        auto isButtonPressed = ::nn::hid::system::IsAnalogStickButtonPressed(g_UniquePadOnCalibration, g_AnalogStickPosition);

        // スティックのボタンがリリースされたかどうかチェック
        // アナログスティックのキャリブレーションはボタンが押されてない状況で開始するのが望ましいので、リリーストリガとする
        if (g_IsButtonPressedForSelectedDevice && !isButtonPressed)
        {
            NN_LOG("Analog Stick Button Preesed\n");
            auto result = ::nn::hid::system::StartAnalogStickManualCalibration(g_UniquePadOnCalibration, g_AnalogStickPosition);
            if (::nn::hid::system::ResultAnalogStickManualCalibrationNotSupported().Includes(result))
            {
                // アナログスティックの手動補正に対応しないデバイス
                NN_LOG("This device not support Manual Calibration\n");
            }
            if (::nn::hid::system::ResultAnalogStickDeviceNotConnected().Includes(result))
            {
                NN_LOG("Device Detached\n");
                // 対象のデバイスが切断された
                g_IsDeviceSelected = false;
            }
            NN_LOG("Analog Stick Calibration Started\n");
            g_IsOnAnalogStickCalibration = true;
        }
        g_IsButtonPressedForSelectedDevice = isButtonPressed;

        // アナログスティックの入力状態をプリント
        ::nn::hid::AnalogStickState analogStickState;
        ::nn::hid::system::GetAnalogStickState(&analogStickState, g_UniquePadOnCalibration, g_AnalogStickPosition);
        NN_LOG("Press Analog Stick to Start Calibration -> ");
        PrintAnalogStickState(analogStickState);
    }

    // 選択されたデバイスに対して、アナログスティックのキャリブレーション処理を行う
    void ProceedAnalogStickCalibration()
    {
        // アナログスティックのキャリブレーション状況を取得する
        ::nn::hid::system::AnalogStickManualCalibrationStage stage;
        auto result = nn::hid::system::GetAnalogStickManualCalibrationStage(&stage, g_UniquePadOnCalibration, g_AnalogStickPosition);
        if(::nn::hid::system::ResultAnalogStickDeviceNotConnected().Includes(result) ||
           ::nn::hid::system::ResultUniquePadDisconnected().Includes(result))
        {
            // 対象のデバイスが切断された
            NN_LOG("Device Detached\n");
            g_IsDeviceSelected = false;
            g_IsOnAnalogStickCalibration = false;
            return;
        }

        // キャリブレーション処理が完了
        if (stage == ::nn::hid::system::AnalogStickManualCalibrationStage_Completed)
        {
            g_IsDeviceSelected = false;
            g_IsOnAnalogStickCalibration = false;
            NN_LOG("Analog Stick Calibration Completed\n");
            return;
        }
        // アナログスティックのボタンの押下状態
        auto isButtonPressed = ::nn::hid::system::IsAnalogStickButtonPressed(g_UniquePadOnCalibration, g_AnalogStickPosition);

        // スティックのボタンがリリースされたかどうかチェック
        if (g_IsButtonPressedForSelectedDevice && !isButtonPressed)
        {
            // キャリブレーション中は処理をキャンセル
            ::nn::hid::system::CancelAnalogStickManualCalibration(g_UniquePadOnCalibration, g_AnalogStickPosition);
            g_IsOnAnalogStickCalibration = false;
        }
        g_IsButtonPressedForSelectedDevice = isButtonPressed;

        // ボタンがリリースポジションにあるかどうか (原点キャリブレーション処理)
        auto isInReleasePosition = ::nn::hid::system::IsAnalogStickInReleasePosition(g_UniquePadOnCalibration, g_AnalogStickPosition);

        // ボタンが外周に沿って移動しているかどうか (外周キャリブレーション処理)
        auto isOnCircumference = ::nn::hid::system::IsAnalogStickInCircumference(g_UniquePadOnCalibration, g_AnalogStickPosition);

        // アナログスティックの入力状態を取得
        ::nn::hid::AnalogStickState analogStickState;
        ::nn::hid::system::GetAnalogStickState(&analogStickState, g_UniquePadOnCalibration, g_AnalogStickPosition);

        // キャリブレーションの進捗をプリント
        PrintAnalogStickCalibrationState(stage, analogStickState, isInReleasePosition, isOnCircumference);
    }

    void Proceed()
    {
        if (g_IsDeviceSelected == false)
        {
            ProceedDeviceSelection();
        }
        else if(g_IsOnAnalogStickCalibration == false)
        {
            WaitForAnalogStickCalibrationToStart();
        }
        else
        {
            ProceedAnalogStickCalibration();
        }
    }
} //namespace

extern "C" void nninitStartup()
{
    // 本サンプルはアプレット向け desc を利用しており、アプレット向けのリソース制限が適用されます。
    // ここでは、デフォルトの nninitStartup() のデフォルトメモリアロケータのサイズが
    // アプレットで利用できるサイズ上限を超えているため、自前で nninitStartup() を用意しています。
}

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::hid::InitializeNpad();
    nn::hid::SetSupportedNpadIdType(NpadIds, sizeof(NpadIds) / sizeof(NpadIds[0]));

    // Set supported NpadStyle
    ::nn::hid::SetSupportedNpadStyleSet(::nn::hid::NpadStyleFullKey::Mask |
                                        ::nn::hid::NpadStyleJoyDual::Mask |
                                        ::nn::hid::NpadStyleJoyLeft::Mask |
                                        ::nn::hid::NpadStyleJoyRight::Mask);

    ::nn::os::MultiWaitType multiWait;
    ::nn::os::SystemEventType connectionEvent;
    ::nn::os::MultiWaitHolderType connectionEventHolder;
    ::nn::os::TimerEventType timerEvent;
    ::nn::os::MultiWaitHolderType timerEventHolder;

    ::nn::os::InitializeMultiWait(&multiWait);

    ::nn::hid::system::BindUniquePadConnectionEvent(&connectionEvent,
                                                    ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&connectionEventHolder, &connectionEvent);
    ::nn::os::LinkMultiWaitHolder(&multiWait, &connectionEventHolder);

    ::nn::os::InitializeTimerEvent(&timerEvent,
                                   ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&timerEventHolder, &timerEvent);
    ::nn::os::LinkMultiWaitHolder(&multiWait, &timerEventHolder);
    ::nn::os::StartPeriodicTimerEvent(&timerEvent, ::nn::TimeSpan::FromMilliSeconds(15), ::nn::TimeSpan::FromMilliSeconds(15));

    bool runs = true;

    while(NN_STATIC_CONDITION(runs))
    {
        auto pHolder = nn::os::WaitAny(&multiWait);

        if(pHolder == &connectionEventHolder)
        {
            ::nn::os::ClearSystemEvent(&connectionEvent);
            HandleUniquePadConnection();
        }
        if(pHolder == &timerEventHolder)
        {
            ::nn::os::ClearTimerEvent(&timerEvent);
            Proceed();
        }
    }
}
