﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// #define VCC

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/init.h>
#include <nn/tma/tma.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_ThreadApi.h>
#include <nne/audio/audio.h>
#include <nn/fs.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nne/vcc/vcc_API.h>

using namespace nn;
using namespace nne::audio;

namespace HdaTest {

//***************************************************************************
//***************************************************************************
//
// Heap
//
//***************************************************************************
//***************************************************************************

namespace {

    const int heapSize = 512 * 1024;
    const int SIN_ENTRIES = 64;
#ifdef NN_SDK_BUILD_RELEASE
    const bool adspStartLogs = false;
#else
    const bool adspStartLogs = true;
#endif
    const uint16_t SinWave[SIN_ENTRIES] =
    {
        0x0000, 0x0645, 0x0C7C, 0x1293, 0x187D, 0x1E2B, 0x238E, 0x2899,
        0x2D41, 0x3178, 0x3536, 0x3871, 0x3B20, 0x3D3E, 0x3EC4, 0x3FB0,
        0x3FFF, 0x3FB0, 0x3EC4, 0x3D3E, 0x3B20, 0x3871, 0x3536, 0x3178,
        0x2D41, 0x2899, 0x238E, 0x1E2B, 0x187D, 0x1293, 0x0C7C, 0x0645,
        0x0000, 0xF9BA, 0xF383, 0xED6C, 0xE782, 0xE1D4, 0xDC71, 0xD766,
        0xD2BE, 0xCE87, 0xCAC9, 0xC78E, 0xC4DF, 0xC2C1, 0xC13B, 0xC04F,
        0xC000, 0xC04F, 0xC13B, 0xC2C1, 0xC4DF, 0xC78E, 0xCAC9, 0xCE87,
        0xD2BE, 0xD766, 0xDC71, 0xE1D4, 0xE782, 0xED6C, 0xF383, 0xF9BA
    };
    const uint32_t  PlaybackTimeInSeconds = 5;
    const size_t HeapSize = 256 * 1024 * 1024;

    uint8_t heapBuffer[heapSize];
    lmem::HeapHandle heap;
}

//***************************************************************************
// heapAllocate
//***************************************************************************
void* heapAllocate(size_t size)
{
    return lmem::AllocateFromExpHeap(heap, size);
}

//***************************************************************************
// heapDeallocate
//***************************************************************************
void heapDeallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    lmem::FreeToExpHeap(heap, p);
}


//***************************************************************************
// PlayTest
//***************************************************************************
void PlayTest(nne::audio::hda::AUDIO_HDA_SAMPLERATE  SampleRate, nne::audio::hda::AUDIO_HDA_BITSPERSAMPLE BitsPerSample, nne::audio::hda::AUDIO_HDA_NUMBEROFCHANNELS NumberOfChannels)
{
    os::SemaphoreType processSemaphore;
    NN_LOG("InitializeSemaphore\n");
    os::InitializeSemaphore(&processSemaphore, 0, 1000);
    NN_LOG("gmix::Initialize\n");
    gmix::Initialize();
    gmix::SetProcessSemaphore(gmix::Session::Name::Default, &processSemaphore);
    NN_LOG("hda::Initialize\n");
    hda::Initialize(nne::audio::hda::AUDIO_HDA_FORMAT_LPCM, SampleRate, BitsPerSample, NumberOfChannels);
    device::Session *play = nullptr;
    NN_LOG("OpenSession\n");
    hda::OpenSession(&play, 0);
    if (nullptr != play)
    {
        size_t bufferSize = play->GetBufferSize();
        size_t numSamples = (bufferSize >> 1); // ASSUMES 16 bit samples
        uint16_t* buffer = (uint16_t*) play->GetBufferAddress();

        for(uint32_t i = 0;i<numSamples;i+=NumberOfChannels)
        {
            static uint32_t j = 0;

            for(uint32_t c = 0;c<NumberOfChannels;c++)
            {
                buffer[i + c] = SinWave[j];
            }
            j = (j + 1) % SIN_ENTRIES;
        }

        play->Start();

        NN_LOG("Streaming %dKHz %d channel stream for %d seconds\n", SampleRate, NumberOfChannels, PlaybackTimeInSeconds);
        uint32_t total = 0;
        for (uint32_t i = 0; i < PlaybackTimeInSeconds * 50; i++)
        {
            static size_t posLast = 0;

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
            size_t pos = play->GetPosition();

            if(pos>=posLast)
            {
                total += pos - posLast;
                posLast = pos;
            }
            else
            {
                total += bufferSize - posLast + pos;
                posLast = pos;
            }
        }

        NN_LOG("Streamed %d bytes in %d seconds\n", total, PlaybackTimeInSeconds);

        play->Stop();

        hda::CloseSession(play);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
    }

    hda::Finalize();
    gmix::Finalize();
    os::FinalizeSemaphore(&processSemaphore);
}


//***************************************************************************
//***************************************************************************
//
// External Entry Points
//
//***************************************************************************
//***************************************************************************

extern "C" void nninitStartup()
{
    const size_t MallocMemorySize = 16 * 1024 * 1024;
    uintptr_t address;
    nn::Result result;

    NN_LOG("SettingMemoryHeapSize...\n");
    result = nn::os::SetMemoryHeapSize(HeapSize);
    if (!result.IsSuccess())
    {
        NN_LOG("SetMemoryHeapSize failed.\n");
        return;
    }

    NN_LOG("AllocatingMemoryBlock...\n");
    result = nn::os::AllocateMemoryBlock(&address, MallocMemorySize);
    NN_ASSERT(result.IsSuccess());
    NN_LOG("InitializingAllocator...\n");
    nn::init::InitializeAllocator(reinterpret_cast<void *>(address), MallocMemorySize);
    NN_LOG("AllocatorInialized...\n");
}

//***************************************************************************
// nnMain
//***************************************************************************
extern "C" void nnMain()
{
    NN_LOG("TMAInitialize...\n");
    tma::Initialize();

    NN_LOG("[HdaTest:%s]\n", __FUNCTION__);

    NN_LOG("CreateExpHeap...\n");
    heap = lmem::CreateExpHeap(heapBuffer, heapSize, nn::lmem::CreationOption_NoOption);
    fs::SetAllocator(heapAllocate, heapDeallocate);

    uintptr_t adspBin, vectorBin;
    size_t adspSize, vectorSize;
    firmware::Initialize(&adspBin, &adspSize, &vectorBin, &vectorSize);

#if defined(VCC)
    nne::vcc::Initialize();
#endif
    adsp::Initialize(adspBin, adspSize, vectorBin, vectorSize, adspStartLogs);

    //PlayTest(hda::AUDIO_HDA_SAMPLERATE_32000, hda::AUDIO_HDA_BITSPERSAMPLE_16, hda::AUDIO_HDA_NUMBEROFCHANNELS_2);
    //PlayTest(hda::AUDIO_HDA_SAMPLERATE_44100, hda::AUDIO_HDA_BITSPERSAMPLE_16, hda::AUDIO_HDA_NUMBEROFCHANNELS_2);
    PlayTest(hda::AUDIO_HDA_SAMPLERATE_48000, hda::AUDIO_HDA_BITSPERSAMPLE_16, hda::AUDIO_HDA_NUMBEROFCHANNELS_2);

    //PlayTest(hda::AUDIO_HDA_SAMPLERATE_32000, hda::AUDIO_HDA_BITSPERSAMPLE_16, hda::AUDIO_HDA_NUMBEROFCHANNELS_6);
    //PlayTest(hda::AUDIO_HDA_SAMPLERATE_44100, hda::AUDIO_HDA_BITSPERSAMPLE_16, hda::AUDIO_HDA_NUMBEROFCHANNELS_6);
    PlayTest(hda::AUDIO_HDA_SAMPLERATE_48000, hda::AUDIO_HDA_BITSPERSAMPLE_16, hda::AUDIO_HDA_NUMBEROFCHANNELS_6);

    adsp::Finalize();
    firmware::Finalize();

    lmem::DestroyExpHeap(heap);

    tma::Finalize();
}

}
