﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>

#include "Renderer.h"

void Renderer::Initialize()
{
    const char* VertexShaderSource =
        "#version 450\n"
        "layout( location = 0 ) in vec4 iPosition;\n"
        "layout( location = 0 ) out vec4 oColor;\n"
        "layout( binding = 0 ) uniform Param {\n"
        "    int frame;\n"
        "};\n"
        "void main() {\n"
        "    float xPos = iPosition.x * ( -( int( float( frame ) / 40.0 / 3.1416 ) % 2 ) * 2 + 1 );\n"
        "    gl_Position = vec4( xPos + cos( float( frame ) / 40.0 ) * 0.6, iPosition.y + cos( float( frame ) / 210.0 ) * 0.6, 0.0, 1.0 );\n"
        "    oColor = vec4( gl_Position.xy * 0.5 + 0.5, sin( float( frame ) / 80.0 ) * 0.5, 1.0 );\n"
        "}\n";

    const char* FragmentShaderSource =
        "#version 450\n"
        "layout( location = 0 ) in vec4 iColor;\n"
        "layout( location = 0 ) out vec4 oColor;\n"
        "void main() {\n"
        "    oColor = iColor;\n"
        "}\n";

    GLint result;
    GLchar infoLog[ 1024 ] = {};

    m_hVertexShader = glCreateShader( GL_VERTEX_SHADER );
    NN_ASSERT_NOT_EQUAL( m_hVertexShader, 0u );
    glShaderSource( m_hVertexShader, 1, &VertexShaderSource, nullptr );
    glCompileShader( m_hVertexShader );
    glGetShaderiv( m_hVertexShader, GL_COMPILE_STATUS, &result );
    if( result == GL_FALSE )
    {
        glGetShaderInfoLog( m_hVertexShader, sizeof( infoLog ), nullptr, infoLog );
        NN_ASSERT( false, "Failed to compile vertex shader: %s\n", infoLog );
    }

    m_hFragmentShader = glCreateShader( GL_FRAGMENT_SHADER );
    NN_ASSERT_NOT_EQUAL( m_hFragmentShader, 0u );
    glShaderSource( m_hFragmentShader, 1, &FragmentShaderSource, nullptr );
    glCompileShader( m_hFragmentShader );
    glGetShaderiv( m_hFragmentShader, GL_COMPILE_STATUS, &result );
    if( result == GL_FALSE )
    {
        glGetShaderInfoLog( m_hFragmentShader, sizeof( infoLog ), nullptr, infoLog );
        NN_ASSERT( false, "Failed to compile fragment shader: %s\n", infoLog );
    }

    m_hProgram = glCreateProgram();
    NN_ASSERT_NOT_EQUAL( m_hProgram, 0u );
    glAttachShader( m_hProgram, m_hVertexShader );
    glAttachShader( m_hProgram, m_hFragmentShader );
    glLinkProgram( m_hProgram );
    glGetProgramiv( m_hProgram, GL_LINK_STATUS, &result );
    if( result == GL_FALSE )
    {
        glGetProgramInfoLog( m_hProgram, sizeof( infoLog ), nullptr, infoLog );
        NN_ASSERT( false, "Failed to link program: %s\n", infoLog );
    }

    const float length = 0.3f;
    const float Vertices[] =
    {
        -length, 0.0f,
        0.0f, length,
        0.0f, -length,
        0.0f, length,
        length, 0.0f,
        0.0f, -length,
        length, 0.0f,
        length * 1.5f, length,
        length * 1.5f, -length
    };

    glGenBuffers( 1, &m_hVertexBuffer );
    NN_ASSERT_NOT_EQUAL( m_hVertexBuffer, 0u );
    glBindBuffer( GL_ARRAY_BUFFER, m_hVertexBuffer );
    glBufferStorage( GL_ARRAY_BUFFER, sizeof( Vertices ), Vertices, 0 );

    glGenVertexArrays( 1, &m_hVao );
    NN_ASSERT_NOT_EQUAL( m_hVao, 0u );
    glBindVertexArray( m_hVao );
    glEnableVertexArrayAttrib( m_hVao, 0 );
    glVertexAttribFormat( 0, 2, GL_FLOAT, GL_FALSE, 0 );
    glVertexAttribBinding( 0, 0 );

    glGenBuffers( 1, &m_hUbo );
    NN_ASSERT_NOT_EQUAL( m_hUbo, 0u );
    glBindBuffer( GL_UNIFORM_BUFFER, m_hUbo );
    glBufferStorage( GL_UNIFORM_BUFFER, sizeof( int ), nullptr, GL_MAP_WRITE_BIT |
        GL_MAP_PERSISTENT_BIT | GL_MAP_COHERENT_BIT );
    m_pMapped = static_cast< int* >( glMapBufferRange( GL_UNIFORM_BUFFER, 0, sizeof( int ),
        GL_MAP_WRITE_BIT | GL_MAP_PERSISTENT_BIT | GL_MAP_COHERENT_BIT ) );
    NN_ASSERT_NOT_NULL( m_pMapped );
}

void Renderer::Finalize()
{
    glBindBuffer( GL_UNIFORM_BUFFER, 0 );
    glBindVertexArray( 0 );
    glUseProgram( 0 );
    glDetachShader( m_hProgram, m_hVertexShader );
    glDetachShader( m_hProgram, m_hFragmentShader );

    glDeleteBuffers( 1, &m_hUbo );
    glDeleteVertexArrays( 1, &m_hVao );
    glDeleteBuffers( 1, &m_hVertexBuffer );
    glDeleteProgram( m_hProgram );
    glDeleteShader( m_hVertexShader );
    glDeleteShader( m_hFragmentShader );
}

void Renderer::Render( int frame )
{
    *m_pMapped = frame;
    float color[] = { 0.0f, 0.0f, 0.4f, 1.0f };
    glClearBufferfv( GL_COLOR, 0, color );
    glUseProgram( m_hProgram );
    glBindVertexArray( m_hVao );
    glBindVertexBuffer( 0, m_hVertexBuffer, 0, sizeof( float ) * 2 );
    glBindBufferRange( GL_UNIFORM_BUFFER, 0, m_hUbo, 0, sizeof( int ) );
    glMemoryBarrier( GL_CLIENT_MAPPED_BUFFER_BARRIER_BIT );
    glDrawArrays( GL_TRIANGLES, 0, 9 );
}
