﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{GllSimple.cpp,PageSampleGllSimple}
 *
 * @brief
 *  OpenGL の初期化処理を含むシンプルなサンプルプログラム
 */

/**
 * @page PageSampleGllSimple GllSimple
 * @tableofcontents
 *
 * @brief
 *  OpenGL の初期化処理を含むシンプルなサンプルプログラムの解説です。
 *
 * @section PageSampleGllSimple_SectionBrief 概要
 *  gll を使用して OpenGL のコマンドをロードし、ロードした機能を呼び出したり、
 * 機能がサポートされているかどうかを判定するシンプルなサンプルです。
 * ロードした機能を使って簡単な画面表示も行います。
 *
 * @section PageSampleGllSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/GllSimple
 *  Samples/Sources/Applications/GllSimple @endlink 以下にあります。
 *
 * @section PageSampleGllSimple_SectionNecessaryEnvironment 必要な環境
 *  画面表示が利用可能である必要があります。
 *
 * @section PageSampleGllSimple_SectionHowToOperate 操作方法
 *  画面をタッチまたはクリックすると終了します。
 *
 * @section PageSampleGllSimple_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleGllSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleGllSimple_SectionDetail 解説
 *  このサンプルプログラムは、まず gll を使って OpenGL のコマンドをロードします。
 *  次に gll を使って特定の機能がサポートされているかどうかを判定し、判定の結果によって処理を分岐します。
 *  その後、ロードされた OpenGL の機能を使って簡単なレンダリングを行います。
 *
 * このサンプルプログラムの処理の流れは以下の通りです。
 *
 * - ディスプレイを取得・レイヤーを初期化
 * - OpenGL のコンテキストを作成
 * - OpenGL を初期化
 * - 特定の機能がサポートされているかどうかにもとづいて分岐した処理を実行
 * - 画面へレンダリング
 */

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/init.h>
#include <nn/hid.h>
#include <nn/gll.h>

#include "GraphicsHelper.h"
#include "Renderer.h"

extern "C" void nnMain()
{
    nn::hid::InitializeTouchScreen();
    nn::hid::TouchScreenState< 1 > touchScreenState = {};

    GraphicsHelper graphicsHelper;
    graphicsHelper.Initialize();

    nngllResult gllResult = nngllInitializeGl();
    NN_UNUSED( gllResult );
    NN_ASSERT_EQUAL( gllResult, nngllResult_Succeeded );

    NN_LOG( "GL_VERSION: %s\n", ::glGetString( GL_VERSION ) );
    NN_LOG( "GL_SHADING_LANGUAGE_VERSION: %s\n", glGetString( GL_SHADING_LANGUAGE_VERSION ) );

    NN_LOG( "\nSupported extensions:\n" );

    // 関数ポインタが null かどうかを調べることにより、その関数がロードされているかどうかを判定できます。
    if( glGetStringi != nullptr )
    {
        // GL 3.0 以降
        int extensionCount = 0;
        glGetIntegerv( GL_NUM_EXTENSIONS, &extensionCount );
        for( int idxExtension = 0; idxExtension < extensionCount; ++idxExtension )
        {
            auto pExtensionName = reinterpret_cast< const char* >( ::glGetStringi( GL_EXTENSIONS, idxExtension ) );
            NN_LOG( "%s\n", pExtensionName );
        }
    }
    else
    {
        // GL 3.0 以前
        for( auto pExtensionNames = reinterpret_cast<const char*>( ::glGetString( GL_EXTENSIONS ) );
            *pExtensionNames != '\0'; ++pExtensionNames )
        {
            NN_LOG( "%c", *pExtensionNames == ' ' ? '\n' : *pExtensionNames );
        }
    }

    // NN_GLL_IS_SUPPORTED() に拡張名を渡すことにより、その拡張がサポートされているかどうかを判定できます。
    if( NN_GLL_IS_SUPPORTED( GL_EXT_depth_bounds_test ) )
    {
        // GL_EXT_depth_bounds_test がサポートされている
        glEnable( GL_DEPTH_BOUNDS_TEST_EXT );
        glDepthBoundsEXT( 0.0f, 0.5f );
    }

    Renderer renderer;
    renderer.Initialize();

    for( int frame = 0; touchScreenState.count < 1; ++frame )
    {
        renderer.Render( frame );
        glFinish();

        graphicsHelper.SwapBuffers();

        nn::hid::GetTouchScreenStates( &touchScreenState, 1 );
    }

    renderer.Finalize();

    graphicsHelper.Finalize();
}
