﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{GfxSimpleCompute.cpp,PageSampleGfxSimpleCompute}
 *
 * @brief
 *  シンプルな GPU 演算処理のサンプルプログラム
 */

/**
 * @page PageSampleGfxSimpleCompute GfxSimpleCompute
 * @tableofcontents
 *
 * @brief
 *  シンプルな GPU 演算処理のサンプルプログラムの解説です。
 *
 * @section PageSampleGfxSimpleCompute_SectionBrief 概要
 *  gfx を使用して、演算シェーダーによる汎用計算を行うシンプルなサンプルです。
 *
 * @section PageSampleGfxSimpleCompute_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/GfxSimpleCompute
 *  Samples/Sources/Applications/GfxSimpleCompute @endlink 以下にあります。
 *
 * @section PageSampleGfxSimpleCompute_SectionNecessaryEnvironment 必要な環境
 *  特にありません。
 *
 * @section PageSampleGfxSimpleCompute_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleGfxSimpleCompute_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleGfxSimpleCompute_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleGfxSimpleCompute_SectionDetail 解説
 *  このサンプルプログラムは、演算シェーダーリソースを読み込み、
 *  単純な配列に対して GPU により並列に処理を行います。
 *  gfx を使って GPU の処理を行うおおよそ最小限のサンプルとなっています。
 *  利用するリソースはビルド時にビルド前イベントにからコンバートされています。
 *  リソースのコンバートにシェーダーコンバーターを利用しています。
 *
 * このサンプルプログラムの処理の流れは以下の通りです。
 *
 * - デバイスを初期化
 * - メモリープールを初期化
 * - キューを初期化
 * - 演算シェーダーリソースを初期化
 * - コマンドバッファーを初期化
 * - タイムスタンプクエリー用のバッファーを初期化
 * - アンオーダードアクセスバッファーを初期化
 * - 計算対象のバッファーの初期値を書き込み
 * - コマンドリストを作成
 * - コマンドリストを実行
 * - GPU の完了を待機
 * - 実行にかかった時間を取得
 * - 計算結果を検証
 * - 各種オブジェクトを破棄
 *
 * このサンプルプログラムでは計算結果が期待通りかどうかの検証を行っています。
 * このサンプルプログラムの成功時の実行結果を以下に示します（かかった時間の表示部分は実行毎に異なります）。
 *
 * @verbinclude GfxSimpleCompute_OutputExample.txt
 */

#include <cstdlib>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/gfx.h>

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    #include <nv/nv_MemoryManagement.h>
#endif

namespace {

const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
const size_t TargetArrayLength = 64 * 1024; // 計算する配列の長さ

void* g_pPoolMemory = NULL;
ptrdiff_t g_MemoryPoolOffset = 0;

nn::util::BytePtr g_pMemoryHeap( NULL );
nn::util::BytePtr g_pMemory( NULL );

nn::gfx::Device g_Device;
nn::gfx::MemoryPool g_MemoryPool;
nn::gfx::MemoryPool g_CommandBufferMemoryPool;
nn::gfx::Queue g_Queue;
nn::gfx::CommandBuffer g_CommandBuffer;
nn::gfx::Buffer g_QueryBuffer;
nn::gfx::Buffer g_ExportBuffer;
nn::gfx::Fence g_Fence;
nn::gfx::ResShaderFile* g_pComputeShaderFile;
nn::gfx::ResShaderProgram* g_pComputeShaderProgram;
int g_SlotExportBuffer;

}

void InitializeDevice()
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetApiVersion( nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion );
    g_Device.Initialize( info );
}

void InitializeMemoryPool()
{
    nn::gfx::MemoryPool::InfoType info;
    info.SetDefault();
    info.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuCached
        | nn::gfx::MemoryPoolProperty_GpuCached );
    void* pPoolMemoryStart = nn::util::BytePtr( g_pPoolMemory ).AlignUp(
        nn::gfx::MemoryPool::GetPoolMemoryAlignment( &g_Device, info ) ).Get();
    info.SetPoolMemory( pPoolMemoryStart, nn::util::align_up( 1024 * 1024,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( &g_Device, info ) ) );
    g_MemoryPool.Initialize( &g_Device, info );

    info.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached
        | nn::gfx::MemoryPoolProperty_GpuUncached );
    pPoolMemoryStart = nn::util::BytePtr( pPoolMemoryStart, info.GetPoolMemorySize() ).AlignUp(
        nn::gfx::MemoryPool::GetPoolMemoryAlignment( &g_Device, info ) ).Get();
    info.SetPoolMemory( pPoolMemoryStart, nn::util::align_up( 1024 * 1024,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( &g_Device, info ) ) );
    g_CommandBufferMemoryPool.Initialize( &g_Device, info );

    g_MemoryPoolOffset = 0;
}

void InitializeQueue()
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability( nn::gfx::QueueCapability_Compute );
    g_Queue.Initialize( &g_Device, info );
}

void InitializeFence()
{
    nn::gfx::FenceInfo info;
    info.SetDefault();
    g_Fence.Initialize( &g_Device, info );
}

void InitializeComputeShader()
{
    nn::Result result;
    nn::fs::FileHandle hFile;

    result = nn::fs::OpenFile( &hFile, "Contents:/ComputeShader.bnsh", nn::fs::OpenMode_Read );
    NN_ASSERT( result.IsSuccess() );

    nn::util::BinaryFileHeader fileHeader;
    size_t readSize;
    result = nn::fs::ReadFile( &readSize, hFile, 0, &fileHeader, sizeof( nn::util::BinaryFileHeader ) );
    NN_ASSERT( result.IsSuccess() );
    NN_ASSERT( readSize == sizeof( nn::util::BinaryFileHeader ) );

    int64_t fileSize;
    result = nn::fs::GetFileSize( &fileSize, hFile );
    NN_ASSERT( result.IsSuccess() );
    g_pMemory.AlignUp( fileHeader.GetAlignment() );
    void* pBuffer = g_pMemory.Get();
    result = nn::fs::ReadFile( &readSize, hFile, 0, pBuffer, static_cast< size_t >( fileSize ) );
    NN_ASSERT( result.IsSuccess() );
    NN_ASSERT( readSize == static_cast< size_t >( fileSize ) );
    nn::fs::CloseFile( hFile );
    g_pMemory.Advance( static_cast< ptrdiff_t >( fileSize ) );

    g_pComputeShaderFile = nn::gfx::ResShaderFile::ResCast( pBuffer );
    nn::gfx::ResShaderContainer* pContainer = g_pComputeShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL( pContainer );
    pContainer->Initialize( &g_Device );

    // このサンプルで使うシェーダーはバリエーションを使っていません。
    NN_ASSERT( pContainer->GetShaderVariationCount() == 1 );
    g_pComputeShaderProgram = pContainer->GetResShaderProgram( 0 );
    nn::gfx::ShaderInitializeResult shaderResult = g_pComputeShaderProgram->Initialize( &g_Device );
    NN_ASSERT_EQUAL( shaderResult, nn::gfx::ShaderInitializeResult_Success );
    NN_UNUSED( shaderResult );
    nn::gfx::Shader* pShader = g_pComputeShaderProgram->GetShader();
    g_SlotExportBuffer = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Compute,
        nn::gfx::ShaderInterfaceType_UnorderedAccessBuffer, "ExpBuf" );
}

void InitializeCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability( nn::gfx::QueueCapability_Compute );
    info.SetCommandBufferType( nn::gfx::CommandBufferType_Direct );
    g_CommandBuffer.Initialize( &g_Device, info );
}

void InitializeQueryBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_QueryBuffer );
    info.SetSize( sizeof( nn::gfx::TimestampBuffer ) * 2 );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        g_MemoryPoolOffset = nn::util::align_up( g_MemoryPoolOffset,
            nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_QueryBuffer.Initialize( &g_Device, info, &g_MemoryPool, g_MemoryPoolOffset, info.GetSize() );
        g_MemoryPoolOffset += info.GetSize();
    }
    else
    {
        g_QueryBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }
}

void InitializeExportBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_UnorderedAccessBuffer );
    info.SetSize( sizeof( int ) * TargetArrayLength );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        g_MemoryPoolOffset = nn::util::align_up( g_MemoryPoolOffset,
            nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_ExportBuffer.Initialize( &g_Device, info, &g_MemoryPool, g_MemoryPoolOffset, info.GetSize() );
        g_MemoryPoolOffset += info.GetSize();
    }
    else
    {
        g_ExportBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }
}

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
void* Allocate( size_t size, size_t alignment, void* )
{
    return aligned_alloc( alignment, nn::util::align_up(size , alignment));
}
void Free( void* addr, void* )
{
    free( addr );
}
void* Reallocate( void* addr, size_t newSize, void* )
{
    return realloc( addr, newSize );
}
#endif

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::Result result;

    size_t cacheSize = 0;
    result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT(result.IsSuccess());

    void* mountRomCacheBuffer = malloc(cacheSize);
    NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

    result = nn::fs::MountRom("Contents", mountRomCacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    // グラフィックスシステムのためのメモリー周りの初期化を行います。
    {
        nv::SetGraphicsAllocator( Allocate, Free, Reallocate, NULL );
        nv::InitializeGraphics( malloc( GraphicsSystemMemorySize ), GraphicsSystemMemorySize );
    }
    // グラフィックス開発者向けツールおよびデバッグレイヤのためのメモリーアロケーターを設定します。
    nv::SetGraphicsDevtoolsAllocator( Allocate, Free, Reallocate, NULL );
#endif

    g_pMemoryHeap.Reset( malloc( 1024 * 1024 * 16 ) );
    g_pMemory = g_pMemoryHeap;

    g_pPoolMemory = malloc( 1024 * 1024 * 16 );

    nn::gfx::Initialize();

    // リソースの初期化
    InitializeDevice();
    InitializeMemoryPool();
    InitializeQueue();
    InitializeFence();
    InitializeComputeShader();
    InitializeCommandBuffer();
    InitializeExportBuffer();
    InitializeQueryBuffer();
    // バッファーへ書き込み
    {
        int* pMapped = g_ExportBuffer.Map< int >();
        for( int idx = 0; idx < TargetArrayLength; ++idx )
        {
            pMapped[ idx ] = idx;
        }
        g_ExportBuffer.FlushMappedRange( 0, sizeof( int ) * TargetArrayLength );
        g_ExportBuffer.Unmap();
    }
    // コマンドの生成
    NN_LOG( "Generating commands... " );
    g_MemoryPoolOffset = nn::util::align_up( g_MemoryPoolOffset,
        nn::gfx::CommandBuffer::GetCommandMemoryAlignment( &g_Device ) );
    g_CommandBuffer.AddCommandMemory( &g_CommandBufferMemoryPool, 0, 1024 );
    g_MemoryPoolOffset += 1024;
    g_pMemory.AlignUp( nn::gfx::CommandBuffer::GetControlMemoryAlignment( &g_Device ) );
    g_CommandBuffer.AddControlMemory( g_pMemory.Get(), 1024 );
    g_pMemory.Advance( 1024 );
    g_CommandBuffer.Begin();
    {
        nn::gfx::GpuAddress gpuAddress;
        g_CommandBuffer.InvalidateMemory(
            nn::gfx::GpuAccess_UnorderedAccessBuffer | nn::gfx::GpuAccess_ShaderCode );
        nn::gfx::Shader* pShader = g_pComputeShaderProgram->GetShader();
        g_CommandBuffer.SetShader( pShader, nn::gfx::ShaderStageBit_Compute );
        g_ExportBuffer.GetGpuAddress( &gpuAddress );
        g_CommandBuffer.SetUnorderedAccessBuffer( g_SlotExportBuffer,
            nn::gfx::ShaderStage_Compute, gpuAddress, sizeof( int ) * TargetArrayLength );
        g_QueryBuffer.GetGpuAddress( &gpuAddress );
        g_CommandBuffer.WriteTimestamp( gpuAddress );
        int workGroupX;
        int workGroupY;
        int workGroupZ;
        pShader->GetWorkGroupSize( &workGroupX, &workGroupY, &workGroupZ );
        int groupCount = ( TargetArrayLength - 1 + ( workGroupX * 4 ) ) / ( workGroupX * 4 );
        g_CommandBuffer.Dispatch( groupCount, 1, 1 );
        gpuAddress.Offset( sizeof( nn::gfx::TimestampBuffer ) );
        g_CommandBuffer.WriteTimestamp( gpuAddress );
        g_CommandBuffer.FlushMemory(
            nn::gfx::GpuAccess_UnorderedAccessBuffer | nn::gfx::GpuAccess_QueryBuffer );
    }
    g_CommandBuffer.End();
    NN_LOG( "finished.\n" );

    // コマンドの実行
    NN_LOG( "Runnning compute shader... " );
    g_Queue.ExecuteCommand( &g_CommandBuffer, &g_Fence );
    g_Queue.Flush();

    nn::TimeSpan timeout = nn::TimeSpan::FromSeconds( 5 );
    nn::gfx::SyncResult syncResult = g_Fence.Sync( timeout );
    if( syncResult == nn::gfx::SyncResult_TimeoutExpired )
    {
        NN_LOG( "timeout.\n" );
    }
    else if( syncResult == nn::gfx::SyncResult_Success )
    {
        nn::gfx::TimestampBuffer* pTimestamp = g_QueryBuffer.Map< nn::gfx::TimestampBuffer >();
        g_QueryBuffer.InvalidateMappedRange( 0, sizeof( nn::gfx::TimestampBuffer ) * 2 );
        nn::TimeSpan elapsed = nn::gfx::GetDuration(
            pTimestamp[ 0 ].GetValue(), pTimestamp[ 1 ].GetValue() );
        g_QueryBuffer.Unmap();

        NN_LOG( "finished (%lld ns).\n", elapsed.GetNanoSeconds() );
    }

    // 結果の検証
    NN_LOG( "Verifying result... " );
    {
        bool succeeded = true;
        int* pMapped = g_ExportBuffer.Map< int >();
        g_ExportBuffer.InvalidateMappedRange( 0, sizeof( int ) * TargetArrayLength );
        for( int idx = 0; idx < TargetArrayLength; ++idx )
        {
            if( pMapped[ idx ] != idx * idx )
            {
                succeeded = false;
                break;
            }
        }
        g_ExportBuffer.Unmap();
        NN_LOG( "%s.\n", succeeded ? "succeeded" : "failed" );
    }

    // リソースの破棄
    g_pComputeShaderProgram->Finalize( &g_Device );
    g_pComputeShaderFile->GetShaderContainer()->Finalize( &g_Device );
    g_QueryBuffer.Finalize( &g_Device );
    g_ExportBuffer.Finalize( &g_Device );
    g_CommandBuffer.Finalize( &g_Device );
    g_Fence.Finalize( &g_Device );
    g_Queue.Finalize( &g_Device );
    g_CommandBufferMemoryPool.Finalize( &g_Device );
    g_MemoryPool.Finalize( &g_Device );
    g_Device.Finalize();

    nn::gfx::Finalize();

    free( g_pMemoryHeap.Get() );
    free( g_pPoolMemory );

    nn::fs::Unmount("Contents");
    free(mountRomCacheBuffer);
} // NOLINT

