﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{GfxSimple.cpp,PageSampleGfxSimple}
 *
 * @brief
 *  シンプルなグラフィックス処理のサンプルプログラム
 */

/**
 * @page PageSampleGfxSimple GfxSimple
 * @tableofcontents
 *
 * @brief
 *  シンプルなグラフィックス処理のサンプルプログラムの解説です。
 *
 * @section PageSampleGfxSimple_SectionBrief 概要
 *  gfx を使用して、シェーダーやテクスチャーを使ったグラフィックス処理を行うシンプルなサンプルです。
 *
 * @section PageSampleGfxSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/GfxSimple
 *  Samples/Sources/Applications/GfxSimple @endlink 以下にあります。
 *
 * @section PageSampleGfxSimple_SectionNecessaryEnvironment 必要な環境
 *  画面表示が利用可能である必要があります。
 *
 * @section PageSampleGfxSimple_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleGfxSimple_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleGfxSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleGfxSimple_SectionDetail 解説
 *  このサンプルプログラムは、シェーダーリソースとテクスチャーリソースを読み込み、
 *  単純なポリゴンをシェーダーバリエーションによってアニメーションさせながらレンダリングします。
 *  利用するリソースはビルド時にビルド前イベントにからコンバートされています。
 *  リソースのコンバートにシェーダーコンバーターとテクスチャーコンバーターを利用しています。
 *
 * このサンプルプログラムの処理の流れは以下の通りです。
 *
 * - ディスプレイを取得・レイヤを初期化
 * - デバイスを初期化
 * - メモリープールを初期化
 * - スワップチェーンを初期化
 * - キューを初期化
 * - コマンドバッファーを初期化
 * - ビューポートを初期化
 * - ラスタライザステートを初期化
 * - ブレンドステートを初期化
 * - 深度ステンシルステートを初期化
 * - 頂点ステートを初期化
 * - 頂点バッファーを初期化
 * - インデクスバッファーを初期化
 * - 定数バッファーを初期化
 * - サンプラーを初期化
 * - デスクリプタープールを初期化
 * - テクスチャーリソースを初期化
 * - シェーダーリソースを初期化
 * - ループ開始
 * - シェーダーバリエーションを選択
 * - コマンドリストを作成
 * - コマンドリストを実行
 * - ディスプレイへプレゼンテーション
 * - ループ開始に戻る
 * - 各種オブジェクトを破棄
 */

#include <cstdlib>
#include <cstring>

#include <nn/nn_Assert.h>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>

#if NN_GFX_IS_TARGET_NVN
    #include <nvn/nvn.h>
    #include <nvn/nvn_FuncPtrInline.h>
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    #include <nv/nv_MemoryManagement.h>
#endif

namespace {

const int RenderWidth = 1280;
const int RenderHeight = 720;

const float Vertices[] =
{
    0.0f, 0.5f, 0.0f, 0.0f, 0.0f,
    -0.5f, 0.0f, 0.0f, 0.0f, 1.0f,
    0.5f, 0.0f, 0.0f, 1.0f, 0.0f,
    0.0f, -0.5f, 0.0f, 1.0f, 1.0f
};

const int Indices[] =
{
    0, 1, 2, 3
};

int g_BufferDescriptorBaseIndex = 0;
int g_TextureDescriptorBaseIndex = 0;
int g_SamplerDescriptorBaseIndex = 0;

//-----------------------------------------------------------------------------
// メモリー

const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;

const int ScanBufferCount = 2;

nn::util::BytePtr g_pMemoryHeap( NULL );
nn::util::BytePtr g_pMemory( NULL );

enum MemoryPoolType
{
    MemoryPoolType_CpuUncached_GpuCached,
    MemoryPoolType_CpuInvisible_GpuCached_Compressible,

    MemoryPoolType_End
};

const size_t MemoryPoolSize[ MemoryPoolType_End ] =
{
    16 * 1024 * 1024,
    20 * 1024 * 1024
};

void* g_pPoolMemory[ MemoryPoolType_End ] = {};
ptrdiff_t g_MemoryPoolOffset[ MemoryPoolType_End ] = {};

size_t g_MaxScratchMemory = 0;
ptrdiff_t g_ScratchMemoryOffset = 0;

void* ReadResource( const char* filename )
{
    nn::Result result;
    nn::fs::FileHandle hFile;

    int64_t fileSize = 0;
    result = nn::fs::OpenFile( &hFile, filename, nn::fs::OpenMode_Read );
    NN_ASSERT( result.IsSuccess() );

    result = nn::fs::GetFileSize( &fileSize, hFile );
    NN_ASSERT( result.IsSuccess() );

    nn::util::BinaryFileHeader fileHeader;
    size_t readSize;
    result = nn::fs::ReadFile( &readSize, hFile, 0, &fileHeader, sizeof( nn::util::BinaryFileHeader ) );
    NN_ASSERT( result.IsSuccess() );
    NN_ASSERT( readSize == sizeof( nn::util::BinaryFileHeader ) );
    size_t alignment = fileHeader.GetAlignment();

    g_pMemory.AlignUp( alignment );
    void* pBuffer = g_pMemory.Get();
    result = nn::fs::ReadFile( &readSize, hFile, 0, pBuffer, static_cast< size_t >( fileSize ) );
    NN_ASSERT( result.IsSuccess() );
    NN_ASSERT( readSize == static_cast< size_t >( fileSize ) );
    g_pMemory.Advance( static_cast< ptrdiff_t >( fileSize ) );

    nn::fs::CloseFile( hFile );

    return pBuffer;
}

// レイヤを初期化
nn::vi::Display* g_pDisplay;
nn::vi::Layer* g_pLayer;
void InitializeLayer()
{
    nn::Result result = nn::vi::OpenDefaultDisplay( &g_pDisplay );
    NN_ASSERT( result.IsSuccess() );
    NN_UNUSED( result );

    result = nn::vi::CreateLayer( &g_pLayer, g_pDisplay );
    NN_ASSERT( result.IsSuccess() );

    result = nn::vi::SetLayerScalingMode( g_pLayer, nn::vi::ScalingMode_FitToLayer );
    NN_ASSERT( result.IsSuccess() );
}

// デバイスを初期化
nn::gfx::Device g_Device;
void InitializeDevice()
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetDebugMode( nn::gfx::DebugMode_Enable );
    info.SetApiVersion( nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion );
    g_Device.Initialize( info );
}

nn::gfx::MemoryPool g_MemoryPool[ MemoryPoolType_End ];
void InitializeMemoryPool()
{
    const int MemoryPoolProperty[ MemoryPoolType_End ] =
    {
        nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached,
        nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached
            | nn::gfx::MemoryPoolProperty_Compressible
    };

    nn::gfx::MemoryPool::InfoType info;
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        info.SetDefault();
        info.SetMemoryPoolProperty( MemoryPoolProperty[ idxMemoryPool ] );
        size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment( &g_Device, info );
        size_t granularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( &g_Device, info );
        g_pPoolMemory[ idxMemoryPool ] = malloc( MemoryPoolSize[ idxMemoryPool ] );
        void* pPoolMemoryAligned = nn::util::BytePtr(
            g_pPoolMemory[ idxMemoryPool ] ).AlignUp( alignment ).Get();
        size_t memoryPoolSizeAligned = nn::util::align_down( MemoryPoolSize[ idxMemoryPool ], granularity );
        info.SetPoolMemory( pPoolMemoryAligned, memoryPoolSizeAligned );
        g_MemoryPool[ idxMemoryPool ].Initialize( &g_Device, info );
        g_MemoryPoolOffset[ idxMemoryPool ] = 0;
    }
}

// スワップチェーンを初期化
nn::gfx::SwapChain g_SwapChain;
void InitializeSwapChain()
{
    nn::gfx::SwapChain::InfoType info;

    info.SetDefault();
    info.SetLayer( g_pLayer );
    info.SetWidth( RenderWidth );
    info.SetHeight( RenderHeight );
    info.SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb );
    info.SetBufferCount( ScanBufferCount );
    if( NN_STATIC_CONDITION( nn::gfx::SwapChain::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
        size_t size = g_SwapChain.CalculateScanBufferSize( &g_Device, info );
        offset = nn::util::align_up( offset, nn::gfx::SwapChain::GetScanBufferAlignment( &g_Device, info ) );
        g_SwapChain.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ], offset, size );
        offset += size;
    }
    else
    {
        g_SwapChain.Initialize( &g_Device, info, NULL, 0, 0 );
    }
}

// キューを初期化
nn::gfx::Queue g_Queue;
void InitializeQueue()
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability( nn::gfx::QueueCapability_Graphics );
    g_Queue.Initialize( &g_Device, info );
}

// コマンドバッファーを初期化
nn::gfx::CommandBuffer g_CommandBuffer[ ScanBufferCount ];
void InitializeCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability( nn::gfx::QueueCapability_Graphics );
    info.SetCommandBufferType( nn::gfx::CommandBufferType_Direct );
    for( int idxCommandBuffer = 0; idxCommandBuffer < ScanBufferCount; ++idxCommandBuffer )
    {
        g_CommandBuffer[ idxCommandBuffer ].Initialize( &g_Device, info );
    }
}

// ビューポートシザーを初期化
nn::gfx::ViewportScissorState g_ViewportScissor;
void InitializeViewport()
{
    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled( true );
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetWidth( static_cast< float >( RenderWidth ) );
        viewportInfo.SetHeight( static_cast< float >( RenderHeight ) );
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetWidth( RenderWidth );
        scissorInfo.SetHeight( RenderHeight );
    }
    info.SetViewportStateInfoArray( &viewportInfo, 1 );
    info.SetScissorStateInfoArray( &scissorInfo, 1 );
    g_ViewportScissor.Initialize( &g_Device, info );
}

// Initialize the rasterizer state
nn::gfx::RasterizerState g_RasterizerState;
static void InitializeRasterizerState()
{
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetCullMode( nn::gfx::CullMode_None );
    info.SetPrimitiveTopologyType( nn::gfx::PrimitiveTopologyType_Triangle );
    info.SetScissorEnabled( true );
    info.SetDepthClipEnabled( false );
    g_RasterizerState.Initialize( &g_Device, info );
}

// Initialize the blend state
nn::gfx::BlendState g_BlendState;
static void InitializeBlendState()
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();
    nn::gfx::BlendTargetStateInfo targetInfo;
    {
        targetInfo.SetDefault();
    };
    info.SetBlendTargetStateInfoArray( &targetInfo, 1 );
    size_t size = nn::gfx::BlendState::GetRequiredMemorySize( info );
    g_pMemory.AlignUp( nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
    g_BlendState.SetMemory( g_pMemory.Get(), size );
    g_pMemory.Advance( size );
    g_BlendState.Initialize( &g_Device, info );
}

// Initialize the depth stencil state
nn::gfx::DepthStencilState g_DepthStencilState;
static void InitializeDepthStencilState()
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    info.SetDepthTestEnabled( false );
    info.SetDepthWriteEnabled( false );
    g_DepthStencilState.Initialize( &g_Device, info );
}

// Initialize the vertex state
nn::gfx::VertexState g_VertexState;
static void InitializeVertexState()
{
    nn::gfx::VertexState::InfoType info;
    info.SetDefault();
    ptrdiff_t stride = sizeof( float ) * 5;
    nn::gfx::VertexAttributeStateInfo attribs[ 2 ];
    {
        attribs[ 0 ].SetDefault();
        attribs[ 0 ].SetBufferIndex( 0 );
        attribs[ 0 ].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
        attribs[ 0 ].SetOffset( 0 );
        attribs[ 0 ].SetShaderSlot( 0 );
    }
    {
        attribs[ 1 ].SetDefault();
        attribs[ 1 ].SetBufferIndex( 0 );
        attribs[ 1 ].SetFormat( nn::gfx::AttributeFormat_32_32_Float );
        attribs[ 1 ].SetOffset( sizeof( float ) * 3 );
        attribs[ 1 ].SetShaderSlot( 1 );
    }
    nn::gfx::VertexBufferStateInfo buffer;
    {
        buffer.SetDefault();
        buffer.SetStride( stride );
    }
    info.SetVertexAttributeStateInfoArray( attribs, 2 );
    info.SetVertexBufferStateInfoArray( &buffer, 1 );
    size_t size = nn::gfx::VertexState::GetRequiredMemorySize( info );
    g_pMemory.AlignUp( nn::gfx::VertexState::RequiredMemoryInfo_Alignment );
    g_VertexState.SetMemory( g_pMemory.Get(), size );
    g_pMemory.Advance( size );
    g_VertexState.Initialize( &g_Device, info, NULL );
}

// 頂点用のバッファーを初期化
nn::gfx::Buffer g_VertexBuffer;
void InitializeVertexBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( Vertices ) );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_VertexBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_VertexBuffer.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_VertexBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }

    void* pMapped = g_VertexBuffer.Map();
    memcpy( pMapped, Vertices, info.GetSize() );
    g_VertexBuffer.Unmap();
}

// インデクス用のバッファーを初期化
nn::gfx::Buffer g_IndexBuffer;
void InitializeIndexBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( Indices ) );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_IndexBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_IndexBuffer.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_IndexBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }

    void* pMapped = g_IndexBuffer.Map();
    memcpy( pMapped, Indices, info.GetSize() );
    g_IndexBuffer.Unmap();
}

// 定数バッファー用のバッファーを初期化
nn::gfx::Buffer g_ConstantBuffer;
size_t g_ConstantBufferSize;
void InitializeConstantBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_ConstantBuffer );
    g_ConstantBufferSize = nn::util::align_up( sizeof( float ) * 4,
        nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
    info.SetSize( g_ConstantBufferSize * ScanBufferCount );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_ConstantBuffer.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_ConstantBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }
}

// サンプラーを初期化
nn::gfx::Sampler g_Sampler;
void InitializeSampler()
{
    nn::gfx::Sampler::InfoType info;
    info.SetDefault();
    info.SetFilterMode( nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint );
    info.SetAddressU( nn::gfx::TextureAddressMode_Mirror );
    info.SetAddressV( nn::gfx::TextureAddressMode_Mirror );
    info.SetAddressW( nn::gfx::TextureAddressMode_Mirror );
    g_Sampler.Initialize( &g_Device, info );
}

nn::gfx::DescriptorPool g_BufferDescriptorPool;
void InitializeBufferDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_BufferView );
    info.SetSlotCount( g_BufferDescriptorBaseIndex + 2 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &g_Device, info ) );
    g_BufferDescriptorPool.Initialize( &g_Device, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::DescriptorPool g_SamplerDescriptorPool;
void InitializeSamplerDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_Sampler );
    info.SetSlotCount( g_SamplerDescriptorBaseIndex + 1 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &g_Device, info ) );
    g_SamplerDescriptorPool.Initialize( &g_Device, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::DescriptorPool g_TextureDescriptorPool;
void InitializeTextureDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_TextureView );
    info.SetSlotCount( g_TextureDescriptorBaseIndex + 1 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &g_Device, info ) );
    g_TextureDescriptorPool.Initialize( &g_Device, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::Fence g_DisplayFence;
void InitializeFence()
{
    nn::gfx::Fence::InfoType info;
    info.SetDefault();
    g_DisplayFence.Initialize( &g_Device, info );
}

nn::gfx::Semaphore g_DisplaySemaphore;
void InitializeSemaphore()
{
    nn::gfx::Semaphore::InfoType info;
    info.SetDefault();
    g_DisplaySemaphore.Initialize( &g_Device, info );
}

nn::gfx::ResTextureFile* g_pResTextureFile;
void InitializeResTextureFile()
{
    void *pResource = ReadResource( "Contents:/SampleTexture.bntx" );
    g_pResTextureFile = nn::gfx::ResTextureFile::ResCast( pResource );
    g_pResTextureFile->Initialize( &g_Device );
    for( int idxTexture = 0, textureCount = g_pResTextureFile->GetTextureDic()->GetCount();
        idxTexture < textureCount; ++idxTexture )
    {
        g_pResTextureFile->GetResTexture( idxTexture )->Initialize( &g_Device );
    }
}

void FinalizeResTextureFile()
{
    for( int idxTexture = 0, textureCount = g_pResTextureFile->GetTextureDic()->GetCount();
        idxTexture < textureCount; ++idxTexture )
    {
        g_pResTextureFile->GetResTexture( idxTexture )->Finalize( &g_Device );
    }
    g_pResTextureFile->Finalize( &g_Device );
}

nn::gfx::ResShaderFile* g_pResShaderFile;
void InitializeResShaderFile()
{
    void* pResource = ReadResource( "Contents:/SampleShader.bnsh" );
    g_pResShaderFile = nn::gfx::ResShaderFile::ResCast( pResource );
    nn::gfx::ResShaderContainer* pContainer = g_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL( pContainer );
    pContainer->Initialize( &g_Device );

    for( int idxVariation = 0, variationCount = pContainer->GetShaderVariationCount();
        idxVariation < variationCount; ++idxVariation )
    {
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram( idxVariation );
        nn::gfx::ShaderInitializeResult result = pProgram->Initialize( &g_Device );
        NN_ASSERT_EQUAL( result, nn::gfx::ShaderInitializeResult_Success );
        NN_UNUSED( result );
    }

#if NN_GFX_IS_TARGET_NVN
    g_MaxScratchMemory = nn::gfx::NvnGetMaxRecommendedScratchMemorySize( &g_Device, &g_pResShaderFile, 1 );
    int scratchMemoryAlignment;
    nvnDeviceGetInteger( g_Device.ToData()->pNvnDevice,
        NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_ALIGNMENT, &scratchMemoryAlignment );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
    offset = nn::util::align_up( offset, scratchMemoryAlignment );
    g_ScratchMemoryOffset = offset;
    offset += g_MaxScratchMemory;
#endif
    NN_UNUSED( g_MaxScratchMemory );
    NN_UNUSED( g_ScratchMemoryOffset );
}

void FinalizeResShaderFile()
{
    auto pContainer = g_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL( pContainer );

    for( int idxVariation = 0, variationCount = pContainer->GetShaderVariationCount();
        idxVariation < variationCount; ++idxVariation )
    {
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram( idxVariation );
        pProgram->Finalize( &g_Device );
    }

    pContainer->Finalize( &g_Device );
}

void InitializeGfxObjects()
{
    g_pMemoryHeap.Reset( malloc( 1024 * 1024 ) );
    g_pMemory = g_pMemoryHeap;

    InitializeDevice();

#if NN_GFX_IS_TARGET_NVN
    nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData( g_Device );
    nvnDeviceGetInteger( deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &g_TextureDescriptorBaseIndex );
    nvnDeviceGetInteger( deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &g_SamplerDescriptorBaseIndex );
#endif

    InitializeMemoryPool();

    InitializeSwapChain();
    InitializeQueue();

    InitializeCommandBuffer();
    InitializeViewport();

    InitializeRasterizerState();
    InitializeBlendState();
    InitializeDepthStencilState();
    InitializeVertexState();

    InitializeVertexBuffer();
    InitializeIndexBuffer();
    InitializeConstantBuffer();
    InitializeSampler();

    InitializeBufferDescriptorPool();
    InitializeSamplerDescriptorPool();
    InitializeTextureDescriptorPool();

    InitializeFence();
    InitializeSemaphore();

    InitializeResTextureFile();
    InitializeResShaderFile();

    NN_ASSERT( g_pMemoryHeap.Distance( g_pMemory.Get() ) < 1024 * 1024 );
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        NN_ASSERT( static_cast< size_t >( g_MemoryPoolOffset[ idxMemoryPool ] )
            < MemoryPoolSize[ idxMemoryPool ] );
    }
}

void MakeCommand( int frame, nn::gfx::ColorTargetView* pTarget, nn::gfx::CommandBuffer* pCommandBuffer )
{
    nn::gfx::GpuAddress gpuAddress;

    pCommandBuffer->Reset();
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::CommandBuffer::GetCommandMemoryAlignment( &g_Device ) );
    // ワンタイムのコマンドバッファーには nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuUncached のメモリープールを使うことがより適しています
    pCommandBuffer->AddCommandMemory( &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ],
        offset + ( frame % ScanBufferCount ) * 1024 * 1024, 1024 * 1024 );
    g_pMemory.AlignUp( nn::gfx::CommandBuffer::GetControlMemoryAlignment( &g_Device ) );
    pCommandBuffer->AddControlMemory( g_pMemory.Get(), 1024 );
    pCommandBuffer->Begin();
    {
        g_VertexBuffer.GetGpuAddress( &gpuAddress );
        pCommandBuffer->SetVertexBuffer( 0, gpuAddress, sizeof( float ) * 5, sizeof( Vertices ) );

        pCommandBuffer->InvalidateMemory( nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor );

        pCommandBuffer->SetDescriptorPool( &g_BufferDescriptorPool );
        pCommandBuffer->SetDescriptorPool( &g_TextureDescriptorPool );
        pCommandBuffer->SetDescriptorPool( &g_SamplerDescriptorPool );

        pCommandBuffer->ClearColor( pTarget, 0.3f, 0.1f, 0.1f, 1.0f, NULL );

#if NN_GFX_IS_TARGET_NVN
        nvnCommandBufferSetShaderScratchMemory( pCommandBuffer->ToData()->pNvnCommandBuffer,
            g_MemoryPool[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ].ToData()->pNvnMemoryPool,
            g_ScratchMemoryOffset, g_MaxScratchMemory );
#endif

        nn::gfx::ResShaderContainer* pContainer = g_pResShaderFile->GetShaderContainer();
        int variation = frame / 32 % pContainer->GetShaderVariationCount();
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram( variation );
        nn::gfx::Shader* pShader = pProgram->GetShader();
        pCommandBuffer->SetShader( pShader, nn::gfx::ShaderStageBit_All );

        pCommandBuffer->SetRenderTargets( 1, &pTarget, NULL );

        pCommandBuffer->SetViewportScissorState( &g_ViewportScissor );

        pCommandBuffer->SetRasterizerState( &g_RasterizerState );
        pCommandBuffer->SetBlendState( &g_BlendState );
        pCommandBuffer->SetDepthStencilState( &g_DepthStencilState );
        pCommandBuffer->SetVertexState( &g_VertexState );

        pCommandBuffer->InvalidateMemory( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
            | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer );

        nn::gfx::DescriptorSlot constantBufferDescriptor;
        nn::gfx::DescriptorSlot textureDescriptor;
        nn::gfx::DescriptorSlot samplerDescriptor;
        g_BufferDescriptorPool.GetDescriptorSlot( &constantBufferDescriptor,
            g_BufferDescriptorBaseIndex + ( frame % ScanBufferCount ) );
        g_TextureDescriptorPool.GetDescriptorSlot( &textureDescriptor, g_TextureDescriptorBaseIndex );
        g_SamplerDescriptorPool.GetDescriptorSlot( &samplerDescriptor, g_SamplerDescriptorBaseIndex );

        int slotMat = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
            nn::gfx::ShaderInterfaceType_ConstantBuffer, "Mat" );
        int slotTex = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
            nn::gfx::ShaderInterfaceType_Sampler, "tex" );
        if( slotMat >= 0 )
        {
            pCommandBuffer->SetConstantBuffer( slotMat,
                nn::gfx::ShaderStage_Pixel, constantBufferDescriptor );
        }
        if( slotTex >= 0 )
        {
            pCommandBuffer->SetTextureAndSampler( slotTex,
                nn::gfx::ShaderStage_Pixel, textureDescriptor, samplerDescriptor );
        }

        g_IndexBuffer.GetGpuAddress( &gpuAddress );
        pCommandBuffer->DrawIndexed( nn::gfx::PrimitiveTopology_TriangleStrip,
            nn::gfx::IndexFormat_Uint32, gpuAddress, 4, 0 );
    }
    pCommandBuffer->End();
}

void FinalizeGfxObjects()
{
    // 各オブジェクトを破棄
    FinalizeResShaderFile();
    FinalizeResTextureFile();

    g_DisplaySemaphore.Finalize( &g_Device );
    g_DisplayFence.Finalize( &g_Device );

    g_Sampler.Finalize( &g_Device );
    g_ConstantBuffer.Finalize( &g_Device );
    g_IndexBuffer.Finalize( &g_Device );
    g_VertexBuffer.Finalize( &g_Device );

    g_RasterizerState.Finalize( &g_Device );
    g_BlendState.Finalize( &g_Device );
    g_DepthStencilState.Finalize( &g_Device );
    g_VertexState.Finalize( &g_Device );

    g_BufferDescriptorPool.Finalize( &g_Device );
    g_TextureDescriptorPool.Finalize( &g_Device );
    g_SamplerDescriptorPool.Finalize( &g_Device );

    g_ViewportScissor.Finalize( &g_Device );
    for( int idxCommandBuffer = 0; idxCommandBuffer < ScanBufferCount; ++idxCommandBuffer )
    {
        g_CommandBuffer[ idxCommandBuffer ].Finalize( &g_Device );
    }
    g_SwapChain.Finalize( &g_Device );
    g_Queue.Finalize( &g_Device );
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        g_MemoryPool[ idxMemoryPool ].Finalize( &g_Device );
        free( g_pPoolMemory[ idxMemoryPool ] );
    }
    g_Device.Finalize();

    free( g_pMemoryHeap.Get() );
}

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
void* Allocate( size_t size, size_t alignment, void* )
{
    // aligned_alloc() の仕様に従い、alignment の整数倍になるよう size を調整します。
    return aligned_alloc( alignment, nn::util::align_up(size, alignment));
}
void Free( void* addr, void* )
{
    free( addr );
}
void* Reallocate( void* addr, size_t newSize, void* )
{
    return realloc( addr, newSize );
}
#endif

}

//
//  Main Function
//  メイン関数です。
//

extern "C" void nnMain()
{
    nn::Result result;

    size_t cacheSize = 0;
    result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT( result.IsSuccess() );

    void* mountRomCacheBuffer = malloc(cacheSize);
    NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

    result = nn::fs::MountRom( "Contents", mountRomCacheBuffer, cacheSize );
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    // グラフィックスシステムのためのメモリー周りの初期化を行います。
    {
        nv::SetGraphicsAllocator( Allocate, Free, Reallocate, NULL );
        nv::InitializeGraphics( malloc( GraphicsSystemMemorySize ), GraphicsSystemMemorySize );
    }
    // グラフィックス開発者向けツールおよびデバッグレイヤのためのメモリーアロケーターを設定します。
    nv::SetGraphicsDevtoolsAllocator( Allocate, Free, Reallocate, NULL );
#endif

    nn::vi::Initialize();
    InitializeLayer();

    nn::gfx::Initialize();
    InitializeGfxObjects();

    g_BufferDescriptorPool.BeginUpdate();
    {
        nn::gfx::GpuAddress gpuAddress;
        g_ConstantBuffer.GetGpuAddress( &gpuAddress );
        g_BufferDescriptorPool.SetBufferView( g_BufferDescriptorBaseIndex, gpuAddress, sizeof( float ) * 4 );
        gpuAddress.Offset( g_ConstantBufferSize );
        g_BufferDescriptorPool.SetBufferView( g_BufferDescriptorBaseIndex + 1, gpuAddress, sizeof( float ) * 4 );
    }
    g_BufferDescriptorPool.EndUpdate();

    g_TextureDescriptorPool.BeginUpdate();
    {
        g_TextureDescriptorPool.SetTextureView( g_TextureDescriptorBaseIndex,
            g_pResTextureFile->GetResTexture( 0 )->GetTextureView() );
    }
    g_TextureDescriptorPool.EndUpdate();

    g_SamplerDescriptorPool.BeginUpdate();
    {
        g_SamplerDescriptorPool.SetSampler( g_SamplerDescriptorBaseIndex, &g_Sampler );
    }
    g_SamplerDescriptorPool.EndUpdate();

    nn::gfx::ColorTargetView* pScanBufferViews[ ScanBufferCount ] = {};
    g_SwapChain.GetScanBufferViews( pScanBufferViews, ScanBufferCount );

    // 毎フレームのレンダリング
    for( int frame = 0; frame < 60 * 10; ++frame )
    {
        // スキャンバッファーの取得
        int nextScanBufferIndex = 0;
        nn::gfx::AcquireScanBufferResult acquireResult = g_SwapChain.AcquireNextScanBufferIndex(
            &nextScanBufferIndex, &g_DisplaySemaphore, &g_DisplayFence );
        NN_ASSERT( acquireResult == nn::gfx::AcquireScanBufferResult_Success );
        NN_UNUSED( acquireResult );
        nn::gfx::ColorTargetView* pTarget = pScanBufferViews[ nextScanBufferIndex ];
        nn::gfx::CommandBuffer* pCommandBuffer = &g_CommandBuffer[ nextScanBufferIndex ];

        // スキャンバッファーの取得を同期 (GPU)
        g_Queue.SyncSemaphore( &g_DisplaySemaphore );

        MakeCommand( frame, pTarget, pCommandBuffer );

        // 定数バッファーを更新
        float* pConstantBuffer = nn::util::BytePtr( g_ConstantBuffer.Map(),
            ( frame % ScanBufferCount ) * g_ConstantBufferSize ).Get< float >();
        {
            pConstantBuffer[ 0 ] = ( frame & 0xFF ) * ( 1.0f / 256.0f );
            pConstantBuffer[ 1 ] = 0.4f;
            pConstantBuffer[ 2 ] = 0.4f;
            pConstantBuffer[ 3 ] = 1.0f;
        }
        g_ConstantBuffer.Unmap();

        // コマンドの実行
        g_Queue.ExecuteCommand( pCommandBuffer, NULL );

        // 結果の表示
        g_Queue.Present( &g_SwapChain, 1 );
        g_Queue.Flush();

        // スキャンバッファーの取得を同期 (CPU)
        g_DisplayFence.Sync( nn::TimeSpan::FromSeconds( 1 ) );
    }

    g_Queue.Sync();

    FinalizeGfxObjects();
    nn::gfx::Finalize();

    nn::vi::DestroyLayer( g_pLayer );
    nn::vi::CloseDisplay( g_pDisplay );
    nn::vi::Finalize();

    nn::fs::Unmount("Contents");
    free(mountRomCacheBuffer);
}
