﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
// This is skeleton code for service process of nns::gfx0 driver.
//-----------------------------------------------------------------------------

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/sf/sf_HipcServer.h>

#include <nns/gfx0.h>

//-----------------------------------------------------------------------------
// nninitStartup() is invoked before calling nnMain().
//
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( 0 );
}




//-----------------------------------------------------------------------------

namespace {

//-----------------------------------------------------------------------------
// nn::sf::HipcSimpleAllInOneServerManager は HIPC のサーバのポートと
// セッションを一元管理する。引数は最大セッション数と最大ポート数。
//
class Gfx0ServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<30, 1>
{
private:
    // OnNeedsToAccept を override し、ポートへのアクセスの動作を決定する。
    // 詳細は OnNeedsToAccept() のリファレンスを参照。
    virtual nn::Result OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(portIndex);
        // ポートから開かれたセッションに、
        // サーバ内における DFC 実装（ CreateGfx0DriverByDfc() ）を紐付ける。
        return AcceptImpl(pPort, nns::gfx0::CreateGfx0DriverByDfc());
    }
};

// 繰返しのサーバ起動／終了が可能となるように placement new で初期化を行う。
// 繰返しの起動と終了が必要ない場合には Gfx0ServerManager は直接配置してもよい。
std::aligned_storage<sizeof(Gfx0ServerManager), NN_ALIGNOF(Gfx0ServerManager)>::type g_Gfx0ServerManagerStorage;

//-----------------------------------------------------------------------------
// サーバ実装用のサービスオブジェクトへの共有ポインタ
Gfx0ServerManager*  g_pGfx0ServerManager;



//-----------------------------------------------------------------------------
//  グラフィックドライバサーバーを初期化する
//
void InitializeGfx0DriverServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pGfx0ServerManager);

    // Gfx0ServerManager オブジェクトのコンストラクト（placement new）
    g_pGfx0ServerManager = new (&g_Gfx0ServerManagerStorage) Gfx0ServerManager;

    // サービス名の登録とポートの初期化
    // portIndex は OnNeedsToAccept の第一引数となるが、
    // 今回は使用しないため、適当に 0 を指定している。
    // sessionCountMax は、現時点では十分に大きな値を指定しておけばよい。
    auto portIndex       = 0;
    auto sessionCountMax = 30;
    g_pGfx0ServerManager->InitializePort(portIndex, sessionCountMax, nns::gfx0::Gfx0DriverServiceName);

    // サーバマネージャの開始
    // ただし、実際のサーバ動作は LoopAuto() 関数の呼び出しによって行なわれる。
    g_pGfx0ServerManager->Start();
}


//-----------------------------------------------------------------------------
//  グラフィックドライバサーバーを終了する
//
void FinalizeGfx0DriverServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pGfx0ServerManager);

    // placement new でコンストラクトされているので明示的にデストラクタを呼ぶ
    g_pGfx0ServerManager->~Gfx0ServerManager();
    g_pGfx0ServerManager = nullptr;
}


//-----------------------------------------------------------------------------
//  グラフィックドライバのサーバメッセージ処理をループで行なう
//
void LoopGfx0DriverServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pGfx0ServerManager);
    g_pGfx0ServerManager->LoopAuto();
}


//-----------------------------------------------------------------------------
//  グラフィックドライバのサーバメッセージ処理ループを停止する要求を送出する
//
void RequestStopGfx0DriverServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pGfx0ServerManager);
    g_pGfx0ServerManager->RequestStop();
}

//-----------------------------------------------------------------------------

}   // namespace


//-----------------------------------------------------------------------------

extern "C" void nnMain()
{
    NN_UNUSED( RequestStopGfx0DriverServer );

    // Start GFX0 service
    InitializeGfx0DriverServer();

    {
        NN_SDK_LOG("[gfx0] Server process is running.\n");

        // This function continues until invoking StopDevice() method.
        LoopGfx0DriverServer();

        NN_SDK_LOG("[gfx0] Server process is finished.\n");
    }

    // Stop GFX0 service
    FinalizeGfx0DriverServer();
}

//-----------------------------------------------------------------------------

