﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3ddemo_ViewerUtility.h"

#include <nn/g3d/g3d_Viewer.h>
#include <nn/g3d.h>
#include <nn/htcs.h>

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#endif

namespace nn { namespace g3d { namespace demo {

namespace {

// nn::os::ThreadStackAlignment でアラインされた値である必要があります。
static const size_t PollThreadStackSize = 8 * 1024;

// 通信スレッドのスタック領域です。
NN_OS_ALIGNAS_THREAD_STACK char g_PollThreadStack[PollThreadStackSize];

}

void DemoViewerBase::Initialize(
    ResourceHolder& resourceHolder,
    nn::g3d::viewer::ViewerCallback callback,
    nn::g3d::viewer::ViewerTextureBindCallback textureBindCallback) NN_NOEXCEPT
{
    m_pResourceHolder = &resourceHolder;

    // ビューアーライブラリーを初期化する前に必要な処理
#ifdef NN_ENABLE_HTC
    if (!nn::htcs::IsInitialized())
    {
        nn::htcs::Initialize(AllocateMemory, FreeMemory);
    }
#endif
    const nn::Result result = nn::fs::MountHostRoot();
    if (result.IsFailure())
    {
        NN_LOG("Failed to mount host root.\n");
    }

    // ビューアーライブラリーの初期化
    {
        nn::gfx::Device* pDevice = GetGfxFramework()->GetDevice();

        nn::g3d::viewer::ViewerServer::InitializeArg arg(
            pDevice,
            AllocateMemory,
            FreeMemory);
        if (callback)
        {
            arg.SetCallback(callback, this);
        }
        else
        {
            arg.SetCallback(this);
        }
        arg.SetTextureBindCallback(textureBindCallback, nullptr);
#ifdef NN_BUILD_TARGET_PLATFORM_OS_WIN
        int systemCodePage = GetACP();
        arg.SetCodePage(systemCodePage);
#else
        int utf8CodePage = 65001;
        arg.SetCodePage(utf8CodePage);
#endif
        nn::g3d::viewer::ViewerResult createResult = nn::g3d::viewer::ViewerServer::Initialize(arg);
        NN_ASSERT(
            createResult == nn::g3d::viewer::ViewerResult_Success ||
            createResult == nn::g3d::viewer::ViewerResult_ViewerDisabled);
    }

    CreatePollThread();
}

void DemoViewerBase::Initialize(ResourceHolder& resourceHolder) NN_NOEXCEPT
{
    Initialize(resourceHolder, nullptr, nullptr);
}

void DemoViewerBase::Finalize() NN_NOEXCEPT
{
    nn::g3d::viewer::ViewerServer::GetInstance().Close();

    FinalizeInternal();

    // 通信用スレッドの破棄
    StopPollThread();
    nn::os::DestroyThread( &m_PollThread );

    // ビューアーライブラリーの終了処理
    nn::g3d::viewer::ViewerServer::Finalize();
#ifdef NN_ENABLE_HTC
    nn::htcs::Finalize();
#endif
    nn::fs::UnmountHostRoot();

    m_pResourceHolder = nullptr;
}

void DemoViewerBase::Open() NN_NOEXCEPT
{
    nn::g3d::viewer::ViewerResult openConnectionResult = nn::g3d::viewer::ViewerServer::GetInstance().Open();
    NN_ASSERT(
        openConnectionResult == nn::g3d::viewer::ViewerResult_Success ||
        openConnectionResult == nn::g3d::viewer::ViewerResult_ViewerDisabled);
}

void DemoViewerBase::Close() NN_NOEXCEPT
{
    nn::g3d::viewer::ViewerServer::GetInstance().Close();

    // 3DEditor との切断が完了するまで通信処理をポーリングします
    while (nn::g3d::viewer::ViewerServer::GetInstance().IsOpened())
    {
        nn::g3d::viewer::ViewerServer::GetInstance().Poll();
    }
}

void DemoViewerBase::CreatePollThread() NN_NOEXCEPT
{
    nn::Result createThreadResult = nn::os::CreateThread(
        &m_PollThread, PollThreadEntryFunction,
        this, g_PollThreadStack, PollThreadStackSize, nn::os::LowestThreadPriority);
    NN_ASSERT(createThreadResult.IsSuccess());
}

void DemoViewerBase::StartPollThread() NN_NOEXCEPT
{
    if (m_IsPollThreadLiving)
    {
        return;
    }

    m_IsPollThreadLiving = true;
    m_IsPollThreadPausing = false;
    nn::os::StartThread(&m_PollThread);
}

void DemoViewerBase::StopPollThread() NN_NOEXCEPT
{
    if (!m_IsPollThreadLiving)
    {
        return;
    }

    m_IsPollThreadLiving = false;
    nn::os::WaitThread(&m_PollThread);
}

//! @brief 通信スレッドを一時停止します。
void DemoViewerBase::PausePollThread() NN_NOEXCEPT
{
    m_IsPollThreadPauseRequested = true;

    // 通信スレッドが一時停止するまで待ちます。
    const nn::TimeSpan PollInterval = nn::TimeSpan::FromMilliSeconds(7);
    while (!m_IsPollThreadPausing)
    {
        nn::os::SleepThread(PollInterval);
    }

    m_IsPollThreadPauseRequested = false;
}

//! @brief 一時停止した通信スレッドを再開します。
void DemoViewerBase::ResumePollThread() NN_NOEXCEPT
{
    m_IsPollThreadPausing = false;
}


void DemoViewerBase::PollThreadEntryFunction(void *pArg) NN_NOEXCEPT
{
    DemoViewerBase* pDemoViewer = reinterpret_cast<DemoViewerBase*>(pArg);
    const nn::TimeSpan PollInterval = nn::TimeSpan::FromMilliSeconds(7);
    while(pDemoViewer->m_IsPollThreadLiving)
    {
        if (!pDemoViewer->m_IsPollThreadPausing)
        {
            nn::g3d::viewer::ViewerServer::GetInstance().Poll();
        }

        if (pDemoViewer->m_IsPollThreadPauseRequested)
        {
            pDemoViewer->m_IsPollThreadPausing = true;
        }

        nn::os::SleepThread(PollInterval);
    }
}

void DemoViewerBase::SetModelLayout(const nn::g3d::viewer::ModelLayoutUpdatedArg& arg) NN_NOEXCEPT
{
    const nn::g3d::ResModel* pResModel = arg.pModelObj->GetResource();
    nn::util::Vector3f translate = arg.translate;
    for (nns::g3d::ModelAnimObj* pModelAnimObj : GetResourceHolder()->modelAnimObjs)
    {
        if (pModelAnimObj->GetModelObj()->GetResource() == pResModel)
        {
            if (pModelAnimObj->GetModelObj() == arg.pModelObj)
            {
                pModelAnimObj->SetScale(arg.scale);
                pModelAnimObj->SetRotate(arg.rotate);
                pModelAnimObj->SetTranslate(translate);
                break;
            }

            // 2個目以降のインスタンスはバウンディングボックス分ずらして配置します。
            pModelAnimObj->Calculate();
            nn::g3d::Sphere* pBoundingSphere = pModelAnimObj->GetModelObj()->GetBounding();
            nn::util::VectorSetX(&translate, nn::util::VectorGetX(translate) + pBoundingSphere->radius * 2);
        }
    }
}

nn::g3d::ResFile* CreateCopiedResFile(nn::g3d::ResFile* pSourceResFile, size_t fileSize, size_t alignment) NN_NOEXCEPT
{
    void* pFile = AllocateMemory(fileSize, alignment);
    if (pFile == nullptr)
    {
        return nullptr;
    }

    memcpy(pFile, pSourceResFile, fileSize);
    NN_ASSERT(nn::g3d::ResFile::IsValid(pFile));
    nn::g3d::ResFile* pResFile = nn::g3d::ResFile::ResCast(pFile);
    pResFile->Setup(GetGfxFramework()->GetDevice());
    return pResFile;
}

void SendMessageTo3dEditor(const char* message) NN_NOEXCEPT
{
    nn::g3d::viewer::ViewerServer::SendUserMessageArg userMessageArg(
        message,
        nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageType_Info,
        nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageDestination_Log);
    nn::g3d::viewer::ViewerResult result = nn::g3d::viewer::ViewerServer::GetInstance().QueueSendUserMessageCommand(userMessageArg);
    NN_UNUSED(result);
}

void SendErrorMessageTo3dEditor(const char* message) NN_NOEXCEPT
{
    nn::g3d::viewer::ViewerServer::SendUserMessageArg userMessageArg(
        message,
        nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageType_Error,
        nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageDestination_Log);
    nn::g3d::viewer::ViewerResult result = nn::g3d::viewer::ViewerServer::GetInstance().QueueSendUserMessageCommand(userMessageArg);
    NN_UNUSED(result);
}


}}} // namespace nn::g3d::demo
