﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nns/g3d.h>
#include <nn/g3d.h>

#include "g3ddemo_DemoUtility.h"
#include "g3ddemo_GfxUtility.h"

namespace nn { namespace g3d { namespace demo {

//--------------------------------------------------------------------------------------------------

//! @brief RenderModelObj が保持するシェーダーセレクターのスキニングオプションを初期化します。
void WriteSkinningOption(nns::g3d::RenderModelObj* pRenderModelObj, int passIndex) NN_NOEXCEPT;
void WriteSkinningOption(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT;


//--------------------------------------------------------------------------------------------------

//! @brief デモで使用するリソースを保持するためのユーティリティです。
class ResourceHolder
{
public:
    Vector<nn::g3d::ResShaderFile*> shaderFiles;
    Vector<nn::g3d::ResFile*> files;
    Vector<void*> modelAnims;
    Vector<nns::g3d::RenderModel*> renderModels;
    Vector<nns::g3d::RenderModelObj*> renderModelObjs;
    Vector<nns::g3d::ModelAnimObj*> modelAnimObjs;
    Vector<nn::g3d::ModelObj*> modelObjs;
    Vector<int*> animationIds;

    struct InitArg
    {
        InitArg()  NN_NOEXCEPT
            : maxShaders(128)
            , maxFiles(128)
            , maxAssigns(256)
            , maxModels(1024)
            , maxModelAnims(2048)
        {
        }

        int maxShaders;
        int maxFiles;
        int maxAssigns;
        int maxModels;
        int maxModelAnims;
    };

    void Initialize(const InitArg& arg = InitArg()) NN_NOEXCEPT
    {
        size_t shaderBufSize = sizeof(nn::g3d::ResShaderFile*) * arg.maxShaders;
        void* pShaderBuffer = AllocateMemory(shaderBufSize);
        NN_ASSERT_NOT_NULL(pShaderBuffer);
        shaderFiles.SetBuffer(pShaderBuffer, shaderBufSize);

        size_t fileBufSize = sizeof(nn::g3d::ResFile*) * arg.maxFiles;
        void* pFileBuffer = AllocateMemory(fileBufSize);
        NN_ASSERT_NOT_NULL(pFileBuffer);
        files.SetBuffer(pFileBuffer, fileBufSize);

        size_t modelAnimBufSize = sizeof(void*) * arg.maxModelAnims;
        void* pModelAnimBuffer = AllocateMemory(modelAnimBufSize);
        NN_ASSERT_NOT_NULL(pModelAnimBuffer);
        modelAnims.SetBuffer(pModelAnimBuffer, modelAnimBufSize);

        size_t renderModelBufSize = sizeof(nns::g3d::RenderModel*) * arg.maxModels;
        void* pRenderModelBuffer = AllocateMemory(renderModelBufSize);
        NN_ASSERT_NOT_NULL(pRenderModelBuffer);
        renderModels.SetBuffer(pRenderModelBuffer, renderModelBufSize);

        size_t renderModelObjBufSize = sizeof(nns::g3d::RenderModelObj*) * arg.maxModels;
        void* pRenderModelObjBuffer = AllocateMemory(renderModelObjBufSize);
        NN_ASSERT_NOT_NULL(pRenderModelObjBuffer);
        renderModelObjs.SetBuffer(pRenderModelObjBuffer, renderModelObjBufSize);

        size_t modelAnimObjBufSize = sizeof(nns::g3d::ModelAnimObj*) * arg.maxModels;
        void* pModelAnimObjBuffer = AllocateMemory(modelAnimObjBufSize);
        NN_ASSERT_NOT_NULL(pModelAnimObjBuffer);
        modelAnimObjs.SetBuffer(pModelAnimObjBuffer, modelAnimObjBufSize);

        size_t modelObjBufSize = sizeof(nn::g3d::ModelObj*) * arg.maxModels;
        void* pModelObjBuffer = AllocateMemory(modelObjBufSize);
        NN_ASSERT_NOT_NULL(pModelObjBuffer);
        modelObjs.SetBuffer(pModelObjBuffer, modelObjBufSize);

        size_t animIdBufSize = sizeof(int*) * arg.maxModelAnims;
        void* pAnimIdBuffer = AllocateMemory(animIdBufSize);
        NN_ASSERT_NOT_NULL(pAnimIdBuffer);
        animationIds.SetBuffer(pAnimIdBuffer, animIdBufSize);
    }

    void Shutdown() NN_NOEXCEPT
    {
        if (void* pBuffer = shaderFiles.GetBuffer())
        {
            FreeMemory(pBuffer);
        }
        if (void* pBuffer = files.GetBuffer())
        {
            FreeMemory(pBuffer);
        }
        if (void* pBuffer = modelAnims.GetBuffer())
        {
            FreeMemory(pBuffer);
        }
        if (void* pBuffer = renderModels.GetBuffer())
        {
            FreeMemory(pBuffer);
        }
        if (void* pBuffer = renderModelObjs.GetBuffer())
        {
            FreeMemory(pBuffer);
        }
        if (void* pBuffer = modelAnimObjs.GetBuffer())
        {
            FreeMemory(pBuffer);
        }
        if (void* pBuffer = modelObjs.GetBuffer())
        {
            FreeMemory(pBuffer);
        }
        if (void* pBuffer = animationIds.GetBuffer())
        {
            FreeMemory(pBuffer);
        }
    }
};

//! @brief リソースホルダにアニメーションを登録するユーティリティです。
//!
//! リソースホルダに登録されているリソースファイルを走査し、
//! 各リソースファイルに存在するすべてのアニメーションをリソースホルダに登録します。
//!
void RegistAnim(ResourceHolder* pHolder) NN_NOEXCEPT;

//! @brief リソースホルダに登録されているすべてのリソースを解放します。
void DestroyAll(nn::gfx::Device* pDevice, ResourceHolder* pHolder) NN_NOEXCEPT;

//! @brief リソースホルダに登録されている pTargetResModel を持つモデルを解放します。
void DestroyModels(nn::gfx::Device* pDevice, ResourceHolder* pHolder, nn::g3d::ResModel* pTargetResModel) NN_NOEXCEPT;

//! @brief リソースホルダに登録されている pTargetResModel を持つモデルアサインを解放します。
void DestroyModelAssigns(nn::gfx::Device* pDevice, ResourceHolder* pHolder, nn::g3d::ResModel* pTargetResModel) NN_NOEXCEPT;

//--------------------------------------------------------------------------------------------------

struct SetupMaterialsArg
{
    explicit SetupMaterialsArg(nn::g3d::ModelObj* pModelObj)  NN_NOEXCEPT
        : pModelObj(pModelObj)
        , pEnvMtx(nullptr)
        , pProjMtx(nullptr)
    {
    }

    nn::g3d::ModelObj* pModelObj;
    nn::util::FloatColumnMajor4x3* pEnvMtx;
    nn::util::FloatColumnMajor4x3* pProjMtx;
};

//! @brief マテリアルの初期化を行うユーティリティです。
void SetupMaterials(const SetupMaterialsArg& arg) NN_NOEXCEPT;

//--------------------------------------------------------------------------------------------------
//! @brief テクスチャーを初期化します。
void SetupTexture(nn::gfx::Device* pDevice, nn::gfx::ResTextureFile* pResTextureFile) NN_NOEXCEPT;
//! @brief テクスチャーを破棄します。
void CleanupTexture(nn::gfx::Device* pDevice, nn::gfx::ResTextureFile* pResTextureFile) NN_NOEXCEPT;
//! @brief リソースファイルの外部ファイルに格納されたテクスチャーを初期化します。
void SetupTexture(nn::gfx::Device* pDevice, nn::g3d::ResFile* pResFile, nn::g3d::TextureBindCallback pCallback) NN_NOEXCEPT;
//! @brief リソースファイルの外部ファイルに格納されたテクスチャーを破棄します。
void CleanupTexture(nn::gfx::Device* pDevice, nn::g3d::ResFile* pResFile) NN_NOEXCEPT;
//! @brief テクスチャーバインド時のコールバックです。
nn::g3d::TextureRef TextureBindCallback(const char* name, void* pUserData) NN_NOEXCEPT;
//! @brief ダミーテクスチャー用の TextureRef を取得します。
nn::g3d::TextureRef GetDummyTextureRef();

//--------------------------------------------------------------------------------------------------
//! @brief モデルをカメラの手前に移動します。
void MoveModelToFrontOfCamera(
    nns::g3d::ModelAnimObj* pModelAnimObj,
    const nn::util::Vector3f& cameraPosition,
    const nn::util::Vector3f& cameraAimTarget
) NN_NOEXCEPT;

//--------------------------------------------------------------------------------------------------
//! @brief バウンディングを描画します。
class BoundingRenderer
{
public:
    BoundingRenderer() NN_NOEXCEPT
        : m_pPrimitiveRenderer(nullptr)
        , m_BufferIndex(0)
    {
    }

    void Initialize(nn::gfx::Device* pDevice) NN_NOEXCEPT
    {
        // プリミティブレンダラの初期化
        InitializePrimitiveRenderer(&m_pPrimitiveRenderer);

        // ラスタライザーステートを初期化
        {
            nn::gfx::RasterizerState::InfoType info;
            info.SetDefault();
            info.SetCullMode(nn::gfx::CullMode_None);
            info.SetPrimitiveTopologyType(nn::gfx::PrimitiveTopologyType_Triangle);
            info.SetScissorEnabled(true);
            info.SetDepthClipEnabled(true);
            m_RasterizerState.Initialize(pDevice, info);
        }

        // ブレンドステートを初期化
        {
            nn::gfx::BlendTargetStateInfo targetInfo;
            targetInfo.SetDefault();
            targetInfo.SetChannelMask(nn::gfx::ChannelMask_Red | nn::gfx::ChannelMask_Green | nn::gfx::ChannelMask_Blue);
            targetInfo.SetBlendEnabled(true);
            targetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
            targetInfo.SetDestinationColorBlendFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha);
            targetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);

            nn::gfx::BlendState::InfoType info;
            info.SetDefault();
            info.SetBlendTargetStateInfoArray(&targetInfo, 1);
            size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize(info);
            m_BlendState.SetMemory(AllocateMemory(memorySize, nn::gfx::BlendState::RequiredMemoryInfo_Alignment), memorySize);
            m_BlendState.Initialize(pDevice, info);
        }
    }

    void Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT
    {
        // ブレンドステートを破棄
        void* pData = m_BlendState.GetMemory();
        m_BlendState.Finalize(pDevice);
        if (pData)
        {
            FreeMemory(pData);
        }

        // ラスタライザーステートを破棄
        m_RasterizerState.Finalize(pDevice);

        // プリミティブレンダラを破棄
        FinalizePrimitiveRenderer(&m_pPrimitiveRenderer);
    }

    void Update(nn::gfx::CommandBuffer* pCommandBuffer, nn::util::Matrix4x3fType* pViewMtx, nn::util::Matrix4x4fType* pProjMtx) NN_NOEXCEPT
    {
        // プリミティブレンダラの更新
        m_BufferIndex = 1 - m_BufferIndex;
        m_pPrimitiveRenderer->object.Update(m_BufferIndex);
        m_pPrimitiveRenderer->object.SetDefaultParameters();
        m_pPrimitiveRenderer->object.SetViewMatrix(pViewMtx);
        m_pPrimitiveRenderer->object.SetProjectionMatrix(pProjMtx);
        pCommandBuffer->SetBlendState(&m_BlendState);
        pCommandBuffer->SetRasterizerState(&m_RasterizerState);
    }

    void DrawSphere(const nn::g3d::Sphere* pSphere, nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
    {
        m_pPrimitiveRenderer->object.DrawSphere(
            pCommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface_Wired,
            nns::gfx::PrimitiveRenderer::Subdiv_Normal,
            pSphere->center,
            pSphere->radius * 2.0f
        );
    }

    void DrawSphere(nn::g3d::ModelObj* pModelObj, nn::gfx::CommandBuffer* pCommandBuffer, int lodLevel) NN_NOEXCEPT
    {
        int selectLodLevel = lodLevel >= 0 ? lodLevel : 0;
        selectLodLevel = pModelObj->GetLodCount() > selectLodLevel ? selectLodLevel : 0;
        const Sphere* pSphere = pModelObj->GetBounding(selectLodLevel);
        DrawSphere(pSphere, pCommandBuffer);
    }

    void DrawBox(nn::g3d::ModelObj* pModelObj, nn::gfx::CommandBuffer* pCommandBuffer, int shapeIndex, int lodLevel, int submeshIndex = -1) NN_NOEXCEPT
    {
        const nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(shapeIndex);
        int selectLodLevel = lodLevel >= 0 ? lodLevel : 0;
        selectLodLevel = pModelObj->GetLodCount() > selectLodLevel ? selectLodLevel : 0;

        // リジッドボディ以外には表示不要
        if (!pShapeObj->IsRigidBody())
        {
            return;
        }

        // ビジビリティーの制御を行います。
        if (!pModelObj->IsShapeVisible(shapeIndex))
        {
            return;
        }

        // BoundingBox
        const nn::g3d::Aabb* pAabb = pShapeObj->GetSubMeshBoundingArray(selectLodLevel);
        if (!pAabb)
        {
            return;
        }

        if (submeshIndex >= 0)
        {
            int drawSubmeshIndex = submeshIndex < pShapeObj->GetSubMeshCount(selectLodLevel) ? submeshIndex : pShapeObj->GetSubMeshCount(selectLodLevel) - 1;
            SetBoundingBoxColor(drawSubmeshIndex);
            nn::util::Vector3fType min = pAabb[drawSubmeshIndex].min;
            nn::util::Vector3fType max = pAabb[drawSubmeshIndex].max;
            nn::util::Vector3fType center;
            nn::util::VectorAdd(&center, min, max);
            nn::util::VectorMultiply(&center, center, 0.5f);
            nn::util::Vector3fType size;
            nn::util::VectorSubtract(&size, max, min);
            m_pPrimitiveRenderer->object.DrawCube(pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface_Wired, center, size);
        }
        else
        {
            int subMeshCount = pShapeObj->GetSubMeshCount(selectLodLevel);
            for (int subMeshIndex = 0; subMeshIndex < subMeshCount; ++subMeshIndex)
            {
                // サブメッシュ毎に色を変更
                SetBoundingBoxColor(subMeshIndex);
                nn::util::Vector3fType min = pAabb[subMeshIndex].min;
                nn::util::Vector3fType max = pAabb[subMeshIndex].max;
                nn::util::Vector3fType center;
                nn::util::VectorAdd(&center, min, max);
                nn::util::VectorMultiply(&center, center, 0.5f);
                nn::util::Vector3fType size;
                nn::util::VectorSubtract(&size, max, min);
                m_pPrimitiveRenderer->object.DrawCube(pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface_Wired, center, size);
            }
        }
    }

private:
    // サブメッシュ毎にカラーを変更する
    void SetBoundingBoxColor(int idxSubMesh) NN_NOEXCEPT
    {
        const uint8_t colors[][3] =
        {
            { 0xB4, 0xB4, 0xB4 },
            { 0xFF, 0x00, 0x00 },
            { 0xFF, 0xFF, 0x00 },
            { 0x00, 0xFF, 0x00 },
            { 0x00, 0xFF, 0xFF },
            { 0x00, 0x00, 0xFF },
            { 0xFF, 0x00, 0xFF },
            { 0xFF, 0x4B, 0x00 },
            { 0x4B, 0xFF, 0x00 },
            { 0x00, 0xFF, 0x4B },
            { 0x00, 0x4B, 0xFF },
            { 0x4B, 0x00, 0xFF },
            { 0xFF, 0x00, 0x4B },
            { 0xFF, 0x4B, 0x4B },
            { 0xFF, 0xFF, 0x4B },
            { 0x4B, 0xFF, 0x4B },
            { 0x4B, 0xFF, 0xFF },
            { 0x4B, 0x4B, 0xFF },
            { 0xFF, 0x4B, 0xFF },
            { 0xFF, 0xB4, 0xB4 },
            { 0xFF, 0xFF, 0xB4 },
            { 0xB4, 0xFF, 0xB4 },
            { 0xB4, 0xFF, 0xFF },
            { 0xB4, 0xB4, 0xFF },
            { 0xFF, 0xB4, 0xFF },
            { 0x4B, 0x4B, 0x4B },
        };

        const int colorsMax = sizeof(colors) / 3;
        int idxColor = idxSubMesh % colorsMax;

        nn::util::Color4u8 color;
        color.SetR(colors[idxColor][0]);
        color.SetG(colors[idxColor][1]);
        color.SetB(colors[idxColor][2]);
        color.SetA(255);
        m_pPrimitiveRenderer->object.SetColor(color);
    }

    nns::gfx::DebugGraphicsFramework::PrimitiveRenderer* m_pPrimitiveRenderer;
    nn::gfx::BlendState m_BlendState;
    nn::gfx::RasterizerState m_RasterizerState;
    int m_BufferIndex;
};

}}} // namespace nn::g3d::demo

