﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// ビューの種類
enum ViewType
{
    ViewType_Default,
    ViewType_Water,
    ViewType_Light0,
    ViewType_Light1,
    ViewType_Light2,
    ViewType_Light3,

    ViewType_Count
};

// アニメーションの種類
enum AnimationType
{
    AnimationType_Skeletal,
    AnimationType_Material,
    AnimationType_Shape,
    AnimationType_BoneVisibility,
    AnimationType_Count,
};

// 描画パスの種類
enum DrawPassType
{
    DrawPassType_Model,
    DrawPassType_Water,
    DrawPassType_Shadow,
    DrawPassType_Geometry,
    DrawPassType_Count
};

// フレームバッファーの種類
enum FrameBufferType
{
    FrameBufferType_Shadow,
    FrameBufferType_Water,
    FrameBufferType_Geometry,
    FrameBufferType_Light,
    FrameBufferType_Count
};

// モデルに関連付いていないシェーディングモデルの種類
enum ShadingModelType
{
    ShadingModelType_Shadow,
    ShadingModelType_Geometry,
    ShadingModelType_TextureAverage,
    ShadingModelType_Light,
    ShadingModelType_Count
};

// メニューの種類
enum MenuType
{
    MenuType_Edit,
    MenuType_Debug1,
    MenuType_Debug2,
    MenuType_Debug3,

    MenuType_Count
};

// メニューのオプションを定義する構造体
struct MenuFlag
{
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<0> LightAnimation;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<1> WaterRendering;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<2> GeometryRendering;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<3> LightRendering;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<4> ShadowRendering;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<5> ModelAnimation;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<6> Wireframe;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<7> Ice;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<8> LodLevel;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<9> BoundingSphere;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<10> BoundingBox;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<11> SingleShape;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<12> TextureCompute;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<13> SingleSubmesh;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<14> MipLevel;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<15> FreezeViewVolume;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<16> DrawViewVolume;
    typedef nn::util::BitFlagSet<32, MenuFlag>::Flag<17> DrawDebugFrameBuffer;
};
typedef nn::util::BitFlagSet<32, MenuFlag> MenuFlagSet;

namespace g3ddemo = nn::g3d::demo;

// Town デモで利用するビューアークラス
class TownViewer : public g3ddemo::DemoViewerBase
{
    NN_DISALLOW_COPY(TownViewer);
public:
    TownViewer() NN_NOEXCEPT
        : m_LoadedResFileCount(0)
        , m_UnloadedResFileCount(0)
    {
    }

    void Update() NN_NOEXCEPT;

    nn::g3d::ResShaderArchive* FindShaderArchive(const char* pShaderName) NN_NOEXCEPT;

    void AttachShader(const char* pShaderName) NN_NOEXCEPT;

    // ビューアーのテクスチャーバインドコールバックはカスタマイズすることができます。
    static nn::g3d::TextureRef ViewerTextureBindCallback(
        const char* name, const nn::gfx::ResTextureFile* pResTextureFile, void* pUserData) NN_NOEXCEPT;

    // 関数コールバックのサンプルです。
    static void TownViewerCallback(void* pOutArg, const void* pInArg, nn::g3d::viewer::CallbackType type, void* pUserData) NN_NOEXCEPT;

private:
    virtual void FinalizeInternal() NN_NOEXCEPT NN_OVERRIDE;
    void PushLoadedModelFile(nn::g3d::ResFile* pResFile) NN_NOEXCEPT;
    void PushUnloadedModelFile(nn::g3d::ResFile* pResFile) NN_NOEXCEPT;
    void CreateModelsFromLoadedResFiles() NN_NOEXCEPT;
    void SetMaterialShaderToUserPtr(nn::g3d::ResFile* pOutResFile) NN_NOEXCEPT;
    void DestroyUnloadedResFiles() NN_NOEXCEPT;
    void SetShaderMemory() NN_NOEXCEPT;
    void RemoveModelsFromResourceHolder(nn::g3d::ResFile* pResFile) NN_NOEXCEPT;
    void UpdateRenderInfo(const nn::g3d::ModelObj* pModelObj, int materialIndex, const char* pName) NN_NOEXCEPT;

private:
    static const int LoadModelFileCountMax = 16;
    nn::g3d::ResFile* m_LoadedResFiles[LoadModelFileCountMax];
    int m_LoadedResFileCount;

    nn::g3d::ResFile* m_UnloadedResFiles[LoadModelFileCountMax];
    int m_UnloadedResFileCount;
};

// ライトの描画を管理するクラス
class TownPointLight
{
    NN_DISALLOW_COPY(TownPointLight);
public:
    TownPointLight() NN_NOEXCEPT
        : m_pRenderModel(nullptr)
        , m_PointLightRadius(0.0f)
        , m_PointLightRadiusDelta(pointLightMaxRadius * 0.1f)
        , m_IsInitialized(false)
    {
    }

    static const int maxLightCount = 128;
    static float pointLightMaxRadius;

    // 初期化
    void Initialize(nn::gfx::Device* pDevice, nn::g3d::ResModel* pResModel, nn::g3d::ResShadingModel* pResShadingModel) NN_NOEXCEPT;

    // 破棄処理
    void Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT;

    // ポイントライトの追加
    void AddLight(nn::gfx::Device* pDevice, const nn::util::Float3& color, const nn::util::Vector3fType& position, const nn::util::Vector3fType& scale) NN_NOEXCEPT;

    // ポイントライトの追加
    void AddLight(nn::gfx::Device* pDevice, const nn::util::Float3& color, const nn::util::Vector3fType& position) NN_NOEXCEPT
    {
        AddLight(pDevice, color, position, NN_UTIL_VECTOR_3F_INITIALIZER(pointLightMaxRadius, pointLightMaxRadius, pointLightMaxRadius));
    }

    // 更新処理
    void Calculate(nn::gfx::Device* pDevice, int uniformBufferIndex, bool enableAnimation) NN_NOEXCEPT;

    // 描画処理
    void Draw(nn::gfx::CommandBuffer* pCommandBuffer, int uniformBufferIndex) const NN_NOEXCEPT;

    int GetRenderModelObjCount() const NN_NOEXCEPT
    {
        return m_RenderModelObjs.GetCount();
    }

    const nns::g3d::RenderModelObj* GetRenderModelObj(int objIndex) const NN_NOEXCEPT
    {
        return (objIndex >= 0 && objIndex < m_RenderModelObjs.GetCount()) ? m_RenderModelObjs[objIndex] : nullptr;
    }

    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

private:
    nns::g3d::RenderModel*                      m_pRenderModel;
    g3ddemo::Vector<nns::g3d::ModelAnimObj*>    m_ModelAnimObjs;
    g3ddemo::Vector<nns::g3d::RenderModelObj*>  m_RenderModelObjs;

    float   m_PointLightRadius;
    float   m_PointLightRadiusDelta;
    bool    m_IsInitialized;
};

// Town デモで利用するシェーダーで用いる共通のユニフォームバッファーを管理するクラス
class TownContext
{
    NN_DISALLOW_COPY(TownContext);
public:
    static const int bufferCount = 2;
    static const int cascadeShadowCount = 4;

    struct Context
    {
        nn::util::FloatColumnMajor4x4 lightViewProj[cascadeShadowCount];
        nn::util::Float4 cascadeDepth;
        nn::util::Float2 tanFov;
        nn::util::Float2 nearFar;
        float diffIntensity;
        nn::util::Float3 padding;
    };

    TownContext() NN_NOEXCEPT
    {
        m_MemoryPoolOffset[0] = -1;
        m_MemoryPoolOffset[1] = -1;
    }

    void Initialize(nn::gfx::Device* pDevice) NN_NOEXCEPT
    {
        for (int bufferIndex = 0; bufferIndex < bufferCount; ++bufferIndex)
        {
            nn::gfx::Buffer::InfoType bufferInfo;
            bufferInfo.SetDefault();
            bufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
            bufferInfo.SetSize(sizeof(Context));

            if (NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
            {
                m_MemoryPoolOffset[bufferIndex] = g3ddemo::GetGfxFramework()->AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
                    bufferInfo.GetSize(),
                    nn::gfx::Buffer::GetBufferAlignment(pDevice, bufferInfo));
                m_ContextBlock[bufferIndex].Initialize(pDevice, bufferInfo,
                    g3ddemo::GetGfxFramework()->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer),
                    m_MemoryPoolOffset[bufferIndex], bufferInfo.GetSize());
            }
            else
            {
                m_ContextBlock[bufferIndex].Initialize(pDevice, bufferInfo, nullptr, 0, 0);
            }
            m_Context[bufferIndex].diffIntensity = 1.0f;
        }
    }

    void Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT
    {
        for (int bufferIndex = 0; bufferIndex < bufferCount; ++bufferIndex)
        {
            // メモリープールから確保したオフセットを解放
            if (m_MemoryPoolOffset[bufferIndex] != -1)
            {
                g3ddemo::GetGfxFramework()->FreePoolMemory(
                    nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, m_MemoryPoolOffset[bufferIndex]);
                m_MemoryPoolOffset[bufferIndex] = -1;
            }

            m_ContextBlock[bufferIndex].Finalize(pDevice);
        }
    }

    void Calculate(int bufferIndex) NN_NOEXCEPT
    {
        Context* pContext = m_ContextBlock[bufferIndex].Map<Context>();
        memcpy(pContext, &m_Context[bufferIndex], sizeof(Context));
        m_ContextBlock[bufferIndex].FlushMappedRange(0, sizeof(Context));
        m_ContextBlock[bufferIndex].Unmap();
    }

    const nn::gfx::Buffer* GetContextBlock(int bufferIndex) const NN_NOEXCEPT
    {
        return &m_ContextBlock[bufferIndex];
    }

    void SetLightProjection(int bufferIndex, int index, const nn::util::Matrix4x4fType& matrix) NN_NOEXCEPT
    {
        NN_ASSERT(0 <= index && index < cascadeShadowCount);
        MatrixStore(&m_Context[bufferIndex].lightViewProj[index], matrix);
    }

    void SetCascadeDepth(int bufferIndex, const float* pCascadeDepth) NN_NOEXCEPT
    {
        memcpy(&m_Context[bufferIndex].cascadeDepth, pCascadeDepth, sizeof(nn::util::Float4));
    }

    void SetNearFar(int bufferIndex, float nearValue, float farValue) NN_NOEXCEPT
    {
        m_Context[bufferIndex].nearFar.x = nearValue;
        m_Context[bufferIndex].nearFar.y = farValue;
    }

    void SetTangentFov(int bufferIndex, float x, float y) NN_NOEXCEPT
    {
        m_Context[bufferIndex].tanFov.x = x;
        m_Context[bufferIndex].tanFov.y = y;
    }

    float GetDiffIntensity(int bufferIndex) const NN_NOEXCEPT
    {
        return m_Context[bufferIndex].diffIntensity;
    }

    void SetDiffIntensity(int bufferIndex, float diffIntensity) NN_NOEXCEPT
    {
        m_Context[bufferIndex].diffIntensity = diffIntensity;
    }

private:
    nn::gfx::Buffer m_ContextBlock[bufferCount];
    ptrdiff_t       m_MemoryPoolOffset[bufferCount];
    Context         m_Context[bufferCount];
};

// Lod レベルを算出する関数オブジェクト構造体
struct CalculateLod : public nn::g3d::ICalculateLodLevelFunctor
{
public:
    CalculateLod() NN_NOEXCEPT
    {
    }

    CalculateLod(const nn::util::Matrix4x3fType* pViewMtx, float cotFovY, const float* pThreasholds, int ThreasholdCount) NN_NOEXCEPT
        : m_pViewMtx(pViewMtx), m_CotFovY(cotFovY), m_pThresholds(pThreasholds), m_ThreasholdCount(ThreasholdCount)
    {
    }

    virtual int operator()(const nn::g3d::Aabb& bounding, const nn::g3d::ShapeObj&) NN_NOEXCEPT
    {
        nn::util::Vector3fType center;
        VectorAdd(&center, bounding.min, bounding.max);
        VectorMultiply(&center, center, 0.5f);
        nn::util::FloatRowMajor4x3 viewMtx34;
        MatrixStore(&viewMtx34, *m_pViewMtx);
        float viewZ = viewMtx34.m02 * VectorGetX(center) + viewMtx34.m12 * VectorGetY(center)
            + viewMtx34.m22 * VectorGetZ(center) + viewMtx34.m32;
        nn::util::Vector3fType diff;
        VectorSubtract(&diff, center, bounding.min);
        float distance = VectorLength(diff);
        float ratio = (viewZ != 0.0f) ? std::abs(distance * m_CotFovY / viewZ) : 1.0f;
        int lodLevel = 0;
        for (int levelIndex = 1; levelIndex < m_ThreasholdCount; ++levelIndex)
        {
            if (ratio < m_pThresholds[levelIndex])
            {
                lodLevel = levelIndex;
            }
        }
        return lodLevel;
    }

private:
    const nn::util::Matrix4x3fType* m_pViewMtx;
    float m_CotFovY;
    const float* m_pThresholds;
    int m_ThreasholdCount;
};

class ViewVolumeRenderer
{
    NN_DISALLOW_COPY(ViewVolumeRenderer);
public:

    ViewVolumeRenderer() NN_NOEXCEPT
        : m_pRenderer(nullptr)
        , m_MemoryOffset(0)
    {
    }

    void Initialize() NN_NOEXCEPT;

    void Finalize() NN_NOEXCEPT;

    void Update(int bufferIndex) NN_NOEXCEPT
    {
        m_pRenderer->object.Update(bufferIndex);
    }

    void SetViewMatrix(const nn::util::Matrix4x3fType& viewMatrix) NN_NOEXCEPT
    {
        m_pRenderer->object.SetViewMatrix(&viewMatrix);
    }

    void SetProjectionMatrix(const nn::util::Matrix4x4fType& projectionMatrix) NN_NOEXCEPT
    {
        m_pRenderer->object.SetProjectionMatrix(&projectionMatrix);
    }

    void Draw(nn::gfx::CommandBuffer* pCommandBuffer, const nn::util::Matrix4x4fType& invViewProjectionMatrix, const nn::util::Color4u8Type& color) NN_NOEXCEPT;

private:
    nns::gfx::DebugGraphicsFramework::PrimitiveRenderer* m_pRenderer;
    ptrdiff_t m_MemoryOffset;
};

// RenderModel に RenderInfo を元にして、ブレンドステートとラスタライザーステートを設定します。
void SetupRenderState(nns::g3d::RenderModel* pRenderModel) NN_NOEXCEPT;

// RenderModelObj の ラスタライザーステートを変更します。
void SetWireframe(nns::g3d::RenderModelObj* pRenderModelObj, int fillMode, int materialIndex) NN_NOEXCEPT;

// メニューを初期化します。
void InitializeMenu() NN_NOEXCEPT;

// メニューモードを取得します。
int GetMenuMode() NN_NOEXCEPT;

// メニューが有効か取得します。
bool IsMenuEnabled() NN_NOEXCEPT;

// メニューの描画設定を構築します。
void CalculateMenuInfo(g3ddemo::ScreenInfo& screenInfo, nn::util::Float4& texAverage) NN_NOEXCEPT;

// キー入力に応じてメニューのパラメーターを更新します。
void UpdateMenuParameter(const g3ddemo::Pad& pad) NN_NOEXCEPT;

// 強制的に設定する LOD レベルを取得します。
int GetForceLodLevel() NN_NOEXCEPT;

// RenderModelObj にユニフォームブロックや LOD に関する設定を行います。
void SetupRenderModelObj(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT;

// RenderModel を初期化します。
void CreateRenderModel(g3ddemo::ResourceHolder* pHolder, nn::gfx::Device* pDevice, nn::g3d::ResFile* pResFile) NN_NOEXCEPT;

// モデルインスタンス生成後呼び出されるコールバック関数の型です。
typedef void(*CreateModelInstanceCallback)(nn::g3d::ModelObj* pModelObj);

// ModelAnimObj を初期化します。
void CreateModelAnimObj(g3ddemo::ResourceHolder* pHolder, nn::gfx::Device* pDevice, nn::g3d::ResFile* pResFile, CreateModelInstanceCallback pCallback) NN_NOEXCEPT;

// RenderModelObj を初期化します。
void CreateRenderModelObj(g3ddemo::ResourceHolder* pHolder, nns::g3d::ModelAnimObj* pModelAnimObj, nns::g3d::RenderModel* pRenderModel) NN_NOEXCEPT;

// ユニフォームブロックの設定を行います。
void BindUniformBlockAndSampler(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT;

// 環境モデルのセットアップを行います。
void SetupEnvModel(nn::gfx::Device* pDevice, nn::g3d::MaterialObj* pEnvMaterial) NN_NOEXCEPT;

// 環境モデルが保持するユニフォームブロックとサンプラーを割り当てます。
void BindEnvUniformBlockAndSampler(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT;

