﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SkeletalAnimation.h"

namespace nn { namespace g3d { namespace demo { namespace skeletalanim {

extern nns::g3d::RenderModelObj*    g_pRenderModelObj[ModelCount];
extern nns::g3d::RenderView         g_RenderView;

enum MenuItem
{
    MenuItem_AnimationPreset,
    MenuItem_RetargetingEnabled,
    MenuItem_ViewSkeletonEnabled,
    MenuItem_AnimStep,

    MenuItemCount,
};

g3ddemo::ScreenInfo  g_ScreenInfo;

int g_SelectedItemIndex;
int g_SelectedAnimationPresetIndex;
bool g_RetargetingEnabled;
bool g_ViewSkeletonEnabled;
float g_AnimStep;

void InitializeMenu(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    g_SelectedItemIndex = 0;
    g_SelectedAnimationPresetIndex = 0;
    g_RetargetingEnabled = true;
    g_ViewSkeletonEnabled = false;
    g_AnimStep = 1.f;
    g_ScreenInfo.Initialize(pDevice);
    g_ScreenInfo.SetWindowPosition(16, 16);
    g_ScreenInfo.SetFontSize(20);
}

void FinalizeMenu(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    g_ScreenInfo.Cleanup(pDevice);
}

MenuResult CalculateMenu(const g3ddemo::Pad &pad) NN_NOEXCEPT
{
    MenuResult result = MenuResult_None;

    // 上下項目の選択
    if (pad.IsTriggered(g3ddemo::Pad::BUTTON_DOWN))
    {
        g_SelectedItemIndex = (g_SelectedItemIndex + 1) % MenuItemCount;
    }
    if (pad.IsTriggered(g3ddemo::Pad::BUTTON_UP))
    {
        g_SelectedItemIndex = (g_SelectedItemIndex + MenuItemCount - 1) % MenuItemCount;
    }

    // 項目の内容を切り替える
    switch (g_SelectedItemIndex)
    {
    case MenuItem_AnimationPreset:
        {
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
            {
                g_SelectedAnimationPresetIndex = (g_SelectedAnimationPresetIndex + 1) % g_AnimationPresetCount;
                result = MenuResult_ChangeSelectedAnim;
            }
            else if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
            {
                g_SelectedAnimationPresetIndex = (g_SelectedAnimationPresetIndex + g_AnimationPresetCount - 1) % g_AnimationPresetCount;
                result = MenuResult_ChangeSelectedAnim;
            }
        }
        break;
    case MenuItem_RetargetingEnabled:
        {
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT) || pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
            {
                g_RetargetingEnabled = !g_RetargetingEnabled;
                result = MenuResult_ChangeSelectedAnim;
            }
        }
        break;
    case MenuItem_ViewSkeletonEnabled:
        {
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT) || pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
            {
                g_ViewSkeletonEnabled = !g_ViewSkeletonEnabled;
            }
        }
        break;
    case MenuItem_AnimStep:
        {
            const float animStepOffset = 0.05f;
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
            {
                g_AnimStep += animStepOffset;
                result = MenuResult_ChangeAnimStep;
            }
            else if(pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
            {
                g_AnimStep = std::max(0.f, g_AnimStep - animStepOffset);
                result = MenuResult_ChangeAnimStep;
            }
        }
        break;
    default:
        break;
    }
    return result;
}

void DrawMenu(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    const char* allowString = ">";
    g_ScreenInfo.StartPrint();
    g_ScreenInfo.Print("[SkeletalAnimation]\n");
    g_ScreenInfo.Print("Up, Down : Select Menu\n");
    g_ScreenInfo.Print("Left, Right : Select Value\n\n");

    const AnimationPreset* pAnimationPreset = GetSelectedAnimationPreset();
    g_ScreenInfo.Print("%s : Animation Preset (%d/%d)\n", (g_SelectedItemIndex == MenuItem_AnimationPreset) ? allowString : "  ", g_SelectedAnimationPresetIndex + 1, g_AnimationPresetCount);
    if (pAnimationPreset->weight[AnimIndex_1] > 0.f)
    {
        g_ScreenInfo.Print("    [%s(%d%%)+%s(%d%%)]\n",
            pAnimationPreset->name[AnimIndex_0], static_cast<int>(pAnimationPreset->weight[AnimIndex_0] * 100),
            pAnimationPreset->name[AnimIndex_1], static_cast<int>(pAnimationPreset->weight[AnimIndex_1] * 100)
        );
    }
    else
    {
        g_ScreenInfo.Print("    [%s]\n", pAnimationPreset->name[AnimIndex_0]);
    }

    g_ScreenInfo.Print("%s : Retargeting    [%s]\n", g_SelectedItemIndex == MenuItem_RetargetingEnabled ? allowString : "  ", g_RetargetingEnabled == true ? "true" : "false");
    g_ScreenInfo.Print("%s : View Skeleton [%s]\n", g_SelectedItemIndex == MenuItem_ViewSkeletonEnabled ? allowString : "  ", g_ViewSkeletonEnabled == true ? "true" : "false");
    g_ScreenInfo.Print("%s : Anim Step      [%.2f]\n", g_SelectedItemIndex == MenuItem_AnimStep ? allowString : "  ", g_AnimStep);
    g_ScreenInfo.EndPrint();
    g_ScreenInfo.AdjustWindowSize();
    g_ScreenInfo.Draw(pCommandBuffer);
}

bool IsRetargetingEnabled() NN_NOEXCEPT
{
    return g_RetargetingEnabled;
}

bool IsViewSkeletonEnabled() NN_NOEXCEPT
{
    return g_ViewSkeletonEnabled;
}

const AnimationPreset *GetSelectedAnimationPreset() NN_NOEXCEPT
{
    return &g_AnimationPresetList[g_SelectedAnimationPresetIndex];
}

float GetAnimStep() NN_NOEXCEPT
{
    return g_AnimStep;
}

void DrawSkeleton(nn::gfx::CommandBuffer* pCommandBuffer, nn::g3d::SkeletonObj* pSkeletonObj) NN_NOEXCEPT
{
    const nn::util::Matrix4x3fType* worldMtxArray = pSkeletonObj->GetWorldMtxArray();
    nns::gfx::PrimitiveRenderer::Renderer* pRenderer = g3ddemo::GetGfxFramework()->GetPrimitiveRenderer();
    pRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType_DepthNoWriteTest);
    nn::util::Color4u8 lightGreen(67, 255, 163, 255);
    pRenderer->SetColor(lightGreen);
    pRenderer->SetViewMatrix(&g_RenderView.GetViewMtx(0));
    pRenderer->SetProjectionMatrix(&g_RenderView.GetProjectionMtx(0));

    for (int boneIndex = 0, boneCount = pSkeletonObj->GetBoneCount(); boneIndex < boneCount; ++boneIndex)
    {
        nn::util::Matrix4x3fType boneWorldMtx = worldMtxArray[boneIndex];
        float radius = 2.0f;

        nn::util::Vector3fType position;
        nn::util::MatrixGetAxisW(&position, boneWorldMtx);
        pRenderer->SetModelMatrix(&nn::util::Matrix4x3f::Identity());
        pRenderer->DrawSphere(pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface_Wired, nns::gfx::PrimitiveRenderer::Subdiv_Normal, position, radius * 2.0f);

        // 3軸の描画
        nn::util::Vector3fType axisX, axisY, axisZ;
        {
            nn::util::MatrixGetAxisX(&axisX, boneWorldMtx);
            nn::util::MatrixGetAxisY(&axisY, boneWorldMtx);
            nn::util::MatrixGetAxisZ(&axisZ, boneWorldMtx);

            nn::util::Vector3f rot;
            pRenderer->SetLineWidth(1.0f);

            nn::util::VectorMultiply(&rot, axisX, radius * 2.0f);
            nn::util::VectorAdd(&rot, position, rot);
            pRenderer->SetColor(nn::util::Color4u8::Red());
            pRenderer->DrawLine(pCommandBuffer, position, rot);

            nn::util::VectorMultiply(&rot, axisY, radius * 2.0f);
            nn::util::VectorAdd(&rot, position, rot);
            pRenderer->SetColor(nn::util::Color4u8::Green());
            pRenderer->DrawLine(pCommandBuffer, position, rot);

            nn::util::VectorMultiply(&rot, axisZ, radius * 2.0f);
            nn::util::VectorAdd(&rot, position, rot);
            pRenderer->SetColor(nn::util::Color4u8::Blue());
            pRenderer->DrawLine(pCommandBuffer, position, rot);

            pRenderer->SetLineWidth(1.0f);
        }

        // 骨の描画
        pRenderer->SetColor(lightGreen);
        radius = 1.0f;
        {
            int parentIndex = pSkeletonObj->GetResBone(boneIndex)->GetParentIndex();
            if (parentIndex == nn::g3d::ResBone::InvalidBoneIndex)
            {
                continue;
            }

            nn::util::Matrix4x3fType parentBoneMtx = worldMtxArray[parentIndex];
            nn::util::Vector3fType parentPosition;
            nn::util::MatrixGetAxisW(&parentPosition, parentBoneMtx);
            nn::util::Vector3fType diff;
            nn::util::VectorSubtract(&diff, position, parentPosition);
            float height = nn::util::VectorLength(diff);

            nn::util::VectorNormalize(&axisY, diff);
            nn::util::VectorLoad(&axisX, NN_UTIL_FLOAT_3_INITIALIZER(1.0f, .0f, .001f));
            nn::util::VectorCross(&axisZ, axisX, axisY);
            nn::util::VectorNormalize(&axisZ, axisZ);
            nn::util::VectorCross(&axisX, axisY, axisZ);
            nn::util::VectorNormalize(&axisX, axisX);

            nn::util::Matrix4x3fType rotMtx;
            nn::util::MatrixIdentity(&rotMtx);
            nn::util::MatrixSetAxisX(&rotMtx, axisX);
            nn::util::MatrixSetAxisY(&rotMtx, axisY);
            nn::util::MatrixSetAxisZ(&rotMtx, axisZ);

            nn::util::Vector3fType offset;
            nn::util::VectorMultiply(&offset, axisY, radius);
            nn::util::VectorAdd(&position, parentPosition, offset);
            nn::util::MatrixSetAxisW(&rotMtx, position);

            pRenderer->SetModelMatrix(&rotMtx);
            pRenderer->DrawCone(pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface_Wired, NN_UTIL_VECTOR_3F_INITIALIZER(.0f, .0f, .0f), radius, height - 2.0f * radius);
        }
    }
}

}}}} // namespace nn::g3d::demo::skeletalanim
