﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <common.glsl>
#include <skeleton_block.glsl>
#include <shape_block.glsl>
#include <view_block.glsl>

#define SKINNING            ( -1 )       // @@ id="skinning" choice="[-1, 8]" type="dynamic"

//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_VERTEX_SHADER

layout(location = 0)  in vec4   i_Position;     // @@ id="_p0"   hint="position0"
layout(location = 1)  in vec3   i_Normal;       // @@ id="_n0"   hint="normal0"
layout(location = 2)  in ivec4  i_Index0;       // @@ id="_i0"   hint="blendindex0"
layout(location = 3)  in ivec4  i_Index1;       // @@ id="_i1"   hint="blendindex1"
layout(location = 4)  in vec4   i_Weight0;      // @@ id="_w0"   hint="blendweight0"
layout(location = 5)  in vec4   i_Weight1;      // @@ id="_w1"   hint="blendweight1"

#endif // NN_G3D_VERTEX_SHADER

#if defined(NN_G3D_GEOMETRY_SHADER) || defined(NN_G3D_FRAGMENT_SHADER)

NN_G3D_RASTER
{
    vec3  geometry; // varying を増やすと必要なメモリサイズが増大するので出来るだけパックする。
} nn_g3d_raster;

#endif

#ifdef NN_G3D_FRAGMENT_SHADER

layout(location = 0)  out vec4 o_Color;

#endif // NN_G3D_FRAGMENT_SHADER

//--------------------------------------------------------------------------------------------------

layout(std140) uniform Outline     // @@ id="outline"
{
    vec3  lineColor;               // @@ id="line_color" default="0.7 0.7 0.9" item="color"
    float halfWidth;               // @@ id="line_half_width" default="0.005"
    float fillScale;               // @@ id="fill_Scale" default="0.1"
};

uniform sampler2D renderTargetDepth; // @@ id="_d0" label="デプステクスチャ"

#ifdef OUTLINE_AS_MATERIAL
// @@ block_id="outline" type="material"
/* @@
    uniform_id="line_color"          order="1"   label="ラインカラー"
    uniform_id="line_half_width"     order="2"   label="エッジの幅の半分"
    uniform_id="fill_Scale"          order="3"   label="隙間を埋めるスケール"
*/
#endif

//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_VERTEX_SHADER

void main()
{
    vec4 pos_w = vec4(0, 0, 0, 1);

    int skinning = SKINNING != -1 ? SKINNING : vtxSkinCount;
    if (skinning == 0)
    {
        // rigid body
        pos_w.xyz = NN_G3D_TRANSFORM_POS(shapeMtx, i_Position);
    }
    else if (skinning == 1)
    {
        // rigid skinning
        int offset = i_Index0.x * 3;
        pos_w.xyz = NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
    }
    else if (skinning <= 4)
    {
        // smooth skinning [1, 4]
        for (int i = 0; i < skinning; ++i)
        {
            int offset = i_Index0[i] * 3;
            float weight = i_Weight0[i];
            pos_w.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
        }
    }
    else if (skinning <= 8)
    {
        // smooth skinning [5, 8]
        for (int i = 0; i < 4; ++i)
        {
            int offset = i_Index0[i] * 3;
            float weight = i_Weight0[i];
            pos_w.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
        }
        for (int i = 4; i < skinning; ++i)
        {
            int offset = i_Index1[i - 4] * 3;
            float weight = i_Weight1[i - 4];
            pos_w.xyz += weight * NN_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
        }
    }

    vec4 pos_v = vec4(NN_G3D_TRANSFORM_POS(cameraMtx, pos_w), 1);

    gl_Position = NN_G3D_PROJECT(projMtx, pos_v);
}

#endif // NN_G3D_VERTEX_SHADER

#ifdef NN_G3D_GEOMETRY_SHADER

layout(triangles_adjacency)               in;
layout(triangle_strip, max_vertices = 4) out;

bool IsFrontFace(vec2 a, vec2 b, vec2 c)
{
    // 外積の z 方向の和が 0 より大きい場合に手前を向いている
    return (c.y * a.x - c.x * a.y) + (c.x * b.y - c.y * b.x) + (b.x * a.y - b.y * a.x) > 0.0;
}

void EmitEdge(vec4 p0, vec4 p1)
{
    vec2 e = fillScale * (p1.xy - p0.xy);
    // 輪郭線をある程度以上伸ばすとモデルに近づいたときに、不要な輪郭線が見えるので上限を設定。上限を超えていたら、伸ばさない。
    if (length(e) > 1.0)
    {
        e = vec2(0, 0);
    }
    vec2 v = normalize(e);
    vec2 n = vec2(v.y, -v.x) * halfWidth;

    OUT.geometry.xy = vec2(p0.x, NN_G3D_PROJECTION_DIR(-p0.y)) * 0.5 + 0.5;
    gl_Position = vec4((p0.xy - n - e) * p0.w, p0.z, p0.w);
    OUT.geometry.z = halfWidth;
    EmitVertex();

    gl_Position = vec4((p0.xy + n - e) * p0.w, p0.z, p0.w);
    OUT.geometry.z = -halfWidth;
    EmitVertex();

    OUT.geometry.xy = vec2(p1.x, NN_G3D_PROJECTION_DIR(-p1.y)) * 0.5 + 0.5;
    gl_Position = vec4((p1.xy - n + e) * p1.w, p1.z, p1.w);
    OUT.geometry.z = halfWidth;
    EmitVertex();

    gl_Position = vec4((p1.xy + n + e) * p1.w, p1.z, p1.w);
    OUT.geometry.z = -halfWidth;
    EmitVertex();

    EndPrimitive();
}

void main()
{
    vec4 v0 = vec4(gl_in[0].gl_Position.xy / gl_in[0].gl_Position.w, gl_in[0].gl_Position.zw);
    vec4 v1 = vec4(gl_in[1].gl_Position.xy / gl_in[1].gl_Position.w, gl_in[1].gl_Position.zw);
    vec4 v2 = vec4(gl_in[2].gl_Position.xy / gl_in[2].gl_Position.w, gl_in[2].gl_Position.zw);
    vec4 v3 = vec4(gl_in[3].gl_Position.xy / gl_in[3].gl_Position.w, gl_in[3].gl_Position.zw);
    vec4 v4 = vec4(gl_in[4].gl_Position.xy / gl_in[4].gl_Position.w, gl_in[4].gl_Position.zw);
    vec4 v5 = vec4(gl_in[5].gl_Position.xy / gl_in[5].gl_Position.w, gl_in[5].gl_Position.zw);

    if (IsFrontFace(v0.xy, v2.xy, v4.xy))
    {
        // 隣接トライアングルを使ってそれぞれのエッジが手前を向いているか調べる。
        // ring buffer のサイズを減らすために１つのエッジのみ出力する。
        if(!IsFrontFace(v0.xy, v1.xy, v2.xy))
        {
            EmitEdge(v0, v2);
        }
        else if(!IsFrontFace(v2.xy, v3.xy, v4.xy))
        {
            EmitEdge(v2, v4);
        }
        else if(!IsFrontFace(v0.xy, v4.xy, v5.xy))
        {
            EmitEdge(v4, v0);
        }
    }
}
#endif

//--------------------------------------------------------------------------------------------------

#ifdef NN_G3D_FRAGMENT_SHADER

void main()
{
    float depth = texture(renderTargetDepth, IN.geometry.xy).r;
    // デプスバッファを使って裏面のアウトラインが出ないようにする。
    // エイリアシングを避けるために depth 比較にオフセットを足す。
    if (depth + 0.001 < gl_FragCoord.z)
    {
        discard;
    }

    // 端に向けて透明にする。
    float offset = halfWidth * 0.5;
    float alpha = 1.0;
    float d = abs(IN.geometry.z);
    if (d > halfWidth - offset)
    {
        alpha = 1.0 - (d - offset) / offset;
    }

    o_Color = vec4(lineColor, alpha);
}

#endif // NN_G3D_FRAGMENT_SHADER
