﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{Outline.cpp,PageSampleG3dOutline}
 *
 * @brief
 * ジオメトリシェーダーを使用し、モデルのアウトライン表示を行うサンプルプログラム
 */

/**
 * @page PageSampleG3dOutline G3dDemo Outline
 * @tableofcontents
 *
 * @image html Applications\G3dDemo\outline.png
 *
 * @brief
 * サンプルプログラム Outline の解説です。ジオメトリシェーダーを使用し、モデルのアウトライン表示を行います。
 *
 * @section PageSampleG3dOutline_SectionBrief 概要
 * ジオメトリシェーダーを使用し、モデルのアウトライン表示を行うサンプルです。
 *
 * @section PageSampleG3dOutline_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/G3dDemo
 * Samples/Sources/Applications/G3dDemo @endlink 以下にあります。
 *
 * @section PageSampleG3dOutline_SectionNecessaryEnvironment 必要な環境
 * 特になし。
 *
 * @section PageSampleG3dOutline_SectionHowToOperate 操作方法
 * Joy-Con、DebugPad、PCの場合キーボード、マウスによる入力を用いて、以下の操作が可能です。
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> Startボタン、スペースキーまたはタッチスクリーン長押し、Joy-Con の+ボタン </td><td> メニューに戻ります。 </td></tr>
 * <tr><td> 左スティックまたはマウス左ボタン押し </td><td> カメラ回転。マウスの場合、画面の中心点からクリック位置への方向、距離によって回転します。 </td></tr>
 * <tr><td> 右スティックまたはマウス右ボタン押し </td><td> カメラ移動。マウスの場合、画面の中心点からクリック位置への方向、距離によって移動します。 </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleG3dOutline_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleG3dOutline_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、メニューから Outline を選択し、実行してください。
 *
 * @section PageSampleG3dOutline_SectionDetail 解説
 * このサンプルプログラムは、シェーダーアーカイブ、モデルリソースを読み込み、1パス目でモデルをレンダリングし、
 * 2パス目でアウトラインをレンダリングし、表示します。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - シェーダーアーカイブ読み込み、初期化
 * - モデルリソース読み込み、初期化およびテクスチャー初期化
 * - サンプラーをディスクリプタプールに登録
 * - nns::g3d を使用したモデルのセットアップ
 * - nns::g3d を使用したアウトライン用モデルのセットアップ
 * - ループ開始
 * - 前フレームで作成したコマンドリスト実行
 * - 各行列更新
 * - モデル更新
 * - 1 パス目のモデル描画コマンドを生成
 * - 2 パス目のアウトライン描画コマンドを生成
 * - ループ開始に戻る
 *
 */

#include "g3ddemo_GfxUtility.h"
#include "g3ddemo_ModelUtility.h"

namespace g3ddemo = nn::g3d::demo;

namespace {

#define RESOURCE_PATH "Resource:/"
const char* g_DemoBfshaPath = RESOURCE_PATH "shader/demo.bfsha";
const char* g_OutlineBfshaPath = RESOURCE_PATH "shader/outline.bfsha";
const char* g_EnvBfresPath = RESOURCE_PATH "env.bfres";
const char* g_OutlineBfresPath = RESOURCE_PATH "outline.bfres";
const char* g_HumanBfresPath = RESOURCE_PATH "humanF.bfres";

nn::util::Vector3fType g_CameraPosition;
nn::util::Vector3fType g_CameraUp;
nn::util::Vector3fType g_CameraTarget;

g3ddemo::ResourceHolder g_Holder;
g3ddemo::ScreenInfo     g_ScreenInfo;
nns::gfx::FrameBuffer   g_FrameBuffer;

nns::g3d::RenderView      g_RenderView;
nns::g3d::ModelAnimObj*   g_pHumanModelAnimObj;
nns::g3d::RenderModelObj* g_pHumanRenderModelObj;

enum DrawPassType
{
    DrawPassType_Default,
    DrawPassType_OutLine,
    DrawPassType_Count
};

const int g_BufferingCount = 2;

void Calculate(nns::gfx::GraphicsFramework* pGfxFramework, void* pUserData)
{
    NN_UNUSED(pGfxFramework);
    int uniformBufferIndex = *reinterpret_cast<int*>(pUserData);

    // 計算
    {
        // カメラコントロール
        const g3ddemo::Pad& pad = g3ddemo::GetPad();
        ControlCamera(&g_CameraPosition, &g_CameraUp, &g_CameraTarget, &g_RenderView.GetViewMtx(0), &pad);
        {
            nn::util::Matrix4x3fType matrix;
            MatrixLookAtRightHanded(&matrix,
                g_CameraPosition,
                g_CameraTarget,
                g_CameraUp);
            g_RenderView.SetViewMtx(0, matrix);
        }

        // 投影行列
        {
            nn::util::Matrix4x4fType matrix;
            MatrixPerspectiveFieldOfViewRightHanded(&matrix, nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f, 10.0f, 40000.0f);
            g_RenderView.SetProjectionMtx(0, matrix);
        }

        // ビューの更新処理
        g_RenderView.Calculate(uniformBufferIndex);

        // モデルの更新処理
        g_pHumanModelAnimObj->CalculateView(0, uniformBufferIndex, g_RenderView.GetViewMtx(0));
        g_pHumanModelAnimObj->Calculate();
        g_pHumanModelAnimObj->CalculateUniformBlock(uniformBufferIndex);
    }
}

void MakeCommand(nns::gfx::GraphicsFramework* pGfxFramework, int bufferIndex, void* pUserData)
{
    NN_UNUSED(pUserData);
    int uniformBufferIndex = *reinterpret_cast<int*>(pUserData);

    // コマンド生成
    {
        pGfxFramework->BeginFrame(bufferIndex);
        nn::gfx::CommandBuffer* pCommandBuffer = pGfxFramework->GetRootCommandBuffer(bufferIndex);

        // モデル描画 コマンド生成
        {
            // カラーターゲットをクリア
            nn::gfx::ColorTargetView* pColor = pGfxFramework->GetColorTargetView();
            pCommandBuffer->ClearColor(pColor, 0.3f, 0.3f, 0.3f, 1.0f, nullptr);

            // 深度ステンシルをクリア
            nn::gfx::DepthStencilView* pDepth = g_FrameBuffer.GetDepthBuffer()->GetDepthStencilView();
            pCommandBuffer->ClearDepthStencil(pDepth, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

            // レンダーターゲットをセット
            pCommandBuffer->SetRenderTargets(1, &pColor, pDepth);

            // ビューポートシザーステートをセット
            pCommandBuffer->SetViewportScissorState(pGfxFramework->GetViewportScissorState());

            g_pHumanRenderModelObj->Draw(pCommandBuffer, DrawPassType_Default, 0, uniformBufferIndex);
            pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_DepthStencil);
            pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture);
        }

        // アウトライン描画 コマンド生成
        {
            g_pHumanRenderModelObj->Draw(pCommandBuffer, DrawPassType_OutLine, 0, uniformBufferIndex);
        }

        // テキスト描画 コマンド生成
        {
            g_ScreenInfo.StartPrint();
            g_ScreenInfo.Print("[Outline]\n");
            g_ScreenInfo.EndPrint();
            g_ScreenInfo.AdjustWindowSize();
            g_ScreenInfo.Draw(pCommandBuffer);
        }

        pGfxFramework->EndFrame(bufferIndex);
    }
}

} // anonymous namespace

//--------------------------------------------------------------------------------------------------
int OutlineMain()
{
    nns::gfx::GraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();
    pGfxFramework->SetFrameworkMode(nns::gfx::GraphicsFramework::FrameworkMode_DeferredSubmission);

    nn::gfx::Device* pDevice = pGfxFramework->GetDevice();

    // スクリーン情報表示を初期化
    g_ScreenInfo.Initialize(pDevice);
    g_ScreenInfo.SetWindowPosition(16, 16);
    g_ScreenInfo.SetFontSize(20);

    // フレームバッファーを初期化(デプスのみ使用)
    nn::gfx::DescriptorSlot depthTextureSlot;
    {
        int width = pGfxFramework->GetDisplayWidth();
        int height = pGfxFramework->GetDisplayHeight();
        nns::gfx::FrameBuffer::InfoType info(width, height);
        g3ddemo::InitializeFrameBuffer(&g_FrameBuffer, &info);

        // デプスをテクスチャーとしてディスクリプタプールに登録
        depthTextureSlot = g3ddemo::RegisterTextureToDescriptorPool(g_FrameBuffer.GetDepthBuffer()->GetTextureView());
    }

    // リソースホルダの初期化
    g_Holder.Initialize();

    // シェーダーアーカイブの初期化
    nn::g3d::ResShaderArchive* pDemoShaderArchive;
    {
        nn::g3d::ResShaderFile* pFile = g3ddemo::LoadResource<nn::g3d::ResShaderFile>(g_DemoBfshaPath);
        pFile->Setup(pDevice);
        g_Holder.shaderFiles.PushBack(pFile);
        pDemoShaderArchive = pFile->GetResShaderArchive();
    }
    nn::g3d::ResShaderArchive* pOutlineShaderArchive;
    {
        nn::g3d::ResShaderFile* pFile = g3ddemo::LoadResource<nn::g3d::ResShaderFile>(g_OutlineBfshaPath);
        pOutlineShaderArchive = pFile->GetResShaderArchive();
        pOutlineShaderArchive->Setup(pDevice);
        g_Holder.shaderFiles.PushBack(pFile);
    }

    // モデルリソースの初期化
    nn::g3d::ResFile* pEnvModelFile;
    {
        pEnvModelFile = g3ddemo::LoadResource<nn::g3d::ResFile>(g_EnvBfresPath);
        pEnvModelFile->Setup(pDevice);
        g_Holder.files.PushBack(pEnvModelFile);
    }
    nn::g3d::ResFile* pOutlineModelFile;
    {
        pOutlineModelFile = g3ddemo::LoadResource<nn::g3d::ResFile>(g_OutlineBfresPath);
        pOutlineModelFile->Setup(pDevice);
        g3ddemo::RegisterSamplerToDescriptorPool(pOutlineModelFile);
        g_Holder.files.PushBack(pOutlineModelFile);
    }
    nn::g3d::ResFile* pHumanModelFile;
    {
        pHumanModelFile = g3ddemo::LoadResource<nn::g3d::ResFile>(g_HumanBfresPath);
        pHumanModelFile->Setup(pDevice);
        g3ddemo::SetupTexture(pDevice, pHumanModelFile, g3ddemo::TextureBindCallback);
        g3ddemo::RegisterSamplerToDescriptorPool(pHumanModelFile);
        g_Holder.files.PushBack(pHumanModelFile);
    }

    // モデルの頂点ステート、デプスステンシルステート、ブレンドステート、ラスタライザーステート、シェーディングモデルを管理するRenderModelを作成
    nns::g3d::RenderModel* pHumanRenderModel;
    {
        nn::g3d::ResModel* pResModel = pHumanModelFile->GetModel(0);

        pHumanRenderModel = nns::g3d::CreateRenderModel(pDevice, pResModel, DrawPassType_Count);
        pHumanRenderModel->CreateDrawPass(pDevice, DrawPassType_Default, pDemoShaderArchive);
        pHumanRenderModel->CreateDrawPass(pDevice, DrawPassType_OutLine, pOutlineShaderArchive->FindShadingModel("outline_for_draw"));
        g_Holder.renderModels.PushBack(pHumanRenderModel);

        // アウトライン描画パスのブレンドステートとデプスステンシルステートを設定
        const nn::gfx::BlendState* pBlendState = g3ddemo::GetBlendState(g3ddemo::BlendMode_Translucent);
        const nn::gfx::DepthStencilState* pDepthStencilState = g3ddemo::GetDepthStencilState(g3ddemo::DepthMode_WriteOff);
        for (int materialIndex = 0, materialCount = pResModel->GetMaterialCount(); materialIndex < materialCount; ++materialIndex)
        {
            pHumanRenderModel->GetRenderMaterial(DrawPassType_OutLine, materialIndex)->SetBlendState(pBlendState);
            pHumanRenderModel->GetRenderMaterial(DrawPassType_OutLine, materialIndex)->SetDepthStencilState(pDepthStencilState);
        }
    }

    // モデルおよびアニメーションの計算処理を簡略化するModelAnimObjを作成
    {
        nn::g3d::ModelObj::Builder builder(pHumanModelFile->GetModel(0));
        builder.ShapeBufferingCount(g_BufferingCount);
        builder.SkeletonBufferingCount(g_BufferingCount);
        builder.MaterialBufferingCount(g_BufferingCount);
        g_pHumanModelAnimObj = nns::g3d::CreateModelAnimObj(pDevice, builder);
        g_Holder.modelAnimObjs.PushBack(g_pHumanModelAnimObj);

        // アニメーション追加
        g_pHumanModelAnimObj->CreateAnim(pHumanModelFile->FindSkeletalAnim("humanF_jump"));
    }

    // モデル描画用のRenderModelObj作成
    {
        g_pHumanRenderModelObj = nns::g3d::CreateRenderModelObj(g_pHumanModelAnimObj->GetModelObj(), pHumanRenderModel);
        g_Holder.renderModelObjs.PushBack(g_pHumanRenderModelObj);

        // このデモではシェーダーを動的に切り替えないため、ここでシェーダーを選択
        g3ddemo::WriteSkinningOption(g_pHumanRenderModelObj);
        g_pHumanRenderModelObj->UpdateShader(pDevice);
    }

    // 描画時に必要となるg3dの管理外のユニフォームブロックを作成し、RenderModelObjに設定
    // ユニフォームブロックの管理を簡単にするため、別モデルのマテリアルユニフォームブロックとして扱っている
    nn::g3d::ModelObj* pEnvModelObj;
    {
        nn::g3d::ResModel* pResModel = pEnvModelFile->GetModel(0);
        nn::g3d::ShaderUtility::BindShaderParam(pResModel->GetMaterial(0), pDemoShaderArchive->FindShadingModel("env"));
        nn::g3d::ModelObj::Builder builder(pResModel);
        builder.MaterialBufferingCount(g_BufferingCount);
        pEnvModelObj = nns::g3d::CreateModelObj(pDevice, builder);
        pEnvModelObj->CalculateMaterial(0);
        pEnvModelObj->CalculateMaterial(1);
        g_Holder.modelObjs.PushBack(pEnvModelObj);

        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = pEnvModelObj->GetMaterial(0)->GetMaterialBlock(0);
        buffers[1] = pEnvModelObj->GetMaterial(0)->GetMaterialBlock(1);
        g_pHumanRenderModelObj->BindUniformBlock(DrawPassType_Default, "env", buffers, g_BufferingCount);
    }

    // アウトライン描画用のRenderModelObjのユニフォームブロックを設定
    nn::g3d::ModelObj* pOutlineModelObj;
    {
        nn::g3d::ResModel* pResModel = pOutlineModelFile->GetModel(0);
        nn::g3d::ShaderUtility::BindShaderParam(pResModel->GetMaterial(0), pOutlineShaderArchive->FindShadingModel("outline_for_setting"));
        nn::g3d::ModelObj::Builder builder(pResModel);
        builder.MaterialBufferingCount(g_BufferingCount);
        pOutlineModelObj = nns::g3d::CreateModelObj(pDevice, builder);
        pOutlineModelObj->CalculateMaterial(0);
        pOutlineModelObj->CalculateMaterial(1);
        g_Holder.modelObjs.PushBack(pOutlineModelObj);

        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = pOutlineModelObj->GetMaterial(0)->GetMaterialBlock(0);
        buffers[1] = pOutlineModelObj->GetMaterial(0)->GetMaterialBlock(1);
        g_pHumanRenderModelObj->BindUniformBlock(DrawPassType_OutLine, "outline", buffers, g_BufferingCount);

        // 1パス目のデプスをアウトライン描画時にデプステクスチャーとして参照するので設定
        nn::g3d::SamplerRef samplerRef = pOutlineModelObj->GetMaterial(0)->FindSampler("_d0");
        g_pHumanRenderModelObj->BindSampler(DrawPassType_OutLine, "_d0", depthTextureSlot, samplerRef.GetDescriptorSlot());
    }

    // ビュー用のユニフォームブロックを設定
    g_RenderView.Initialize(pDevice, 1, g_BufferingCount);
    {
        const nn::gfx::Buffer* buffers[g_BufferingCount];
        buffers[0] = g_RenderView.GetViewUniformBlock(0, 0);
        buffers[1] = g_RenderView.GetViewUniformBlock(0, 1);
        g_pHumanRenderModelObj->BindUniformBlock("view", buffers, g_BufferingCount);
    }

    // カメラ初期化
    {
        VectorSet(&g_CameraPosition, 190.0f, 142.5f, 190.0f);
        VectorSet(&g_CameraUp, 0.0f, 1.0f, 0.0f);
        VectorSet(&g_CameraTarget, 0.0f, 71.25f, 0.0f);

        nn::util::Matrix4x3f matrix;
        MatrixLookAtRightHanded(&matrix,
            g_CameraPosition,
            g_CameraTarget,
            g_CameraUp);
        g_RenderView.SetViewMtx(0, matrix);
    }

    // シェーダーに必要なスクラッチメモリーを設定
    g3ddemo::SetShaderScratchMemory(g_Holder.shaderFiles);

    int uniformBufferIndex = 0;
    pGfxFramework->SetCalculateCallback(Calculate, &uniformBufferIndex);
    pGfxFramework->SetMakeCommandCallback(MakeCommand, &uniformBufferIndex);
    pGfxFramework->ResetFrame();

    g3ddemo::Pad& pad = g3ddemo::GetPad();
    pad.Reset();
    // メインループ
    while (NN_STATIC_CONDITION(1))
    {
        // タッチ長押しで終了
        if (g3ddemo::isExitDemo())
        {
            break;
        }

        if (!pad.Read() || pad.IsTriggered(g3ddemo::Pad::BUTTON_START))
        {
            break;
        }

        // フレーム処理実行
        pGfxFramework->ProcessFrame();

        uniformBufferIndex ^= 1;
    }
    pGfxFramework->QueueFinish();

    // 終了処理
    g3ddemo::DestroyAll(pDevice, &g_Holder);

    // ビュー用ユニフォームブロックの破棄
    {
        g_RenderView.Finalize(pDevice);
    }

    // フレームバッファーの破棄
    {
        g3ddemo::FinalizeFrameBuffer(&g_FrameBuffer);
    }

    // スクリーン情報表示の破棄
    {
        g_ScreenInfo.Cleanup(pDevice);
    }

    pGfxFramework->SetCalculateCallback(nullptr, nullptr);
    pGfxFramework->SetMakeCommandCallback(nullptr, nullptr);

    return EXIT_SUCCESS;
} // NOLINT
