﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include "g3ddemo_ViewerUtility.h"

namespace g3ddemo = nn::g3d::demo;

static const char*  RenderInfoClearColorName = "clear_color";
static const int    ViewCount = 1;

enum DrawPassType
{
    DrawPassType_Default,
    DrawPassType_Debug,
    DrawPassType_Count
};

// Edit デモで利用する viewer クラスです。
class EditViewer : public g3ddemo::DemoViewerBase
{
    NN_DISALLOW_COPY(EditViewer);
public:
    EditViewer() NN_NOEXCEPT;

    int GetModelInstanceCount() const NN_NOEXCEPT;

    // 3DEditor でモデルがロードされた時に作成するモデルのインスタンス数を設定します。
    void SetModelInstanceCount(int count) NN_NOEXCEPT;

    void Update() NN_NOEXCEPT;

    // 3DEditor でテクスチャーがロードされたときにコールされます。
    virtual void TextureFileLoaded(const nn::g3d::viewer::TextureFileLoadedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    // 3DEditor でテクスチャーがアンロードされたときにコールされます。
    virtual void TextureFileUnloaded(const nn::g3d::viewer::TextureFileUnloadedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    // 3DEditor でモデルがロードされたときにコールされます。
    virtual void ModelFileLoaded(nn::g3d::viewer::ModelFileLoadedOutArg& outArg, const nn::g3d::viewer::ModelFileLoadedArg& inArg) NN_NOEXCEPT NN_OVERRIDE;

    // 3DEditor でモデルがアンロードされたときにコールされます。
    virtual void ModelFileUnloaded(const nn::g3d::viewer::ModelFileUnloadedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    virtual void ModelAttached(const nn::g3d::viewer::ModelAttachedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    virtual void ModelDetached(const nn::g3d::viewer::ModelDetachedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    virtual void ShaderAssignUpdated(const nn::g3d::viewer::ShaderAssignUpdatedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    virtual void MaterialUpdated(const nn::g3d::viewer::MaterialUpdatedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    virtual void ShapeUpdated(const nn::g3d::viewer::ShapeUpdatedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    virtual void BoneBindUpdated(const nn::g3d::viewer::BoneBindUpdatedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    // 3DEditor でモデルの配置情報が編集されたときにコールされます。
    virtual void ModelLayoutUpdated(const nn::g3d::viewer::ModelLayoutUpdatedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    // 3DEditor からモデルの配置情報を要求されたときにコールされます。
    virtual void SendModelLayoutRequested(nn::g3d::viewer::SendModelLayoutRequestedOutArg& outputData, const nn::g3d::viewer::SendModelLayoutRequestedArg& arg)NN_NOEXCEPT NN_OVERRIDE;

    virtual void ShaderAttached(const nn::g3d::viewer::ShaderAttachedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    virtual void ShaderDetached(const nn::g3d::viewer::ShaderDetachedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    virtual void SceneAnimBound(const nn::g3d::viewer::SceneAnimBoundArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    virtual void SceneAnimUnbound(const nn::g3d::viewer::SceneAnimUnboundArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    virtual void ApplySceneAnimRequested(const nn::g3d::viewer::ApplySceneAnimRequestedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    virtual void RenderInfoUpdated(const nn::g3d::viewer::RenderInfoUpdatedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    virtual void SendRenderInfoRequested(const nn::g3d::viewer::SendRenderInfoRequestedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    // 3DEditor でモデルやマテリアルが選択されたときに実行される処理です。
    virtual void TargetSelected(const nn::g3d::viewer::TargetSelectedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    // アタッチシェーダー時、遅延コンパイル後にシェーダープログラムを変更した後に実行される処理です。
    virtual void ShaderProgramUpdated(const nn::g3d::viewer::ShaderProgramUpdatedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

    virtual void SamplerParamUpdated(const nn::g3d::viewer::SamplerParamUpdatedArg& arg) NN_NOEXCEPT NN_OVERRIDE;

private:
    void ChangeShader(nn::g3d::ResShaderArchive* pResShaderArchive) NN_NOEXCEPT;

    void UpdateShader() NN_NOEXCEPT;

    static void QueueAttachModel(nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT;

private:
    int m_ModelInstanceCount;
};

// Edit デモで表示するメニューを管理するクラスです。
class EditMenu
{
    NN_DISALLOW_COPY(EditMenu);
public:

    enum Item
    {
        ItemModelInstanceCount,
        ItemModel,
        ItemAnim,
        ItemShader,
        ItemCount
    };

    EditMenu() NN_NOEXCEPT
        : m_SelectedItemIndex(ItemModel)
        , m_pResourceHolder(nullptr)
    {
        m_SelectedItemIndexArray[ItemModelInstanceCount] = 1;
        m_SelectedItemIndexArray[ItemModel] = 1;
        m_SelectedItemIndexArray[ItemAnim] = 0;
        m_SelectedItemIndexArray[ItemShader] = 0;
    }

    void Initialize(g3ddemo::ResourceHolder& pHolder, nn::g3d::ResShaderArchive* pBranchShader, nn::g3d::ResShaderOption* pOutputOption) NN_NOEXCEPT
    {
        m_pResourceHolder = &pHolder;
        m_pBranchShader = pBranchShader;
        m_pOutputOption = pOutputOption;
    }

    // キー入力をパラメータに適用します。
    void CalculateCPU(EditViewer* pEditViewer, const g3ddemo::Pad &pad) NN_NOEXCEPT;

    // 設定されているパラメータの値を screenInfo に書き込みます。
    void CalculateGPU(g3ddemo::ScreenInfo& screenInfo) NN_NOEXCEPT;

    int GetSelectedItemIndex(Item item) const NN_NOEXCEPT
    {
        return m_SelectedItemIndexArray[item];
    }

    void SetSelectedItemIndex(Item item, int value) NN_NOEXCEPT
    {
        m_SelectedItemIndexArray[item] = value;
    }

private:
    int m_SelectedItemIndex;
    int m_SelectedItemIndexArray[ItemCount];

    g3ddemo::ResourceHolder*    m_pResourceHolder;
    nn::g3d::ResShaderArchive*  m_pBranchShader;
    nn::g3d::ResShaderOption*   m_pOutputOption;
};

// モデルインスタンス生成後呼び出されるコールバック関数の型です。
typedef void(*CreateModelInstanceCallback)(nn::g3d::ModelObj* pModelObj);

// RenderModelObj を生成します。
void CreateRenderModelObjs(nn::g3d::ResModel* pResModel, int instanceCount, CreateModelInstanceCallback pFunction) NN_NOEXCEPT;

// RenderModelObj を描画するために必要なユニフォームブロックを割り当てます。
void RegisterUniformBlock(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT;

// シーンにビュー行列とプロジェクション行列を設定します。
void SetViewProjectionMatrix(int index, const nn::util::Matrix4x3fType& viewMtx, const nn::util::Matrix4x4fType& projMtx) NN_NOEXCEPT;

// EnvModel を用いて フレームバッファーの背景色を設定します。
void SetClearColorFromEnvModel() NN_NOEXCEPT;

// ResRenderInfo からフレームバッファーの背景色を設定します。
void SetClearColor(const nn::g3d::ResRenderInfo* pResRenderInfo) NN_NOEXCEPT;

// Debug シェーダーのキーを書き換えます。
void WriteDynamicKey(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT;
void WriteDynamicKey() NN_NOEXCEPT;

// モデルの位置をリセットします。
void ResetModelPosition(const nn::g3d::ModelObj* pTargetModelObj) NN_NOEXCEPT;

// 選択するモデルのインデックスを設定します。
void SetSelectedModelIndex(int index) NN_NOEXCEPT;

// 選択しているモデルのインデックスを取得します。
int GetSelectedModelIndex() NN_NOEXCEPT;

// RenderModelObj に必要な設定を行います。
void SetupRenderModelObjForEdit(nns::g3d::RenderModelObj* pRenderModelObj) NN_NOEXCEPT;
