﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{FsRom.cpp,PageSampleFsRom}
 *
 * @brief
 *  リソースデータ（ROM）のファイルシステム操作のサンプルプログラム
 */

/**
 * @page PageSampleFsRom リソースデータ（ROM）のファイルシステム操作
 * @tableofcontents
 *
 * @brief
 *  リソースデータ（ROM）のファイルシステム操作のサンプルプログラムの解説です。
 *
 * @section PageSampleFsRom_SectionBrief 概要
 *  ここでは、ROM ファイルシステムを使ったサンプルプログラムの説明をします。
 *
 *  ファイル・ディレクトリの操作方法については、
 *  @ref nn::fs "ファイルシステムライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleFsRom_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/FsRom Samples/Sources/Applications/FsRom @endlink 以下にあります。
 *
 * @section PageSampleFsRom_SectionNecessaryEnvironment 必要な環境
 *  特になし。
 *
 * @section PageSampleFsRom_SectionHowToOperate 操作方法
 *  特になし。
 *
 * @section PageSampleFsRom_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleFsRom_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleFsRom_SectionDetail 解説
 *
 * @subsection PageSampleFsRom_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  FsRom.cpp
 *  @includelineno FsRom.cpp
 *
 * @subsection PageSampleFsRom_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの全体像は以下の通りです。
 *
 *  - MountRom 関数を利用したリソースデータ（ROM）のファイルシステムの操作例
 *
 *  リソースデータのファイルシステムは、MountRom 関数を使用することで利用できます。
 *
 *  このサンプルではこのソースファイルと同じディレクトリに配置した "dataSrc" ディレクトリをリソースデータの参照先として設定し、その中の test/testdata というエントリを読み込んでいます。@n
 *  リソースデータの参照先はプロジェクトファイルで設定しています。
 *  NintendoSDK ドキュメントの「FS ライブラリ > マニュアル > 機能説明 > リソースデータ（ROM）」の項目も併せて参照してください。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  FsRom_OutputExample.txt
 *
 */

#include <cstdlib>
#include <cstring>
#include <new>

#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>

extern "C" void nnMain()
{
    nn::Result result;

    size_t cacheSize = 0;

    char* cacheBuffer = nullptr;

    // ファイルシステムをマウントします。
    // マウントにはキャッシュバッファが必要です。
    {
        NN_LOG("Mount Rom\n");

        // ファイルシステムのメタデータキャッシュに必要なバッファサイズを取得します。
        // 取得失敗時はライブラリ内でアボートするため、エラーハンドリングは不要です。
        (void)nn::fs::QueryMountRomCacheSize(&cacheSize);

        // キャッシュバッファを確保します。
        cacheBuffer = new(std::nothrow) char[cacheSize];
        if (cacheBuffer == nullptr)
        {
            NN_ASSERT(false, "Cache buffer is null.\n");
            return;
        }

        // ファイルシステムをマウントします。
        // キャッシュバッファはアンマウントするまで解放しないでください。
        result = nn::fs::MountRom("rom", cacheBuffer, cacheSize);
        // 失敗した際は必ずアボートしてください。
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    // リソースデータを読み込みます。
    {
        const char* FilePath = "rom:/test/testdata";

        nn::fs::FileHandle file;
        result = nn::fs::OpenFile(&file, FilePath, nn::fs::OpenMode_Read);
        if (nn::fs::ResultPathNotFound::Includes(result))
        {
            // 対象ファイルが存在しません。
            // 存在するファイルしか開かない場合は、このエラーハンドリングは不要です。
            NN_ASSERT(false, "File not found.\n");
            return;
        }
        // MountRom でマウントした場合は OpenMode_Write を指定してオープンできないため、ResultTargetLocked のハンドリングは不要です。
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。

        char buffer[1024];
        size_t readSize = 0;

        // リソースデータを buffer に読み込みます。
        // 読み込み失敗時はライブラリ内でアボートするため、エラーハンドリングは不要です。
        NN_LOG("Read\t%s\n", FilePath);
        (void)nn::fs::ReadFile(&readSize, file, 0, buffer, sizeof(buffer));

        NN_LOG("\tsize: %d\n", readSize);
        NN_LOG("\tdata: ");
        for (int i = 0; i < static_cast<int>(readSize); ++i)
        {
            NN_LOG("%02x ", buffer[i]);
        }
        NN_LOG("\n");

        nn::fs::CloseFile(file);
    }

    // ファイルシステムをアンマウントします。
    {
        NN_LOG("Unmount Rom\n\n");

        nn::fs::Unmount("rom");

        // キャッシュバッファを解放します。
        delete[] cacheBuffer;
        cacheBuffer = nullptr;
    }
    return;
}
