﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{FsHost.cpp,PageSampleFsHost}
 *
 * @brief
 *  ホスト PC 上のファイルシステム操作のサンプルプログラム
 */

/**
 * @page PageSampleFsHost ホスト PC 上のファイルシステム操作
 * @tableofcontents
 *
 * @brief
 *  ホスト PC 上のファイルシステム操作のサンプルプログラムの解説です。
 *
 * @section PageSampleFsHost_SectionBrief 概要
 *  ここでは、ホスト PC 上のファイルシステムを使ったサンプルプログラムの説明をします。
 *
 *  ファイル・ディレクトリの操作方法については、
 *  @ref nn::fs "ファイルシステムライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleFsHost_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/FsHost Samples/Sources/Applications/FsHost @endlink 以下にあります。
 *
 * @section PageSampleFsHost_SectionNecessaryEnvironment 必要な環境
 *  特になし。
 *
 * @section PageSampleFsHost_SectionHowToOperate 操作方法
 *  特になし。
 *
 * @section PageSampleFsHost_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 *  また、このデモでは C:/Windows/Temp/FsSmplDir, C:/Windows/Temp/FsSmplFile, $(ProjectDir)/FsSmplFile
 *  というエントリを作成し、操作後削除します。
 *
 *  デモ実行時に当該エントリが存在する場合や、デモプログラムが
 *  当該パスにアクセスできない場合正しく動作しない場合がありますのでご注意ください。
 *
 * @section PageSampleFsHost_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleFsHost_SectionDetail 解説
 *
 * @subsection PageSampleFsHost_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  FsHost.cpp
 *  @includelineno FsHost.cpp
 *
 * @subsection PageSampleFsHost_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの全体像は以下の通りです。
 *
 *  - MountHost 関数を利用したホスト PC 上のファイルシステムの操作例
 *  - MountHostRoot 関数を利用したホスト PC 上のファイルシステムの操作例
 *
 *  ホスト PC 上のファイルシステムは、MountHost 及び MountHostRoot 関数を
 *  使用することで利用できます。
 *
 *  MountHost 関数でマウントした場合は "マウント名:/" で始まるパスで、
 *  MountHostRoot 関数でマウントした場合は ホスト PC 上の絶対パスで、
 *  それぞれ各種ファイルシステムの操作を行います。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  FsHost_OutputExample.txt
 *
 */

#include <cstdlib>
#include <cstring>

#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>

extern "C" void nnMain()
{
    nn::Result result;

    // ホスト PC 上のファイルシステムをマウントします。
    // マウント名に "host" を、ファイルシステムのルートパスに "C:/Windows/Temp" を指定します。
    NN_LOG("Mount C:/Windows/Temp to host\n");
    result = nn::fs::MountHost("host", "C:/Windows/Temp");
    if( nn::fs::ResultPathNotFound::Includes(result) )
    {
        // ホスト PC 上に対象のパスが見つかりません。
        // ホスト PC 上に "C:/Windows/Temp" があるか確かめてください。
        NN_ASSERT(false, "Target directory not found.\n");
        return;
    }
    else if( nn::fs::ResultTargetNotFound::Includes(result) )
    {
        // ホスト PC が認識されませんでした。
        // ホスト PC に正しく接続してください。
        NN_ASSERT(false, "Host PC not found.\n");
        return;
    }
    // 上記以外の失敗は実装ミスのため、必ずアボートしてください。
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    static const size_t FileSize = 512;

    // ディレクトリ・ファイルを作成します。
    {
        NN_LOG("Create\thost:/FsSmplDir\n");
        result = nn::fs::CreateDirectory("host:/FsSmplDir");
        if( nn::fs::ResultPathNotFound::Includes(result) )
        {
            // パスに含まれるディレクトリが存在しません。
            // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
        }
        else if( nn::fs::ResultPathAlreadyExists::Includes(result)
              || nn::fs::ResultTargetLocked::Includes(result) )
        {
            // 対象ディレクトリが既に存在しています。
            // 対象ディレクトリを削除し再度作成する、無視して続行する等の処理を必要に応じて追加してください。
        }
        else if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
        {
            // ストレージに空き領域が不足しています。
            // ホスト側ストレージの空き領域を増やしてください。
            NN_ASSERT("Usable space not enough.\n");
            return;
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。

        NN_LOG("Create\thost:/FsSmplDir/dir1\n");
        result = nn::fs::CreateDirectory("host:/FsSmplDir/dir1");
        if( nn::fs::ResultPathNotFound::Includes(result) )
        {
            // パスに含まれるディレクトリが存在しません。
            // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
        }
        else if( nn::fs::ResultPathAlreadyExists::Includes(result)
              || nn::fs::ResultTargetLocked::Includes(result) )
        {
            // 対象ディレクトリが既に存在しています。
            // 対象ディレクトリを削除し再度作成する、無視して続行する等の処理を必要に応じて追加してください。
        }
        else if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
        {
            // ストレージに空き領域が不足しています。
            // ホスト側ストレージの空き領域を増やしてください。
            NN_ASSERT("Usable space not enough.\n");
            return;
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。

        NN_LOG("Create\thost:/FsSmplDir/file1\n");
        result = nn::fs::CreateFile("host:/FsSmplDir/file1", FileSize);
        if( nn::fs::ResultPathNotFound::Includes(result) )
        {
            // パスに含まれるディレクトリが存在しません。
            // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
        }
        else if( nn::fs::ResultPathAlreadyExists::Includes(result)
              || nn::fs::ResultTargetLocked::Includes(result) )
        {
            // 対象ファイルが既に存在しています。
            // ファイルが既に存在していても構わない場合は、このエラーハンドリングは不要です。
            // エラーハンドリングしない場合、ファイルのサイズが FileSize である保証が無いことに注意してください。
            // 必要であれば対象ファイルを削除してから再度作成してください。
        }
        else if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
        {
            // ストレージに空き領域が不足しています。
            // ホスト側ストレージの空き領域を増やしてください。
            NN_ASSERT("Usable space not enough.\n");
            return;
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。
    }

    // ディレクトリを操作します。
    {
        NN_LOG("Show entries in host:/FsSmplDir\n");
        nn::fs::DirectoryHandle directoryHandle;
        result = nn::fs::OpenDirectory(&directoryHandle, "host:/FsSmplDir", nn::fs::OpenDirectoryMode_All);
        if( nn::fs::ResultPathNotFound::Includes(result) )
        {
            // 対象ディレクトリが存在しません。
            // 対象ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。

        nn::fs::DirectoryEntry directoryEntry[2];
        int64_t readNum = 0;

        // 読み込みに失敗した場合はライブラリ内でアボートするため、エラーハンドリングは不要です。
        (void)nn::fs::ReadDirectory(&readNum, directoryEntry, directoryHandle, 2);

        for (int i = 0; i < readNum; i++)
        {
            NN_LOG("%10s <type = %4s, size = %lld>\n", directoryEntry[i].name,
                   directoryEntry[i].directoryEntryType == nn::fs::DirectoryEntryType_Directory ? "Dir" : "File",
                   directoryEntry[i].fileSize);
        }

        nn::fs::CloseDirectory(directoryHandle);
    }

    // ファイルを操作します。
    {
        nn::fs::FileHandle fileHandle;
        result = nn::fs::OpenFile(&fileHandle, "host:/FsSmplDir/file1", nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
        if( nn::fs::ResultPathNotFound::Includes(result) )
        {
            // 対象ファイルが存在しません。
            // 対象ファイルが必ず存在する場合は、このエラーハンドリングは不要です。
        }
        else if( nn::fs::ResultTargetLocked::Includes(result) )
        {
            // 対象ファイルが既にオープンされています。
            // ファイルが既にオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。

        static const size_t BufferSize = 1024;
        static char writeBuffer[BufferSize];
        for (int i = 0; i < BufferSize; i++)
        {
            writeBuffer[i] = static_cast<char>(i);
        }
        NN_LOG("Write\thost:/FsSmplDir/file1\n");
        result = nn::fs::WriteFile(fileHandle, 0, writeBuffer, BufferSize, nn::fs::WriteOption());
        if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
        {
            // ストレージに空き領域が不足しています。
            // ホスト側ストレージの空き領域を増やしてください。
            NN_ASSERT("Usable space not enough.\n");
            return;
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。


        NN_LOG("Flush\thost:/FsSmplDir/file1\n");

        // フラッシュに失敗した場合はライブラリ内でアボートするため、エラーハンドリングは不要です。
        (void)nn::fs::FlushFile(fileHandle);

        int64_t fileSize = 0;
        NN_LOG("Read\thost:/FsSmplDir/file1\n");

        // サイズ取得に失敗した場合はライブラリ内でアボートするため、エラーハンドリングは不要です。
        (void)nn::fs::GetFileSize(&fileSize, fileHandle);

        NN_ASSERT_EQUAL(fileSize, 1024);

        static char readBuffer[BufferSize];

        // 読み込みに失敗した場合はライブラリ内でアボートするため、エラーハンドリングは不要です。
        (void)nn::fs::ReadFile(fileHandle, 0, readBuffer, static_cast<size_t>(fileSize));

        NN_ASSERT(std::memcmp(writeBuffer, readBuffer, BufferSize) == 0);

        nn::fs::CloseFile(fileHandle);
    }

    // ディレクトリ・ファイルを削除します。
    result = nn::fs::DeleteDirectoryRecursively("host:/FsSmplDir");
    if( nn::fs::ResultPathNotFound::Includes(result) )
    {
        // 対象ディレクトリが存在しません。
        // 対象ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
    }
    else if( nn::fs::ResultTargetLocked::Includes(result) )
    {
        // 対象ファイルが既にオープンされています。
        // ファイルがオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
    }
    // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
    // これ以上のエラーハンドリングは不要です。

    // アンマウントします。
    NN_LOG("Unmount host\n\n");
    nn::fs::Unmount("host");

//-----------------------------------------------------------------------------

    // ホスト PC 上のファイルシステムのルートパスには相対パスを指定できます。
    // 起点となるパスは、プロジェクトプロパティや TargetManager で設定できます。
    // 詳細については NintendoSDK のドキュメントを参照してください。
    NN_LOG("MountHost with relative path\n");
    result = nn::fs::MountHost("host", "./");
    if( nn::fs::ResultTargetNotFound::Includes(result) )
    {
        // ホスト PC が認識されませんでした。
        // ホスト PC に正しく接続してください。
        NN_ASSERT(false, "Target not found.\n");
        return;
    }
    // 上記以外の失敗は実装ミスのため、必ずアボートしてください。
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_LOG("Create\thost:/FsSmplFile\n");
    result = nn::fs::CreateFile("host:/FsSmplFile", FileSize);
    if( nn::fs::ResultPathNotFound::Includes(result) )
    {
        // パスに含まれるディレクトリが存在しません。
        // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
    }
    else if( nn::fs::ResultPathAlreadyExists::Includes(result)
          || nn::fs::ResultTargetLocked::Includes(result) )
    {
        // 対象ファイルが既に存在しています。
        // ファイルが既に存在していても構わない場合は、このエラーハンドリングは不要です。
        // エラーハンドリングしない場合、ファイルのサイズが FileSize である保証が無いことに注意してください。
        // 必要であれば対象ファイルを削除してから再度作成してください。
    }
    else if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
    {
        // ストレージに空き領域が不足しています。
        // ホスト側ストレージの空き領域を増やしてください。
        NN_ASSERT("Usable space not enough.\n");
        return;
    }
    // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
    // これ以上のエラーハンドリングは不要です。

    NN_LOG("Delete\thost:/FsSmplFile\n");
    result = nn::fs::DeleteFile("host:/FsSmplFile");
    if( nn::fs::ResultPathNotFound::Includes(result) )
    {
        // 対象ファイルが存在しません。
        // 対象ファイルが必ず存在する場合は、このエラーハンドリングは不要です。
    }
    else if( nn::fs::ResultTargetLocked::Includes(result) )
    {
        // 対象ファイルが既にオープンされています。
        // ファイルがオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
    }
    // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
    // これ以上のエラーハンドリングは不要です。

    // アンマウントします。
    NN_LOG("Unmount host\n\n");
    nn::fs::Unmount("host");

//-----------------------------------------------------------------------------

    // ホスト PC 上のパスと同じパス名で操作する場合は、MountHostRoot を利用します。
    NN_LOG("Mount Host Root\n");
    result = nn::fs::MountHostRoot();
    if( nn::fs::ResultTargetNotFound::Includes(result) )
    {
        // ホスト PC が認識されませんでした。
        // ホスト PC に正しく接続してください。
        NN_ASSERT(false, "Target not found.\n");
        return;
    }
    // 上記以外の失敗は実装ミスのため、必ずアボートしてください。
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_LOG("Create\tC:/Windows/Temp/FsSmplFile\n");
    result = nn::fs::CreateFile("C:/Windows/Temp/FsSmplFile", FileSize);
    if( nn::fs::ResultPathNotFound::Includes(result) )
    {
        // パスに含まれるディレクトリが存在しません。
        // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
    }
    else if( nn::fs::ResultPathAlreadyExists::Includes(result)
          || nn::fs::ResultTargetLocked::Includes(result) )
    {
        // 対象ファイルが既に存在しています。
        // ファイルが既に存在していても構わない場合は、このエラーハンドリングは不要です。
        // エラーハンドリングしない場合、ファイルのサイズが FileSize である保証が無いことに注意してください。
        // 必要であれば対象ファイルを削除してから再度作成してください。
    }
    else if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
    {
        // ストレージに空き領域が不足しています。
        // ホスト側ストレージの空き領域を増やしてください。
        NN_ASSERT("Usable space not enough.\n");
        return;
    }
    // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
    // これ以上のエラーハンドリングは不要です。

    NN_LOG("Delete\tC:/Windows/Temp/FsSmplFile\n");
    result = nn::fs::DeleteFile("C:/Windows/Temp/FsSmplFile");
    if( nn::fs::ResultPathNotFound::Includes(result) )
    {
        // 対象ファイルが存在しません。
        // 対象ファイルが必ず存在する場合は、このエラーハンドリングは不要です。
    }
    else if( nn::fs::ResultTargetLocked::Includes(result) )
    {
        // 対象ファイルが既にオープンされています。
        // ファイルがオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
    }
    // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
    // これ以上のエラーハンドリングは不要です。

    // アンマウントします。
    NN_LOG("Unmount Host Root");
    nn::fs::UnmountHostRoot();

    return;
} // NOLINT(impl/function_size)
