﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{FriendsPlayHistory.cpp,PageSampleFriendsPlayHistory}
 *
 * @brief
 *  いっしょに遊んだユーザー登録のサンプルプログラム
 */

/**
 * @page PageSampleFriendsPlayHistory いっしょに遊んだユーザーの登録
 * @tableofcontents
 *
 * @brief
 *  いっしょに遊んだユーザーを登録するサンプルプログラムの解説です。
 *
 * @section PageSampleFriendsPlayHistory_SectionBrief 概要
 *  このプログラムは、いっしょに遊んだユーザーの登録手順を紹介するサンプルです。@n
 *  関数リファレンス (nn::friends) も併せて参照してください。
 *
 * @section PageSampleFriendsPlayHistory_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/FriendsPlayHistory Samples/Sources/Applications/FriendsPlayHistory @endlink 以下にあります。
 *
 * @section PageSampleFriendsPlayHistory_SectionNecessaryEnvironment 必要な環境
 *  ユーザーアカウントを 2 つ以上、作成する必要があります。
 *
 * @section PageSampleFriendsPlayHistory_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleFriendsPlayHistory_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleFriendsPlayHistory_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleFriendsPlayHistory_SectionDetail 解説
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - ユーザーアカウントを Open する。
 *  - いっしょに遊んだユーザーの登録鍵を取得する。
 *  - いっしょに遊んだ情報を交換する。
 *  - いっしょに遊んだユーザーを登録する。
 *  - ユーザーアカウントを Close する。
 */

#include <nn/friends.h>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/account.h>

extern "C" void nnMain()
{
    nn::account::Initialize();

    nn::account::Uid users[nn::account::UserCountMax];
    int count;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, nn::account::UserCountMax));

    if (count >= 2)
    {
        // サンプルでは、解説のため先頭 2 人のユーザーがいっしょに遊んだと仮定します。
        nn::account::UserHandle handle0 = {};
        nn::account::UserHandle handle1 = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle0, users[0]));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle1, users[1]));

        // いっしょに遊んだユーザーの登録鍵を事前に取得します。
        nn::friends::PlayHistoryRegistrationKey key0 = {};
        nn::friends::PlayHistoryRegistrationKey key1 = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key0, users[0], true));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key1, users[1], true));

        // いっしょに遊んだユーザーの情報（いっしょに遊んだユーザーの登録鍵とアプリ内表示名）を交換します。
        // サンプルでは 2 ユーザーとも同一デバイス上に存在するため、特になにも行いません。
        nn::friends::InAppScreenName inAppScreenName0 = {"InAppScreenName[0]"};
        nn::friends::InAppScreenName inAppScreenName1 = {"InAppScreenName[1]"};

        // 相手の情報を登録します。
        nn::friends::AddPlayHistory(users[0], key1, inAppScreenName1, inAppScreenName0);
        nn::friends::AddPlayHistory(users[1], key0, inAppScreenName0, inAppScreenName1);

        nn::account::CloseUser(handle0);
        nn::account::CloseUser(handle1);
    }
    else
    {
        NN_LOG("[sample] Warning: Please create user accounts.\n");
    }

    NN_LOG("[sample] End.\n");
}
