﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NN_PERF_PROFILE_ENABLED
#include <nn/perf.h>
#include <nn/util/util_Color.h>
#include "FrameworkUtility.h"

#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>

//------------------------------------------------------------------------------
//  フレームワークユーティリティ
//------------------------------------------------------------------------------
nns::gfx::GraphicsFramework* g_pGraphicsFramework = nullptr;
nns::gfx::GraphicsFramework::FrameworkMode g_PrevFrameworkMode;
int g_CurrentFrameIndex;

// GPU タイムスタンプ計測のクエリ用コマンドバッファ。
// 本来はフレーム処理内でエンドコマンドの完了を待つ必要がありますが、このデモでは行っていません。
nns::gfx::GraphicsFramework::CommandBuffer g_CommandBufferBegin[3];
nns::gfx::GraphicsFramework::CommandBuffer g_CommandBufferEnd[3];

void InitializeFrameworkUtility(nns::gfx::GraphicsFramework* pGraphicsFramework)
{
    NN_ASSERT(g_pGraphicsFramework == nullptr);
    NN_ASSERT(pGraphicsFramework->IsInitialized());
    g_pGraphicsFramework = pGraphicsFramework;

    // QueryCommandBuffer
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
    info.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);
    size_t commandMemorySize = 1024 * 1024 * 1;
    size_t controlMemorySize = 256;

    NN_ASSERT(pGraphicsFramework->GetBufferCount() <= 3);
    g_pGraphicsFramework->InitializeCommandBuffer(&g_CommandBufferBegin[0], info, commandMemorySize, controlMemorySize);
    g_pGraphicsFramework->InitializeCommandBuffer(&g_CommandBufferBegin[1], info, commandMemorySize, controlMemorySize);
    g_pGraphicsFramework->InitializeCommandBuffer(&g_CommandBufferBegin[2], info, commandMemorySize, controlMemorySize);
    g_pGraphicsFramework->InitializeCommandBuffer(&g_CommandBufferEnd[0], info, commandMemorySize, controlMemorySize);
    g_pGraphicsFramework->InitializeCommandBuffer(&g_CommandBufferEnd[1], info, commandMemorySize, controlMemorySize);
    g_pGraphicsFramework->InitializeCommandBuffer(&g_CommandBufferEnd[2], info, commandMemorySize, controlMemorySize);

    g_CurrentFrameIndex = 0;
    g_PrevFrameworkMode = pGraphicsFramework->GetFrameworkMode();
}

void FinalizeFrameworkUtility()
{
    NN_ASSERT_NOT_NULL(g_pGraphicsFramework);

    // QueryCommandBuffer
    g_pGraphicsFramework->FinalizeCommandBuffer(&g_CommandBufferBegin[0]);
    g_pGraphicsFramework->FinalizeCommandBuffer(&g_CommandBufferBegin[1]);
    g_pGraphicsFramework->FinalizeCommandBuffer(&g_CommandBufferBegin[2]);
    g_pGraphicsFramework->FinalizeCommandBuffer(&g_CommandBufferEnd[0]);
    g_pGraphicsFramework->FinalizeCommandBuffer(&g_CommandBufferEnd[1]);
    g_pGraphicsFramework->FinalizeCommandBuffer(&g_CommandBufferEnd[2]);

    g_pGraphicsFramework = nullptr;
}

void Calculate()
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_AUTO_MEASURE_NAME("Calculate");
    g_pGraphicsFramework->Calculate();
}

void MakeCommand(int bufferIndex)
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_AUTO_MEASURE_NAME("RecordCommand");
    g_pGraphicsFramework->MakeCommand(bufferIndex);

    // MakeBeginTimeStampQueryCommand
    g_pGraphicsFramework->ResetCommandBuffer(&g_CommandBufferBegin[bufferIndex]);
    g_CommandBufferBegin[bufferIndex].object.Begin();
    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Red());
    NN_PERF_BEGIN_MEASURE_GPU(&g_CommandBufferBegin[bufferIndex].object);
    g_CommandBufferBegin[bufferIndex].object.FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    g_CommandBufferBegin[bufferIndex].object.End();

    // MakeEndTimeStampQueryCommand
    g_pGraphicsFramework->ResetCommandBuffer(&g_CommandBufferEnd[bufferIndex]);
    g_CommandBufferEnd[bufferIndex].object.Begin();
    NN_PERF_END_MEASURE_GPU(&g_CommandBufferEnd[bufferIndex].object);
    g_CommandBufferEnd[bufferIndex].object.FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    g_CommandBufferEnd[bufferIndex].object.End();
}

void QueueAppCommand(int bufferIndex)
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_AUTO_MEASURE_NAME("QueueAppCommand");
    g_pGraphicsFramework->ExecuteCommand(bufferIndex);
}

void AcquireTexture(int bufferIndex)
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Blue());
    NN_PERF_AUTO_MEASURE_NAME("AcquireTexture");
    g_pGraphicsFramework->AcquireTexture(bufferIndex);
}

void QueueFinish()
{
    g_pGraphicsFramework->QueueFinish();
}

void QueuePresentTexture(int presentInterval)
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Yellow());
    NN_PERF_AUTO_MEASURE_NAME("PresentTexture");
    g_pGraphicsFramework->QueuePresentTexture(presentInterval);
}

void QueueWaitSync(int bufferIndex)
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_AUTO_MEASURE();
    g_pGraphicsFramework->QueueWaitSync(bufferIndex);
}

void WaitDisplaySync(int bufferIndex, nn::TimeSpan timeout)
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::White());
    NN_PERF_AUTO_MEASURE();
    g_pGraphicsFramework->WaitDisplaySync(bufferIndex, timeout);
}

void WaitGpuSync(int bufferIndex, nn::TimeSpan timeout)
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Black());
    NN_PERF_AUTO_MEASURE();
    g_pGraphicsFramework->WaitGpuSync(bufferIndex, timeout);
}

void QueueBeginTimeStampQueryCommand(int bufferIndex)
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_AUTO_MEASURE();
    g_pGraphicsFramework->GetQueue()->ExecuteCommand(&g_CommandBufferBegin[bufferIndex].object, NULL);
}

void QueueEndTimeStampQueryCommand(int bufferIndex)
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_AUTO_MEASURE();
    g_pGraphicsFramework->GetQueue()->ExecuteCommand(&g_CommandBufferEnd[bufferIndex].object, NULL);
    g_pGraphicsFramework->GetQueue()->Flush();
}

void ProcessFrame()
{
    if (g_pGraphicsFramework->GetFrameworkMode() != g_PrevFrameworkMode)
    {
        if (g_CurrentFrameIndex > 0)
        {
            if (g_PrevFrameworkMode == nns::gfx::GraphicsFramework::FrameworkMode_DeferredSubmission || g_PrevFrameworkMode == nns::gfx::GraphicsFramework::FrameworkMode_Immediate)
            {
                QueuePresentTexture(g_pGraphicsFramework->GetPresentInterval());
                g_pGraphicsFramework->GetQueue()->Flush();
            }
            QueueFinish();
        }

        g_CurrentFrameIndex = 0;
        g_PrevFrameworkMode = g_pGraphicsFramework->GetFrameworkMode();
    }

    int bufferCount = g_pGraphicsFramework->GetBufferCount();
    int currentBufferIndex = g_CurrentFrameIndex % bufferCount;
    int previousBufferIndex = (currentBufferIndex + bufferCount - 1) % bufferCount;
    int nextBufferIndex = (currentBufferIndex + 1) % bufferCount;

    if (g_pGraphicsFramework->GetFrameworkMode() == nns::gfx::GraphicsFramework::FrameworkMode_DeferredExecution)
    {
        Calculate();
        AcquireTexture(currentBufferIndex);
        QueueWaitSync(currentBufferIndex);
        MakeCommand(currentBufferIndex);
        QueueBeginTimeStampQueryCommand(currentBufferIndex);
        QueueAppCommand(currentBufferIndex);
        QueuePresentTexture(g_pGraphicsFramework->GetPresentInterval());
        QueueEndTimeStampQueryCommand(currentBufferIndex);
        WaitDisplaySync(currentBufferIndex, g_pGraphicsFramework->GetWaitSyncTimeout());
        if (g_CurrentFrameIndex > 0)
        {
            WaitGpuSync(previousBufferIndex, g_pGraphicsFramework->GetWaitSyncTimeout());
        }
    }
    else if (g_pGraphicsFramework->GetFrameworkMode() == nns::gfx::GraphicsFramework::FrameworkMode_DeferredSubmission)
    {
        if (g_CurrentFrameIndex > 0)
        {
            QueueBeginTimeStampQueryCommand(previousBufferIndex);
            QueueAppCommand(previousBufferIndex);
            QueuePresentTexture(g_pGraphicsFramework->GetPresentInterval());
            QueueEndTimeStampQueryCommand(previousBufferIndex);
        }
        Calculate();
        AcquireTexture(currentBufferIndex);
        //QueueWaitSync(currentBufferIndex);    // 不要
        MakeCommand(currentBufferIndex);
        WaitDisplaySync(currentBufferIndex, g_pGraphicsFramework->GetWaitSyncTimeout());
        if (g_CurrentFrameIndex > 0)
        {
            WaitGpuSync(previousBufferIndex, g_pGraphicsFramework->GetWaitSyncTimeout());
        }
    }
    else if (g_pGraphicsFramework->GetFrameworkMode() == nns::gfx::GraphicsFramework::FrameworkMode_Immediate)
    {
        if (g_CurrentFrameIndex == 0)
        {
            AcquireTexture(currentBufferIndex);
            //QueueWaitSync(currentBufferIndex);    // 不要
            WaitDisplaySync(currentBufferIndex, g_pGraphicsFramework->GetWaitSyncTimeout());
        }
        Calculate();
        MakeCommand(currentBufferIndex);
        QueueBeginTimeStampQueryCommand(currentBufferIndex);
        QueueAppCommand(currentBufferIndex);
        QueuePresentTexture(g_pGraphicsFramework->GetPresentInterval());
        QueueEndTimeStampQueryCommand(currentBufferIndex);

        AcquireTexture(nextBufferIndex);
        //QueueWaitSync(nextBufferIndex);    // 不要
        WaitDisplaySync(nextBufferIndex, g_pGraphicsFramework->GetWaitSyncTimeout());
        WaitGpuSync(currentBufferIndex, g_pGraphicsFramework->GetWaitSyncTimeout());
    }
    g_CurrentFrameIndex++;
    if (g_CurrentFrameIndex > 10000000)
    {
        g_CurrentFrameIndex = (g_CurrentFrameIndex % bufferCount) + bufferCount;
    }
}
