﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{EnsGenerateCredential.cpp,PageSampleEnsGenerateCredential}
 *
 * @brief
 *  nn::ens::GenerateCredential のサンプルプログラム
 */

/**
 * @page PageSampleEnsGenerateCredential 認証情報の生成
 * @tableofcontents
 *
 * @brief
 *  サービスを利用するために必要な認証情報を生成するサンプルプログラムの解説です。
 *
 * @section EnsGenerateCredential_SectionBrief 概要
 *  サービスを利用するために必要な認証情報( nn::ens::Credential )を生成するサンプルプログラムの解説をします。
 *
 *  認証情報( nn::ens::Credential )は nn::ens::GenerateCredential() を使って生成します。
 *
 *  nn::ens::Credential は、 nn::ens のネットワークサービスを利用するために必要となる認証情報であり、
 *  これに含まれる nn::ens::UserId はサービスを利用するユーザーを一意に特定する ID です。
 *
 *  nn::ens のネットワークサービスの利用にはネットワークサービスアカウントのIDトークンが必要です。
 *  @ref nn::account "nn::account のリファレンス" やドキュメントも併せて参照してください。
 *
 * @section EnsGenerateCredential_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/EnsGenerateCredential Samples/Sources/Applications/EnsGenerateCredential @endlink 以下にあります。
 *
 * @section EnsGenerateCredential_SectionNecessaryEnvironment 必要な環境
 *  - ネットワーク接続できる環境
 *  - ネットワークサービスアカウントの準備
 *
 * @section EnsGenerateCredential_SectionHowToOperate 操作方法
 *  プレイヤー選択などはUIに従って操作してください。
 *  プレイヤー選択後、自動的に処理が実行され、サンプルが終了します。
 *
 * @section EnsGenerateCredential_SectionPrecaution 注意事項
 *  実行結果はログに出力されます。
 *
 * @section EnsGenerateCredential_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section EnsGenerateCredential_SectionDetail 解説
 *
 * @subsection EnsGenerateCredential_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  EnsGenerateCredential.cpp
 *  @includelineno EnsGenerateCredential.cpp
 *
 * @subsection EnsGenerateCredential_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 * - 各種ライブラリの初期化(account, nifm, socket, curl)
 * - ユーザーの選択とオープン( SelectAndOpenUser() )
 * - ネットワーク接続( EnsureNetworkConnection() )
 * - ネットワークサービスアカウントのIDトークンの取得( GetNsaIdToken() )
 * - nn::ens の利用開始
 * - nn::ens::GenerateCredential() の利用
 * - nn::ens の利用終了
 *
 * nn::Result を返す API において、事前条件を満たしていれば必ず成功するものは
 * @ref NN_ABORT_UNLESS_RESULT_SUCCESS を利用してハンドリングしています。
 *
 * このサンプルの実行結果を以下に示します。
 * ただし、 nn::ens::UserId の値は実行するたびに異なるものが表示されます。
 *
 * @verbinclude EnsGenerateCredential_ExampleOutput.txt
 *
 */

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/account.h>
#include <nn/account/account_Result.h>
#include <nn/ens.h>
#include <nn/err.h>
#include <nn/nifm.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <nn/util/util_ScopeExit.h>
#include <curl/curl.h>

namespace
{
    // socket 用メモリインスタンス
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    // nn::ens::StartServiceLoop() を実行するスレッドには、nn::ens::RequiredStackSize 以上のスタックサイズが必要。
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStack[nn::ens::RequiredStackSize];

    // nn::ens を動かすワークメモリーは最低でも nn::ens::RequiredMemorySizeMin が必要で、
    // さらに一度のリクエストで扱う通信データサイズを加算する必要がある。
    //
    // ここでは 4MB を加算する。この場合一度に 4MB までのデータの送信、もしくは受信が行えるようになる。
    NN_ALIGNAS(4096) nn::Bit8 g_ServiceWorkMemory[nn::ens::RequiredMemorySizeMin + 4 * 1024 * 1024];
}

// nn::ens::StartServiceLoop() を別スレッドで実行するための関数
void WorkerThread(void*) NN_NOEXCEPT
{
    NN_LOG("Start Extended Network Services Loop.\n");

    // 第一引数は"利用するサーバーを示すキー文字列"です。
    // このサンプルでは空文字列を指定していますが、実際にはアプリケーションに割り振られた値を指定してください。
    // nn::ens::StartServiceLoop() はブロック関数です。 nn::ens::StopServiceLoop() の呼び出しによって抜けます。
    nn::ens::StartServiceLoop("acbaa", g_ServiceWorkMemory, sizeof(g_ServiceWorkMemory));

    NN_LOG("End Extended Network Services Loop.\n");
}

// ユーザー選択、オープンに成功すると true が返る
bool SelectAndOpenUser(nn::account::UserHandle* pOut) NN_NOEXCEPT
{
    nn::account::Uid uid;
    auto result = nn::account::ShowUserSelector(&uid);

    if (nn::account::ResultCancelledByUser::Includes(result))
    {
        NN_LOG("nn::account::ShowUserSelector() is canceled by user.\n");
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(pOut, uid));

    return true;
}

// ネットワーク利用要求が通ると true が返る
bool EnsureNetworkConnection() NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        nn::nifm::SubmitNetworkRequestAndWait();
        if (nn::nifm::IsNetworkAvailable())
        {
            return true;
        }

        auto result = nn::nifm::HandleNetworkRequestResult();
        if(result.IsSuccess())
        {
            return true;
        }
        else if (!nn::nifm::ResultErrorHandlingCompleted::Includes(result))
        {
            // エラーが解消できない
            NN_LOG("Network is not available.\n");
            return false;
        }
        // リトライでインターネット接続できる可能性がある
    }
}

// ネットワークサービスアカウントのIDトークンの取得に成功すると true が返る
bool GetNsaIdToken(char* pToken, size_t tokenSize, const nn::account::UserHandle& handle) NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        // ネットワークサービスアカウントの確定
        auto result = nn::account::EnsureNetworkServiceAccountAvailable(handle);
        if (nn::account::ResultCancelledByUser::Includes(result))
        {
            NN_LOG("nn::account::EnsureNetworkServiceAccountAvailable is canceled by user.");
            return false;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // ネットワークサービスアカウントのIDトークンのキャッシュを確保
        nn::account::AsyncContext context;
        result = nn::account::EnsureNetworkServiceAccountIdTokenCacheAsync(&context, handle);
        if(nn::account::ResultNetworkServiceAccountUnavailable::Includes(result))
        {
            // ネットワークサービスアカウントが利用可能でない(nn::account::EnsureNetworkServiceAccountAvailable()で解消可能)
            continue;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::os::SystemEvent e;
        NN_ABORT_UNLESS_RESULT_SUCCESS(context.GetSystemEvent(&e));
        e.Wait();
        result = context.GetResult();
        if(result.IsFailure())
        {
            if(nn::account::ResultNetworkServiceAccountUnavailable::Includes(result))
            {
                // ネットワークサービスアカウントが利用可能でない(nn::account::EnsureNetworkServiceAccountAvailable()で解消可能)
                continue;
            }
            else
            {
                // エラー表示が必要(ネットワーク通信に関するエラーなど)
                nn::err::ShowError(result);
                return false;
            }
        }

        // ネットワークサービスアカウントのIDトークンのキャッシュを取得
        size_t nsaIdTokenActualSize;
        result = nn::account::LoadNetworkServiceAccountIdTokenCache(
            &nsaIdTokenActualSize,
            pToken,
            tokenSize,
            handle);

        if (nn::account::ResultNetworkServiceAccountUnavailable::Includes(result) || nn::account::ResultTokenCacheUnavailable::Includes(result))
        {
            // nn::account::EnsureNetworkServiceAccountAvailable()で解消可能もしくは、
            // nn::account::EnsureNetworkServiceAccountIdTokenCacheAsync()の実行が必要。
            continue;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        return true;
    }
}

extern "C" void nnMain()
{
    // 各種ライブラリ初期化
    nn::account::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfig));
    curl_global_init(CURL_GLOBAL_DEFAULT);
    NN_UTIL_SCOPE_EXIT
    {
        curl_global_cleanup();
        nn::socket::Finalize();
    };

    // ユーザー選択、オープン
    nn::account::UserHandle userHandle;
    if(!SelectAndOpenUser(&userHandle))
    {
        return;
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::account::CloseUser(userHandle);
    };

    // ネットワーク利用要求
    if(!EnsureNetworkConnection())
    {
        return;
    }

    // ネットワークサービスアカウントのIDトークン取得
    char nsaIdToken[nn::account::NetworkServiceAccountIdTokenLengthMax];
    if(!GetNsaIdToken(nsaIdToken, sizeof(nsaIdToken), userHandle))
    {
        return;
    }

    // Extended Network Services の利用開始
    nn::os::ThreadType thread;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &thread, WorkerThread, nullptr,
        g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority));
    nn::os::StartThread(&thread);

    // nn::ens::GenerateCredential() の実行
    {
        nn::ens::AsyncContext context;
        nn::ens::Credential credential;

        nn::ens::GenerateCredential(&context, &credential, nsaIdToken);

        context.GetEvent().Wait();
        auto result = context.GetResult();

        if(result.IsSuccess())
        {
            // nn::ens::Credential には、nn::ens のネットワークサービスを利用するユーザーを一意に決める nn::ens::UserId が含まれています。
            // 同じユーザーIDに対する認証情報は変化することはありません。
            // nn::ens::Credential 構造体をセーブデータなどに永続化し、次回以降の起動へ引き継ぐことで、同じユーザーとしてサービスを利用することができます。
            NN_LOG("nn::ens::UserId = 0x%016llx\n", credential.userId.value);
        }
        else
        {
            NN_LOG("Failed to nn::ens::GenerateCredential(). (%03d-%04d)\n",
                result.GetModule(), result.GetDescription());

            // nn::ens の API 失敗は、基本的にはエラービューアでのエラー表示が必要ですが、
            // ネットワークサービスアカウントのIDトークンの失効時のエラーは、トークン取り直しによって解消することができます。
            // 現時点ではこのトークン失効のエラーをハンドリングすることはできません。将来のリリースにて改定予定です。
            nn::err::ShowError(result);
        }
    }

    // Extended Network Services の利用終了
    nn::ens::StopServiceLoop();
    nn::os::DestroyThread(&thread);
}

