﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
// This is skeleton code for service process of nns::display0 driver.
//-----------------------------------------------------------------------------

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/sf/sf_HipcServer.h>

#include <nns/display0.h>

//-----------------------------------------------------------------------------
// nninitStartup() is invoked before calling nnMain().
//
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( 0 );
}




//-----------------------------------------------------------------------------

namespace {

//-----------------------------------------------------------------------------
// nn::sf::HipcSimpleAllInOneServerManager は HIPC のサーバのポートと
// セッションを一元管理する。引数は最大セッション数と最大ポート数。
//
class Display0ServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<30, 1>
{
private:
    // OnNeedsToAccept を override し、ポートへのアクセスの動作を決定する。
    // 詳細は OnNeedsToAccept() のリファレンスを参照。
    virtual nn::Result OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(portIndex);
        // ポートから開かれたセッションに、
        // サーバ内における DFC 実装（ CreateDisplay0DriverByDfc() ）を紐付ける。
        return AcceptImpl(pPort, nns::display0::CreateDisplay0DriverByDfc());
    }
};

// 繰返しのサーバ起動／終了が可能となるように placement new で初期化を行う。
// 繰返しの起動と終了が必要ない場合には Display0ServerManager は直接配置してもよい。
std::aligned_storage<sizeof(Display0ServerManager), NN_ALIGNOF(Display0ServerManager)>::type g_Display0ServerManagerStorage;

//-----------------------------------------------------------------------------
// サーバ実装用のサービスオブジェクトへの共有ポインタ
Display0ServerManager*  g_pDisplay0ServerManager;



//-----------------------------------------------------------------------------
//  グラフィックドライバサーバーを初期化する
//
void InitializeDisplay0DriverServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_pDisplay0ServerManager);

    // Display0ServerManager オブジェクトのコンストラクト（placement new）
    g_pDisplay0ServerManager = new (&g_Display0ServerManagerStorage) Display0ServerManager;

    // サービス名の登録とポートの初期化
    // portIndex は OnNeedsToAccept の第一引数となるが、
    // 今回は使用しないため、適当に 0 を指定している。
    // sessionCountMax は、現時点では十分に大きな値を指定しておけばよい。
    auto portIndex       = 0;
    auto sessionCountMax = 30;
    g_pDisplay0ServerManager->InitializePort(portIndex, sessionCountMax, nns::display0::Display0DriverServiceName);

    // サーバマネージャの開始
    // ただし、実際のサーバ動作は LoopAuto() 関数の呼び出しによって行なわれる。
    g_pDisplay0ServerManager->Start();
}


//-----------------------------------------------------------------------------
//  グラフィックドライバサーバーを終了する
//
void FinalizeDisplay0DriverServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pDisplay0ServerManager);

    // placement new でコンストラクトされているので明示的にデストラクタを呼ぶ
    g_pDisplay0ServerManager->~Display0ServerManager();
    g_pDisplay0ServerManager = nullptr;
}


//-----------------------------------------------------------------------------
//  グラフィックドライバのサーバメッセージ処理をループで行なう
//
void LoopDisplay0DriverServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pDisplay0ServerManager);
    g_pDisplay0ServerManager->LoopAuto();
}


//-----------------------------------------------------------------------------
//  グラフィックドライバのサーバメッセージ処理ループを停止する要求を送出する
//
void RequestStopDisplay0DriverServer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_pDisplay0ServerManager);
    g_pDisplay0ServerManager->RequestStop();
}

//-----------------------------------------------------------------------------

}   // namespace


//-----------------------------------------------------------------------------

extern "C" void nnMain()
{
    NN_UNUSED( RequestStopDisplay0DriverServer );

    // Start Display0 service
    InitializeDisplay0DriverServer();

    {
        NN_SDK_LOG("[display0] Server process is running.\n");

        // This function continues until invoking StopDevice() method.
        LoopDisplay0DriverServer();

        NN_SDK_LOG("[display0] Server process is finished.\n");
    }

    // Stop Display0 service
    FinalizeDisplay0DriverServer();
}

//-----------------------------------------------------------------------------

