﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/ovln/ovln_CommonTypes.h>
#include <nn/ovln/ovln_FormatForOverlay.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <cstring>

void PrintBatteryLevel(nn::ovln::format::ControllerPowerInfo& battery)
{
    if (battery.isValid == true)
    {
        NN_LOG("       Battery Level ");
        switch (battery.powerInfo.batteryLevel)
        {
        case nn::hid::system::NpadBatteryLevel_High:
            NN_LOG("HIGH\n");
            break;
        case nn::hid::system::NpadBatteryLevel_Midium:
            NN_LOG("MIDIUM\n");
            break;
        case nn::hid::system::NpadBatteryLevel_Low:
            NN_LOG("LOW\n");
            break;
        case nn::hid::system::NpadBatteryLevel_CriticalLow:
            NN_LOG("CRITICAL LOW\n");
            break;
        case nn::hid::system::NpadBatteryLevel_None:
            NN_LOG("NONE\n");
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        if (battery.powerInfo.isCharging)
        {
            NN_LOG("       Battery CHARGING\n");
        }
        else
        {
            NN_LOG("       Battery NOT CHARGING\n");

        }
        if (battery.powerInfo.isPowered)
        {
            NN_LOG("       Battery CHARGING\n");
        }
        else
        {
            NN_LOG("       Battery NOT CHARGING\n");

        }
    }
    else
    {
        NN_LOG("       Battery Information Invalid\n");
    }
}

void PrintColor(nn::ovln::format::ControllerColorInfo& color)
{
    if (color.isValid == true)
    {
        NN_LOG("       Main Color %03d %03d %03d\n", color.color.main.v[0], color.color.main.v[1], color.color.main.v[2]);
        NN_LOG("       Sub  Color %03d %03d %03d\n", color.color.sub.v[0],  color.color.sub.v[1],  color.color.sub.v[2]);
    }
    else
    {
        NN_LOG("       Color Information Invalid\n");
    }
}

void PrintDeviceType(nn::hid::system::NpadDeviceTypeSet deviceType)
{
    // デバイスタイプ毎の出力
    if (deviceType.Test<nn::hid::system::NpadDeviceType::FullKeyController>())
    {
        // Switch Proコントローラー
        NN_LOG("    -> FullKeyController\n");
    }
    if (deviceType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>())
    {
        // Joy-Con (L)
        NN_LOG("    -> JoyLeft\n");
    }
    if (deviceType.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
    {
        // Joy-Con (R)
        NN_LOG("    -> JoyRight\n");
    }
    if (deviceType.Test<nn::hid::system::NpadDeviceType::HandheldJoyLeft>())
    {
        // Joy-Con (L)  携帯機状態
        NN_LOG("    -> Handheld JoyLeft\n");
    }
    if (deviceType.Test<nn::hid::system::NpadDeviceType::HandheldJoyRight>())
    {
        // Joy-Con (R)  携帯機状態
        NN_LOG("    -> Handheld JoyRight\n");
    }
}

void PrintControllerInfo(nn::ovln::format::ControllerInfo controllerInfo)
{
    // デバイスタイプ毎の出力
    if (controllerInfo.deviceType.Test<nn::hid::system::NpadDeviceType::FullKeyController>())
    {
        // Switch Proコントローラー
        NN_LOG("    -> FullKeyController\n");
    }
    else if (controllerInfo.deviceType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>() &&
        controllerInfo.deviceType.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
    {
        // Joy-Con 2本持ち
        NN_LOG("    -> Joy Dual");
        if (controllerInfo.joyType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>() &&
            controllerInfo.joyType.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
        {
            NN_LOG(" (Left and Right)\n");
        }
        else if (controllerInfo.joyType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>())
        {
            NN_LOG(" (Left only)\n");
        }
        else if (controllerInfo.joyType.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
        {
            NN_LOG(" (Right only)\n");
        }
    }
    else if (controllerInfo.deviceType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>())
    {
        // Joy-Con (L)
        NN_LOG("    -> Joy-Con (L)");
        if (controllerInfo.GetJoyHoldType() == nn::hid::NpadJoyHoldType_Vertical)
        {
            NN_LOG(" (Vertical)\n");
        }
        else if (controllerInfo.GetJoyHoldType() == nn::hid::NpadJoyHoldType_Vertical)
        {
            NN_LOG(" (Horizontal)\n");
        }
    }
    else if (controllerInfo.deviceType.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
    {
        // Joy-Con (L)
        NN_LOG("    -> Joy-Con (R)");
        if (controllerInfo.GetJoyHoldType() == nn::hid::NpadJoyHoldType_Vertical)
        {
            NN_LOG(" (Vertical)\n");
        }
        else if (controllerInfo.GetJoyHoldType() == nn::hid::NpadJoyHoldType_Vertical)
        {
            NN_LOG(" (Horizontal)\n");
        }
    }
    else if (controllerInfo.deviceType.Test<nn::hid::system::NpadDeviceType::Handheld>())
    {
        // 携帯機コントローラー
        NN_LOG("    -> Handheld");
        if (controllerInfo.joyType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>() &&
            controllerInfo.joyType.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
        {
            NN_LOG(" (Left and Right)\n");
        }
        else if (controllerInfo.joyType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>())
        {
            NN_LOG(" (Left only)\n");
        }
        else if (controllerInfo.joyType.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
        {
            NN_LOG(" (Right only)\n");
        }
    }
    else if (controllerInfo.deviceType.Test<nn::hid::system::NpadDeviceType::Unknown>())
    {
        // 未知のデバイス (将来追加されたデバイスなど)
        NN_LOG("    -> Unknown");
    }

    // インジケーターの表示
    NN_LOG("    -> Indicator %c%c%c%c\n", (controllerInfo.ledPattern & 0x01) ? 'o' : '_',
        (controllerInfo.ledPattern & 0x02) ? 'o' : '_',
        (controllerInfo.ledPattern & 0x04) ? 'o' : '_',
        (controllerInfo.ledPattern & 0x08) ? 'o' : '_');

    // 電池残量/色情報の表示
    if (controllerInfo.deviceType.Test<nn::hid::system::NpadDeviceType::FullKeyController>())
    {
        NN_LOG("    -> FullKey Battery Info\n");
        PrintBatteryLevel(controllerInfo.fullKeyBattery);
        NN_LOG("    -> FullKey Color Info\n");
        PrintColor(controllerInfo.fullKeyColor);
    }
    else
    {
        NN_LOG("    -> JoyLeft Battery Info\n");
        PrintBatteryLevel(controllerInfo.joyBattery.left);
        NN_LOG("    -> JoyLeft Color Info\n");
        PrintColor(controllerInfo.joyColor.left);
        NN_LOG("    -> JoyRight Battery Info\n");
        PrintBatteryLevel(controllerInfo.joyBattery.right);
        NN_LOG("    -> JoyRight Color Info\n");
        PrintColor(controllerInfo.joyColor.right);
    }
}

void DispatchControllerMessage(const nn::ovln::Message& message) NN_NOEXCEPT
{
    switch (message.tag)
    {
        case nn::ovln::format::ControllerAttachToConsoleDataTag:
        {
            nn::ovln::format::ControllerAttachToConsoleData data;
            if (message.dataSize == sizeof(data))
            {
                std::memcpy(&data, &message.data, sizeof(data));
                NN_LOG("[DispatchControllerMessage] Attach to console message:\n");
                switch (data.eventType)
                {
                case nn::ovln::format::ControllerAttachEventType_ConsoleRailAttached:
                    NN_LOG("    -> Attached to Rail\n");
                    break;
                case nn::ovln::format::ControllerAttachEventType_ConsoleRailDetached:
                    NN_LOG("    -> Detached from Rail\n");
                    break;
                case nn::ovln::format::ControllerAttachEventType_ExternalGripAttached:
                    NN_LOG("    -> Attached to Grip\n");
                    break;
                case nn::ovln::format::ControllerAttachEventType_ExternalGripDetached:
                    NN_LOG("    -> Detached from Grip\n");
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }

                PrintDeviceType(data.deviceType);
                NN_LOG("    -> Color");
                PrintColor(data.color);
            }
            break;
        }
        case nn::ovln::format::ControllerPairingCompletedDataTag:
        {
            nn::ovln::format::ControllerPairingCompletedData data;
            if (message.dataSize == sizeof(data))
            {
                std::memcpy(&data, &message.data, sizeof(data));
                NN_LOG("[DispatchControllerMessage] PairingCompleted:\n");
                switch (data.pairingMethod)
                {
                case nn::ovln::format::ControllerPairingMethod_Button:
                    NN_LOG("    -> Button Pairing\n");
                    break;
                case nn::ovln::format::ControllerPairingMethod_Joint:
                    NN_LOG("    -> Rail Pairing\n");
                    break;
                case nn::ovln::format::ControllerPairingMethod_USB:
                    NN_LOG("    -> USB Pairing\n");
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }

                PrintDeviceType(data.deviceType);
                NN_LOG("    -> Color");
                PrintColor(data.color);
            }
            break;
        }
        case nn::ovln::format::ControllerBatteryLevelLowExDataTag:
        {
            nn::ovln::format::ControllerBatteryLevelLowExData data;
            if (message.dataSize == sizeof(data))
            {
                std::memcpy(&data, &message.data, sizeof(data));
                NN_LOG("[DispatchControllerMessage] Battery level low message:\n");
                PrintControllerInfo(data.controllerInfo);
                NN_LOG("    -> NpadId: %d\n", data.npadId);
            }
            break;
        }
        case nn::ovln::format::ControllerDisconnectionDataTag:
        {
            nn::ovln::format::ControllerDisconnectionData data;
            if (message.dataSize == sizeof(data))
            {
                std::memcpy(&data, &message.data, sizeof(data));
                NN_LOG("[DispatchControllerMessage] Disconnection message:\n");
                switch (data.reason)
                {
                case nn::ovln::format::ControllerDisconnectionReason_Battery:
                    NN_LOG("    -> Battery Level Low\n");
                    break;
                case nn::ovln::format::ControllerDisconnectionReason_Unknown:
                    NN_LOG("    -> Unknown Reason\n");
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
                PrintControllerInfo(data.controllerInfo);
            }
            break;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

