﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/fs.h>
#include <nn/gfx.h>
#include <nn/init.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/oe.h>
#include <nn/os.h>
#include <nn/vi.h>

#include <nn/lmem/lmem_ExpHeap.h>

#include <nv/nv_MemoryManagement.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#endif

#include "Demo1.h"
#include "Demo1Gfx.h"
#include "Demo1GraphicsSystem.h"
#include "Demo1PluginManager.h"

namespace {

void* Allocate( size_t size, size_t alignment, void* pUserData) NN_NOEXCEPT
{
    return GetStandardAllocator().Allocate(size, alignment);
}

void Free( void* addr, void* pUserData) NN_NOEXCEPT
{
    GetStandardAllocator().Free( addr );
}

void* Reallocate( void* addr, size_t newSize, void* pUserData) NN_NOEXCEPT
{
    return GetStandardAllocator().Reallocate( addr, newSize );
}

void ProcessMessage() NN_NOEXCEPT
{
    nn::oe::Message message;
    bool isMessage = nn::oe::TryPopNotificationMessage(&message);
    if (isMessage)
    {
        switch (message)
        {
        case nn::oe::MessageOperationModeChanged:
            // 動作モード（携帯／据置）が変更されたことを示す通知です。
            // nn::oe::GetOperationMode() により現在の動作モードを取得することができます。
            NN_LOG("nn::oe::MessageOperationModeChanged\n");
            break;
        case nn::oe::MessagePerformanceModeChanged:
            // 性能モード（ノーマル／ブースト）が変更されたことを示す通知です。
            // nn::oe::GetPerformanceMode() により現在の性能モードを取得することができます。
            NN_LOG("nn::oe::MessagePerformanceModeChanged\n");
            break;
        case nn::oe::MessageResume:
            // スリープから復帰したことを示す通知です。
            NN_LOG("nn::oe::MessageResume\n");
            break;
        default:
            // 未知のメッセージは無視するようにします。
            NN_LOG("Unhandled message = 0x%08x\n", message);
            break;
        }
    }
}
}

class PlatformDependentForNx : public PlatformDependentPluginBase
{
    NN_DISALLOW_COPY(PlatformDependentForNx);
    NN_DISALLOW_MOVE(PlatformDependentForNx);

public:
    PlatformDependentForNx() NN_NOEXCEPT {}

    virtual ~PlatformDependentForNx() NN_NOEXCEPT  {}

    virtual void Initialize() NN_NOEXCEPT
    {
        // グラフィックスシステムのためのメモリ周りの初期化を行います。
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator( Allocate, Free, Reallocate, NULL );
        nv::InitializeGraphics(GetStandardAllocator().Allocate(GraphicsSystemMemorySize, 1),
            GraphicsSystemMemorySize );

        // グラフィックス開発者向けツールおよびデバッグレイヤのためのメモリアロケータを設定します。
        nv::SetGraphicsDevtoolsAllocator( Allocate, Free, Reallocate, NULL );
        // GLSLC のためのメモリアロケータを設定します。
        glslcSetAllocator( Allocate, Free, Reallocate, NULL );

        // アプリケーション動作環境操作ライブラリを初期化。
        nn::oe::Initialize();

        // ノーマルモード時の性能を選択
        nn::oe::SetPerformanceConfiguration(nn::oe::PerformanceMode_Normal,
            nn::oe::PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1331Mhz);

        // ブーストモード時の性能を選択
        nn::oe::SetPerformanceConfiguration(nn::oe::PerformanceMode_Boost,
            nn::oe::PerformanceConfiguration_Cpu1020MhzGpu768MhzEmc1600Mhz);

        // BG フォーカス状態になってもプログラムを中断しないように設定
        nn::oe::SetFocusHandlingMode(nn::oe::FocusHandlingMode_Notify);

        // 動作モード変更メッセージを通知するように設定
        nn::oe::SetOperationModeChangedNotificationEnabled(true);

        // 性能モード変更メッセージを通知するように設定
        nn::oe::SetPerformanceModeChangedNotificationEnabled(true);

        // 一時中断状態から動作を再開したことを示すメッセージを通知するように設定
        nn::oe::SetResumeNotificationEnabled(true);

        return;
    }

    virtual void Finalize() NN_NOEXCEPT {}

    virtual void Procedure() NN_NOEXCEPT
    {
        ProcessMessage();
    }
};

SET_PLUGIN( "PlatformDependentForNx", PlatformDependentForNx, PluginProperty_None );

