﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <sstream>

#include <nn/nn_Assert.h>

#include "Demo1IrSensorModeState.h"

namespace
{

const char* AmbientNoiseLevel[] =
{
    "Low",
    "Middle",
    "High",
    "Unknown",
};

}

MenuItem::MenuItem(const char* pName, ReadWriteBase* pReadWrite) NN_NOEXCEPT
    : m_Name(pName)
    , m_pMenuItemAccessor(pReadWrite)
{
}

void MenuItem::Increment(int8_t delta) const NN_NOEXCEPT
{
    (*m_pMenuItemAccessor)(delta);
}

void MenuItem::Read(std::stringstream& sstr) const NN_NOEXCEPT
{
    (*m_pMenuItemAccessor)(sstr);
}

const char* MenuItem::GetName() const NN_NOEXCEPT
{
    return m_Name.c_str();
}

class CurrentMode : public ReadWriteBase
{
public:
    CurrentMode(IrSensorMode* pNextProcessor, int* pMenuSelection) NN_NOEXCEPT
        : m_pNextProcessor(pNextProcessor)
        , m_pMenuSelection(pMenuSelection)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        switch (*m_pNextProcessor)
        {
        case IrSensorMode_Moment:
            sstr << "Moment";
            break;
        case IrSensorMode_Clustering:
            sstr << "Clustering";
            break;
        case IrSensorMode_ImageTransfer:
            sstr << "Image Transfer";
            break;
        case IrSensorMode_HandAnalysis:
            sstr << "Hand Analysis";
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        int8_t mode = static_cast<int8_t>(*m_pNextProcessor) + delta;
        *m_pNextProcessor = static_cast<IrSensorMode>(
            std::min(
                std::max(mode, static_cast<int8_t>(IrSensorMode_Moment)),
                static_cast<int8_t>(IrSensorMode_Count - 1)
            )
        );
        *m_pMenuSelection = 0;
    }

private:
    IrSensorMode* m_pNextProcessor;
    int* m_pMenuSelection;
};


class ExposureTime : public ReadWriteBase
{
public:
    ExposureTime(
        ::nn::irsensor::IrCameraConfig* pConfig,
        ::nn::TimeSpanType exposureMin,
        ::nn::TimeSpanType exposureMax) NN_NOEXCEPT
        : m_pConfig(pConfig)
        , m_ExposureMin(exposureMin)
        , m_ExposureMax(exposureMax)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << m_pConfig->exposureTime.GetMicroSeconds();
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        m_pConfig->exposureTime += ::nn::TimeSpanType::FromMicroSeconds(delta * 10);
        m_pConfig->exposureTime = std::min(
            std::max(m_pConfig->exposureTime, m_ExposureMin),
            m_ExposureMax
        );
    }

private:
    ::nn::irsensor::IrCameraConfig* m_pConfig;
    ::nn::TimeSpanType m_ExposureMin;
    ::nn::TimeSpanType m_ExposureMax;
};

class LightTarget : public ReadWriteBase
{
public:
    LightTarget(
        ::nn::irsensor::IrCameraConfig* pConfig) NN_NOEXCEPT
        : m_pConfig(pConfig)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        switch (m_pConfig->lightTarget)
        {
        case nn::irsensor::IrCameraLightTarget_AllObjects:
            sstr << "AllObjects";
            break;
        case nn::irsensor::IrCameraLightTarget_FarObjects:
            sstr << "FarObjects";
            break;
        case nn::irsensor::IrCameraLightTarget_NearObjects:
            sstr << "NearObjects";
            break;
        case nn::irsensor::IrCameraLightTarget_None:
            sstr << "None";
            break;
        default:
            break;
        }
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        int8_t lightTarget = static_cast<int8_t>(m_pConfig->lightTarget) +
            static_cast<int8_t>(delta);
        lightTarget = std::max(
            std::min(lightTarget, static_cast<int8_t>(nn::irsensor::IrCameraLightTarget_None)),
            static_cast<int8_t>(nn::irsensor::IrCameraLightTarget_AllObjects)
        );
        m_pConfig->lightTarget = static_cast<nn::irsensor::IrCameraLightTarget>(lightTarget);
    }

private:
    ::nn::irsensor::IrCameraConfig* m_pConfig;
};

class DigitalGain : public ReadWriteBase
{
public:
    DigitalGain(
        ::nn::irsensor::IrCameraConfig* pConfig) NN_NOEXCEPT
        : m_pConfig(pConfig)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << m_pConfig->gain;
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        m_pConfig->gain += delta;
        m_pConfig->gain = std::min(
            std::max(m_pConfig->gain, nn::irsensor::IrCameraGainMin),
            nn::irsensor::IrCameraGainMax
        );
    }

private:
    ::nn::irsensor::IrCameraConfig* m_pConfig;
};

class NegativeImage : public ReadWriteBase
{
public:
    NegativeImage(
        ::nn::irsensor::IrCameraConfig* pConfig) NN_NOEXCEPT
        : m_pConfig(pConfig)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << (m_pConfig->isNegativeImageUsed ? "ON" : "OFF");
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
        m_pConfig->isNegativeImageUsed = !m_pConfig->isNegativeImageUsed;
    }

private:
    ::nn::irsensor::IrCameraConfig* m_pConfig;
};

IrSensorModeState::IrSensorModeState(
    IrSensorMode* pNextProcessor,
    int* pMenuSelection,
    nn::irsensor::IrCameraHandle irCameraHandle) NN_NOEXCEPT
    : m_IrCameraHandle(irCameraHandle)
    , m_pCurrentMode(NULL)
    , m_pLightTarget(NULL)
    , m_pDigitalGain(NULL)
    , m_pNegativeImage(NULL)
    , m_pSamplingNumber1(NULL)
    , m_pSamplingNumber2(NULL)
{
    m_pCurrentMode = new CurrentMode(pNextProcessor, pMenuSelection);
    MenuItem currentModeMenuItem("Current Mode",
        m_pCurrentMode
    );
    m_ReadWriteMenu.push_back(currentModeMenuItem);
}

IrSensorModeState::~IrSensorModeState() NN_NOEXCEPT
{
    delete m_pCurrentMode;
    delete m_pLightTarget;
    delete m_pDigitalGain;
    delete m_pNegativeImage;
    delete m_pSamplingNumber1;
    delete m_pSamplingNumber2;
}

void IrSensorModeState::AddCommonReadWriteMenu(
    std::vector<MenuItem>* pMenu,
    ::nn::irsensor::IrCameraConfig* pConfig,
    ::nn::TimeSpanType exposureMin, ::nn::TimeSpanType exposureMax) NN_NOEXCEPT
{
    m_pExposureTime = new ExposureTime(pConfig, exposureMin, exposureMax);
    MenuItem exposureTimeMenuItem("Exposure Time",
        m_pExposureTime
    );
    pMenu->push_back(exposureTimeMenuItem);

    m_pLightTarget = new LightTarget(pConfig);
    MenuItem lightTargetMenuItem("Light Target",
        m_pLightTarget
    );
    pMenu->push_back(lightTargetMenuItem);

    m_pDigitalGain = new DigitalGain(pConfig);
    MenuItem digitalGainMenuItem("Digital Gain",
        m_pDigitalGain
    );
    pMenu->push_back(digitalGainMenuItem);

    m_pNegativeImage = new NegativeImage(pConfig);
    MenuItem negativeImageMenuItem("Negative Image",
        m_pNegativeImage
    );
    pMenu->push_back(negativeImageMenuItem);
}


class SamplingNumber1 : public ReadWriteBase
{
public:
    SamplingNumber1(
        int64_t* pSamplingNumber) NN_NOEXCEPT
        : m_pSamplingNumber(pSamplingNumber)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << *m_pSamplingNumber;
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
    }

private:
    int64_t* m_pSamplingNumber;
};

class SamplingNumber2 : public ReadWriteBase
{
public:
    SamplingNumber2(
        nn::irsensor::IrCameraAmbientNoiseLevel* pAmbientNoiseLevel) NN_NOEXCEPT
        : m_pAmbientNoiseLevel(pAmbientNoiseLevel)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << AmbientNoiseLevel[*m_pAmbientNoiseLevel];
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
    }

private:
    nn::irsensor::IrCameraAmbientNoiseLevel* m_pAmbientNoiseLevel;
};

void IrSensorModeState::AddCommonReadOnlyMenu(
    std::vector<MenuItem>* pMenu,
    int64_t* pSamplingNumber,
    nn::irsensor::IrCameraAmbientNoiseLevel* pAmbientNoiseLevel) NN_NOEXCEPT
{
    m_pSamplingNumber1 =new SamplingNumber1(pSamplingNumber);
    MenuItem samplingNumber1MenuItem("Sampling Number",
        m_pSamplingNumber1
    );
    pMenu->push_back(samplingNumber1MenuItem);

    m_pSamplingNumber2 = new SamplingNumber2(pAmbientNoiseLevel);
    MenuItem samplingNumber2MenuItem("Sampling Number",
        m_pSamplingNumber2
    );
    pMenu->push_back(samplingNumber2MenuItem);
}

const std::vector<MenuItem>& IrSensorModeState::GetReadWriteMenu() const NN_NOEXCEPT
{
    return m_ReadWriteMenu;
}

const std::vector<MenuItem>& IrSensorModeState::GetReadOnlyMenu() const NN_NOEXCEPT
{
    return m_ReadOnlyMenu;
}

void IrSensorModeState::Stop() const NN_NOEXCEPT
{
    nn::irsensor::StopImageProcessor(m_IrCameraHandle);
}
