﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <sstream>

#include <nn/nn_Assert.h>
#include <nn/os.h>

#include "Demo1Color.h"
#include "Demo1ImageTransferModeState.h"
#include "Demo1ScenePluginBase.h"

namespace
{
NN_STATIC_ASSERT(nn::irsensor::ImageTransferProcessorFormat_320x240 == 0);
NN_STATIC_ASSERT(nn::irsensor::ImageTransferProcessorFormat_160x120 == 1);
NN_STATIC_ASSERT(nn::irsensor::ImageTransferProcessorFormat_80x60 == 2);
}

class Format : public ReadWriteBase
{
public:
    NN_IMPLICIT Format(
        nn::irsensor::ImageTransferProcessorConfig* pImageTransferConfig,
        ImageProperties* pProperties[3]
        ) NN_NOEXCEPT
        : m_pImageTransferConfig(pImageTransferConfig)
    {
        for (int i = 0; i < NN_ARRAY_SIZE(m_pProperties); i++)
        {
            m_pProperties[i] = pProperties[i];
        }
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << m_pProperties[m_pImageTransferConfig->format]->GetName();
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        int8_t format = static_cast<int8_t>(m_pImageTransferConfig->format) +
            static_cast<int8_t>(delta);
        format = std::max(
            std::min(
                format,
                static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_80x60)
            ),
            static_cast<int8_t>(nn::irsensor::ImageTransferProcessorFormat_320x240)
        );
        m_pImageTransferConfig->format = static_cast<nn::irsensor::ImageTransferProcessorFormat>(
            format
            );
    }

private:
    nn::irsensor::ImageTransferProcessorConfig* m_pImageTransferConfig;
    ImageProperties* m_pProperties[3];
};

ImageTransferModeState::ImageTransferModeState(
    IrSensorMode* pNextProcessor,
    int* pMenuSelection,
    nn::irsensor::IrCameraHandle irCameraHandle,
    void* pWorkMemory,
    GraphicsSystem *pGraphicsSystem) NN_NOEXCEPT
    : IrSensorModeState(pNextProcessor, pMenuSelection, irCameraHandle)
    , m_Format(nn::irsensor::ImageTransferProcessorFormat_320x240)
    , m_ImageTransferProcessorState()
    , m_ImageTransferProcessorConfig()
    , m_pImageTransferWorkBuffer(pWorkMemory)
    , m_pImageCopyWorkBuffer(
        static_cast<uint8_t*>(pWorkMemory) + nn::irsensor::ImageTransferProcessorWorkBufferSize320x240
    )
    , m_pFormat(NULL)
    , m_pGraphicsSystem(pGraphicsSystem)
{
    {
        nn::irsensor::GetImageTransferProcessorDefaultConfig(&m_ImageTransferProcessorConfig);

        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_Read);
        info.SetSize(nn::irsensor::ImageTransferProcessorImageSize320x240);

        pGraphicsSystem->AllocateBuffer(&m_Buffer, &info);

        m_pProperties[0] = new ImageProperties(
            nn::irsensor::IrCameraImageWidth,
            nn::irsensor::IrCameraImageHeight,
            nn::irsensor::ImageTransferProcessorImageSize320x240,
            nn::irsensor::ImageTransferProcessorWorkBufferSize320x240,
            "320x240",
            4000
        );
        m_pProperties[1] = new ImageProperties(
            nn::irsensor::IrCameraImageWidth >> nn::irsensor::ImageTransferProcessorFormat_160x120,
            nn::irsensor::IrCameraImageHeight >> nn::irsensor::ImageTransferProcessorFormat_160x120,
            nn::irsensor::ImageTransferProcessorImageSize160x120,
            nn::irsensor::ImageTransferProcessorWorkBufferSize160x120,
            "160x120",
            1000
        );
        m_pProperties[2] = new ImageProperties(
            nn::irsensor::IrCameraImageWidth >> nn::irsensor::ImageTransferProcessorFormat_80x60,
            nn::irsensor::IrCameraImageHeight >> nn::irsensor::ImageTransferProcessorFormat_80x60,
            nn::irsensor::ImageTransferProcessorImageSize80x60,
            nn::irsensor::ImageTransferProcessorWorkBufferSize80x60,
            "80x60",
            250
        );

        for (int i = 0; i < NN_ARRAY_SIZE(m_Textures); i++)
        {
            CreateTexture(
                pGraphicsSystem,
                m_pProperties[i]->GetWidth(),
                m_pProperties[i]->GetHeight(),
                nn::gfx::ImageFormat_R8_Unorm, &m_Textures[i]
            );
        }

        CreateSampler(
            pGraphicsSystem,
            &m_Sampler
        );
    }

    nn::irsensor::ImageTransferProcessorConfig* pImageTransferConfig =
        &m_ImageTransferProcessorConfig;
    AddCommonReadWriteMenu(&m_ReadWriteMenu,
        &pImageTransferConfig->irCameraConfig,
        nn::irsensor::ImageTransferProcessorExposureTimeMin,
        nn::irsensor::ImageTransferProcessorExposureTimeMax
    );

    m_pFormat = new Format(pImageTransferConfig, m_pProperties);
    MenuItem formatMenuItem("Format",
        m_pFormat
    );
    m_ReadWriteMenu.push_back(formatMenuItem);

    AddCommonReadOnlyMenu(
        &m_ReadOnlyMenu,
        &m_ImageTransferProcessorState.samplingNumber,
        &m_ImageTransferProcessorState.ambientNoiseLevel
    );

}

ImageTransferModeState::~ImageTransferModeState() NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_pProperties); i++)
    {
        delete m_pProperties[i];
        m_pProperties[i] = NULL;
    }

    m_pGraphicsSystem->FreeBuffer(&m_Buffer);
    for (int i = 0; i < NN_ARRAY_SIZE(m_Textures); i++)
    {
        m_pGraphicsSystem->FreeTexture(&m_Textures[i].GetTexture());
        m_Textures[i].GetTextureView().Finalize(&m_pGraphicsSystem->GetDevice());
    }

    m_Sampler.GetSampler().Finalize(&m_pGraphicsSystem->GetDevice());
    delete m_pFormat;
}

void ImageTransferModeState::Start() NN_NOEXCEPT
{
    nn::irsensor::RunImageTransferProcessor(
        m_IrCameraHandle,
        m_ImageTransferProcessorConfig,
        static_cast<void*>(m_pImageTransferWorkBuffer),
        m_pProperties[m_ImageTransferProcessorConfig.format]->GetWorkBufferSize()
    );
    m_Format = m_ImageTransferProcessorConfig.format;
}

namespace
{

void CopyBufferToImage(
    nn::gfx::Texture* pTexture,
    const nn::gfx::Buffer* pBuffer,
    int width,
    int height,
    nn::gfx::CommandBuffer* pCommandBuffer)
{
    NN_ASSERT_NOT_NULL(pTexture);
    NN_ASSERT_NOT_NULL(pBuffer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    nn::gfx::TextureCopyRegion dstRegion;
    dstRegion.SetDefault();
    dstRegion.SetDepth(1);

    dstRegion.SetWidth(width);
    dstRegion.SetHeight(height);
    pCommandBuffer->CopyBufferToImage(pTexture, dstRegion, pBuffer, 0);
}

}

void ImageTransferModeState::GetBufferToImage(const nn::gfx::Buffer* pBuffer,
    void* copyBuffer,
    nn::irsensor::ImageTransferProcessorFormat format,
    nn::irsensor::ImageTransferProcessorState* pImageTransferProcessorState,
    const nn::irsensor::IrCameraHandle& handle) const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pBuffer);

    static nn::os::Tick s_PrevTick = nn::os::GetSystemTick();
    nn::os::Tick currentTick = nn::os::GetSystemTick();

    int64_t waitMilliSeconds = m_pProperties[format]->GetWaitTimeMilli();
    size_t writeSize = m_pProperties[format]->GetSize();


    // 理想時間だけ待つ
    if (nn::os::ConvertToTimeSpan(currentTick - s_PrevTick).GetMilliSeconds() > waitMilliSeconds)
    {
        nn::irsensor::GetImageTransferProcessorState(
            pImageTransferProcessorState,
            copyBuffer,
            m_pProperties[format]->GetSize(), handle
        );
        s_PrevTick = nn::os::GetSystemTick();

        uint8_t* sourcePixels = pBuffer->Map<uint8_t>();
        memcpy(sourcePixels, copyBuffer, m_pProperties[format]->GetSize());
        pBuffer->FlushMappedRange(0, writeSize);
        pBuffer->Unmap();
    }
}

void ImageTransferModeState::CopyImageBuffer(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    CopyBufferToImage(&m_Textures[m_Format].GetTexture(), &m_Buffer,
        m_pProperties[m_Format]->GetWidth(), m_pProperties[m_Format]->GetHeight(),
        pCommandBuffer);
}

void ImageTransferModeState::Update() NN_NOEXCEPT
{
    GetBufferToImage(
        &m_Buffer,
        m_pImageCopyWorkBuffer,
        m_Format,
        &m_ImageTransferProcessorState, m_IrCameraHandle
    );
}

void ImageTransferModeState::RenderImageTransferProcessorState(
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer,
    const int screenIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // DepthStencilTypeをDepthStencilType_DepthNoWriteTestへ設定
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer,
        nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType corner;
    nn::util::Vector3fType size;

    float xStart;
    float xEnd;
    float yStart;
    float yEnd;
    if (screenIndex == 0)
    {
        xStart = 0.45f, xEnd = 0.0f;
        yStart = 0.2f, yEnd = 0.8f;
    }
    else
    {
        xStart = 0.95f, xEnd = 0.5f;
        yStart = 0.2f, yEnd = 0.8f;
    }

    nn::util::VectorSet(&corner, xStart, yStart, 0.0f);
    nn::util::VectorSet(&size, xEnd - xStart, yEnd - yStart, 0.0f);

    pPrimitiveRenderer->SetColor(Color::White);
    pPrimitiveRenderer->DrawQuad(
        pCommandBuffer,
        corner,
        size,
        m_Textures[m_Format].GetDescriptorSlot(),
        m_Sampler.GetDescriptorSlot());
}


void ImageTransferModeState::Render(
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer,
    int index) NN_NOEXCEPT
{
    CopyImageBuffer(pCommandBuffer);
    RenderImageTransferProcessorState(pPrimitiveRenderer, pCommandBuffer, index);
}
