﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <sstream>

#include <nn/nn_Assert.h>

#include "Demo1Color.h"
#include "Demo1HandAnalysisModeState.h"
#include "Demo1ScenePluginBase.h"

class Mode : public ReadWriteBase
{
public:
    NN_IMPLICIT Mode(nn::irsensor::HandAnalysisConfig* pHandAnalysisConfig) NN_NOEXCEPT
        : m_pHandAnalysisConfig(pHandAnalysisConfig)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_pHandAnalysisConfig->mode == nn::irsensor::HandAnalysisMode_Silhouette)
        {
            sstr << "Silhouette";
        }
        else if (m_pHandAnalysisConfig->mode == nn::irsensor::HandAnalysisMode_Image)
        {
            sstr << "Image";
        }
        else if (m_pHandAnalysisConfig->mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage)
        {
            sstr << "Silhouette and Image";
        }
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        int8_t mode = static_cast<int8_t>(m_pHandAnalysisConfig->mode) + delta;
        mode = std::min(
                std::max(mode, static_cast<int8_t>(nn::irsensor::HandAnalysisMode_Silhouette)
            ),
            static_cast<int8_t>(nn::irsensor::HandAnalysisMode_SilhouetteAndImage)
        );
        m_pHandAnalysisConfig->mode = static_cast<nn::irsensor::HandAnalysisMode>(mode);
    }

private:
    nn::irsensor::HandAnalysisConfig* m_pHandAnalysisConfig;
};


class ShapeArea : public ReadWriteBase
{
public:
    NN_IMPLICIT ShapeArea(const PolygonProperties* pPolygonProperties) NN_NOEXCEPT
        : m_pPolygonProperties(pPolygonProperties)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << m_pPolygonProperties->signedArea;
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
    }

private:
    const PolygonProperties* m_pPolygonProperties;
};

class Shape : public ReadWriteBase
{
public:
    NN_IMPLICIT Shape(
        nn::irsensor::HandAnalysisConfig* pHandAnalysisConfig,
        nn::irsensor::HandAnalysisSilhouetteState* pHandAnalysisSilhouetteState) NN_NOEXCEPT
        : m_pHandAnalysisConfig(pHandAnalysisConfig)
        , m_pHandAnalysisSilhouetteState(pHandAnalysisSilhouetteState)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_pHandAnalysisConfig->mode == nn::irsensor::HandAnalysisMode_Silhouette ||
            m_pHandAnalysisConfig->mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage)
        {
            if (m_pHandAnalysisSilhouetteState->handCount > 0)
            {
                const nn::irsensor::Hand& hand = m_pHandAnalysisSilhouetteState->hands[0];
                int numFinger = 0;

                for (int i = 0; i < nn::irsensor::HandFinger_Count; ++i)
                {
                    if (hand.fingers[i].isValid)
                    {
                        numFinger++;
                    }
                }

                switch (numFinger)
                {
                case 0:
                case 1:
                    sstr << "Rock";
                    break;
                case 2:
                case 3:
                    sstr << "Scissor";
                    break;
                case 5:
                case 6:
                    sstr << "Paper";
                    break;
                default:
                    sstr << "Unknown";
                    break;
                }
            }
            else
            {
                sstr << "Unknown";
            }
        }
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
    }

private:
    nn::irsensor::HandAnalysisConfig* m_pHandAnalysisConfig;
    nn::irsensor::HandAnalysisSilhouetteState* m_pHandAnalysisSilhouetteState;
};

HandAnalysisModeState::HandAnalysisModeState(
    IrSensorMode* pNextProcessor,
    int* pMenuSelection,
    nn::irsensor::IrCameraHandle irCameraHandle,
    GraphicsSystem* pGraphicsSystem) NN_NOEXCEPT
    : IrSensorModeState(pNextProcessor, pMenuSelection, irCameraHandle)
    , m_NextHandAnalysisConfig()
    , m_CurrentHandAnalysisConfig()
    , m_HandAnalysisSilhouetteState()
    , m_HandAnalysisImageState()
    , m_PolygonProperties()
    , m_pGraphicsSystem(pGraphicsSystem)
    , m_pMode(NULL)
    , m_pShapeArea(NULL)
    , m_pShape(NULL)
{
    {
        m_NextHandAnalysisConfig.mode = nn::irsensor::HandAnalysisMode_SilhouetteAndImage;
        m_CurrentHandAnalysisConfig = m_NextHandAnalysisConfig;

        CreateTexture(
            pGraphicsSystem,
            nn::irsensor::IrHandAnalysisImageWidth,
            nn::irsensor::IrHandAnalysisImageHeight,
            nn::gfx::ImageFormat_R16_Unorm, &m_Texture
        );
        CreateSampler(
            pGraphicsSystem,
            &m_Sampler);

        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_Read);
        info.SetSize(ImageSize);
        pGraphicsSystem->AllocateBuffer(&m_Buffer, &info);
    }

    nn::irsensor::HandAnalysisConfig* pHandAnalysisConfig = &m_NextHandAnalysisConfig;
    m_pMode = new Mode(pHandAnalysisConfig);
    MenuItem modeMenuItem("Mode",
        m_pMode
    );
    m_ReadWriteMenu.push_back(modeMenuItem);

    AddCommonReadOnlyMenu(
        &m_ReadOnlyMenu,
        &m_HandAnalysisImageState.samplingNumber,
        &m_HandAnalysisImageState.ambientNoiseLevel
    );
    const PolygonProperties* pPolygonProperties = &m_PolygonProperties;
    nn::irsensor::HandAnalysisSilhouetteState* pHandAnalysisSilhouetteState =
        &m_HandAnalysisSilhouetteState;
    m_pShapeArea = new ShapeArea(pPolygonProperties);
    MenuItem shapeAreaMenuItem("Shape Area",
        m_pShapeArea
    );
    m_ReadOnlyMenu.push_back(shapeAreaMenuItem);

    m_pShape = new Shape(pHandAnalysisConfig, pHandAnalysisSilhouetteState);
    MenuItem shapeMenuItem("Shape",
        m_pShape
    );
    m_ReadOnlyMenu.push_back(shapeMenuItem);
}

HandAnalysisModeState::~HandAnalysisModeState() NN_NOEXCEPT
{
    m_pGraphicsSystem->FreeBuffer(&m_Buffer);

    m_pGraphicsSystem->FreeTexture(&m_Texture.GetTexture());
    m_Texture.GetTextureView().Finalize(&m_pGraphicsSystem->GetDevice());
    m_Sampler.GetSampler().Finalize(&m_pGraphicsSystem->GetDevice());

    delete m_pMode;
    delete m_pShapeArea;
    delete m_pShape;
}

void HandAnalysisModeState::Start() NN_NOEXCEPT
{
    nn::irsensor::RunHandAnalysis(m_IrCameraHandle, m_NextHandAnalysisConfig);
    m_CurrentHandAnalysisConfig = m_NextHandAnalysisConfig;
}

void HandAnalysisModeState::Update() NN_NOEXCEPT
{
    if ((m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_Silhouette) ||
        (m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage))
    {
        int count = 0;
        nn::irsensor::GetHandAnalysisSilhouetteState(
            &m_HandAnalysisSilhouetteState,
            &count,
            1,
            0,
            m_IrCameraHandle
        );
    }
    if ((m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_Image) ||
        (m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage))
    {
        int count = 0;
        nn::irsensor::GetHandAnalysisImageState(
            &m_HandAnalysisImageState,
            &count,
            1,
            0,
            m_IrCameraHandle
        );

        uint16_t* pSource = static_cast<uint16_t*>(m_HandAnalysisImageState.image);
        uint16_t* pDest = m_Buffer.Map< uint16_t >();
        memcpy(pDest, pSource, ImageSize);
        for (size_t i = 0; i < ImageSize; i++)
        {
            const float Max = static_cast<float>(0xffff);
            const float Value = static_cast<float>(pDest[i]) / Max;
            const float ScaledValue = Value * 100.0f > 1.0f ? 1.0f : Value * 100.0f;

            pDest[i] = static_cast<uint16_t>(ScaledValue * Max);
        }
        m_Buffer.FlushMappedRange(0, ImageSize);
        m_Buffer.Unmap();
    }
}

void HandAnalysisModeState::RenderImageState(
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer,
    const int screenIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // DepthStencilTypeはDepthStencilType_DepthNoWriteTestへ設定
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer,
        nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    // グリッドを描画
    nn::util::Vector3fType corner;
    nn::util::Vector3fType size;

    float xStart;
    float xEnd;
    float yStart;
    float yEnd;
    if (screenIndex == 0)
    {
        xStart = 0.45f, xEnd = 0.0f;
        yStart = 0.2f, yEnd = 0.8f;
    }
    else
    {
        xStart = 0.95f, xEnd = 0.5f;
        yStart = 0.2f, yEnd = 0.8f;
    }

    nn::util::VectorSet(&corner, xStart, yStart, 0.0f);
    nn::util::VectorSet(&size, xEnd - xStart, yEnd - yStart, 0.0f);

    pPrimitiveRenderer->SetColor(Color::White);
    pPrimitiveRenderer->DrawQuad(
        pCommandBuffer,
        corner,
        size,
        m_Texture.GetDescriptorSlot(),
        m_Sampler.GetDescriptorSlot());
}

namespace
{
// シルエットを描画
void DrawSilhouette(
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::irsensor::HandAnalysisSilhouetteState* pSilhouetteState,
    PolygonProperties* pPolygonProperties,
    const float xStart,
    const float yStart,
    const float xEnd,
    const float yEnd,
    nn::util::Vector3fType& begin,
    nn::util::Vector3fType& end,
    const float xscale,
    const float yscale
) NN_NOEXCEPT
{
    for (int shapeIndex = 0; shapeIndex < pSilhouetteState->shapeCount; shapeIndex++)
    {
        std::vector<nn::util::Float2> silhouette;
        const nn::irsensor::Shape& Shape = pSilhouetteState->shapes[shapeIndex];

        nn::util::Float2 p1;
        nn::util::Float2 p2;
        p2 = pSilhouetteState->points[Shape.firstPointIndex];
        pPrimitiveRenderer->SetColor(Color::Green);
        for (int i = 0; i < Shape.pointCount - 1; ++i)
        {
            p1 = p2;
            p2 = pSilhouetteState->points[(Shape.firstPointIndex + i + 1) %
                NN_ARRAY_SIZE(pSilhouetteState->points)];

            nn::util::VectorSet(&begin,
                xStart + (xEnd - xStart) * (p1.x * xscale + 1.0f) / 2.0f,
                yStart + (yEnd - yStart) * (p1.y * yscale + 1.0f) / 2.0f, 0.0f);
            nn::util::VectorSet(&end,
                xStart + (xEnd - xStart) * (p2.x * xscale + 1.0f) / 2.0f,
                yStart + (yEnd - yStart) * (p2.y * yscale + 1.0f) / 2.0f, 0.0f);

            pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);

            silhouette.push_back(p1);
        }
        silhouette.push_back(p2);

        std::vector<size_t> convexHullIndices;
        // Convex Hull を計算する
        ExtractConvexHull(convexHullIndices, silhouette);
        if (convexHullIndices.size() > 0)
        {
            p2 = pSilhouetteState->points[Shape.firstPointIndex + convexHullIndices[0]];
            pPrimitiveRenderer->SetColor(Color::Yellow);
            for (size_t i = 0; i < convexHullIndices.size() - 1; ++i)
            {
                p1 = p2;
                p2 = pSilhouetteState->points[Shape.firstPointIndex + convexHullIndices[i + 1]];

                nn::util::VectorSet(&begin,
                    xStart + (xEnd - xStart) * (p1.x * xscale + 1.0f) / 2.0f,
                    yStart + (yEnd - yStart) * (p1.y * yscale + 1.0f) / 2.0f, 0.0f);
                nn::util::VectorSet(&end,
                    xStart + (xEnd - xStart) * (p2.x * xscale + 1.0f) / 2.0f,
                    yStart + (yEnd - yStart) * (p2.y * yscale + 1.0f) / 2.0f, 0.0f);

                pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
            }
        }

        ComputePolygonProperties(pPolygonProperties, &silhouette[0], silhouette.size());
    }
}

// 指先を描画
void DrawFingertip(
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::irsensor::HandAnalysisSilhouetteState* pSilhouetteState,
    const float xStart,
    const float yStart,
    const float xEnd,
    const float yEnd,
    nn::util::Vector3fType& begin,
    const float xscale,
    const float yscale
) NN_NOEXCEPT
{
    for (int handIndex = 0; handIndex < pSilhouetteState->handCount; handIndex++)
    {
        const nn::irsensor::Hand& hand = pSilhouetteState->hands[handIndex];

        // 検出物の中央を描画
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * (hand.palm.center.x * xscale + 1.0f) / 2.0f,
            yStart + (yEnd - yStart) * (hand.palm.center.y * yscale + 1.0f) / 2.0f, 0.0f);
        pPrimitiveRenderer->SetColor(Color::Green);
        pPrimitiveRenderer->DrawSphere(
            pCommandBuffer,
            nns::gfx::PrimitiveRenderer::Surface_Solid,
            nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
            begin,
            8.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));

        for (int i = 0; i < nn::irsensor::HandFinger_Count; ++i)
        {
            if (hand.fingers[i].isValid)
            {
                nn::util::VectorSet(&begin,
                    xStart + (xEnd - xStart) * (hand.fingers[i].tip.x * xscale + 1.0f) / 2.0f,
                    yStart + (yEnd - yStart) * (hand.fingers[i].tip.y * yscale + 1.0f) / 2.0f, 0.0f);
                pPrimitiveRenderer->SetColor(Color::Red);
                pPrimitiveRenderer->DrawSphere(
                    pCommandBuffer,
                    nns::gfx::PrimitiveRenderer::Surface_Solid,
                    nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
                    begin,
                    8.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));

                nn::util::VectorSet(&begin,
                    xStart + (xEnd - xStart) * (hand.fingers[i].root.x * xscale + 1.0f) / 2.0f,
                    yStart + (yEnd - yStart) * (hand.fingers[i].root.y * yscale + 1.0f) / 2.0f, 0.0f);
                pPrimitiveRenderer->SetColor(Color::Blue);
                pPrimitiveRenderer->DrawSphere(
                    pCommandBuffer,
                    nns::gfx::PrimitiveRenderer::Surface_Solid,
                    nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
                    begin,
                    8.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));
            }
        }
    }
}

void WriteHandAnalysisState(
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::irsensor::HandAnalysisSilhouetteState* pSilhouetteState,
    PolygonProperties* pPolygonProperties,
    const int screenIndex) NN_NOEXCEPT
{
    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // DepthStencilTypeはDepthStencilType_DepthNoWriteTestへ設定
    pPrimitiveRenderer->SetDepthStencilState(
        pCommandBuffer,
        nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest
    );

    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;
    pPrimitiveRenderer->SetLineWidth(10.f);

    float xStart;
    float xEnd;
    float yStart;
    float yEnd;
    if (screenIndex == 0)
    {
        xStart = 0.45f, xEnd = 0.0f;
        yStart = 0.8f, yEnd = 0.2f;
    }
    else
    {
        xStart = 0.95f, xEnd = 0.5f;
        yStart = 0.8f, yEnd = 0.2f;
    }

    const float ScaleX = 0.75f;
    const float ScaleY = 1.0f;

    // シルエットを描画
    DrawSilhouette(
        pPrimitiveRenderer,
        pCommandBuffer,
        pSilhouetteState,
        pPolygonProperties,
        xStart,
        yStart,
        xEnd,
        yEnd,
        begin,
        end,
        ScaleX,
        ScaleY
    );

    // 指先を描画
    DrawFingertip(
        pPrimitiveRenderer,
        pCommandBuffer,
        pSilhouetteState,
        xStart,
        yStart,
        xEnd,
        yEnd,
        begin,
        ScaleX,
        ScaleY
    );
}

}

void HandAnalysisModeState::Render(
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer,
    int index) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    // 読み込んだカメラ画像 nn::gfx::buffer を、テクスチャにタイリングします
    if ((m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_Image) ||
        (m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage))
    {
        nn::gfx::TextureCopyRegion dstRegion;
        dstRegion.SetDefault();
        dstRegion.SetDepth(1);

        dstRegion.SetWidth(nn::irsensor::IrHandAnalysisImageWidth);
        dstRegion.SetHeight(nn::irsensor::IrHandAnalysisImageHeight);
        pCommandBuffer->CopyBufferToImage(&m_Texture.GetTexture(), dstRegion, &m_Buffer, 0);

        RenderImageState(pPrimitiveRenderer, pCommandBuffer, index);
    }
    if ((m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_Silhouette) ||
        (m_CurrentHandAnalysisConfig.mode == nn::irsensor::HandAnalysisMode_SilhouetteAndImage))
    {
        WriteHandAnalysisState(
            pPrimitiveRenderer,
            pCommandBuffer,
            &m_HandAnalysisSilhouetteState,
            &m_PolygonProperties,
            index
        );
    }
}
