﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>

#include <nn/nn_Assert.h>
#define NN_PERF_PROFILE_ENABLED
#include <nn/perf.h>
#include <nn/vi.h>

#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include <nns/gfx/gfx_PrimitiveRenderer.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
#include <nvnTool/nvnTool_GlslcInterface.h>
#endif
#endif

#include "Demo1.h"
#include "Demo1Color.h"
#include "Demo1GraphicsSystem.h"

const size_t MemoryPoolSize = 16 * 1024 * 1024;
const size_t MemoryPoolSizeInvisible = 64 * 1024 * 1024;

const int DescriptorPoolSlotCount = 2048;

const size_t CommandBufferCommandMemoryChunkSize = 512 * 1024;
const size_t CommandBufferControlMemoryChunkSize = 64 * 1024;

const int SwapInterval = 1;

GraphicsSystem::GraphicsSystem() NN_NOEXCEPT
    : m_pApplicationHeap(NULL)
    , m_pMemoryPool(NULL)
    , m_MemoryPoolOffset(0)
    , m_MemoryPoolCommandOffset(0)
    , m_pMemoryPoolInvisible(NULL)
    , m_MemoryPoolInvisibleOffset(0)
    , m_TextureSlotCount(0)
    , m_SamplerSlotCount(0)
    , m_DisplayWidth(1280)
    , m_DisplayHeight(720)
    , m_DebugFontHeap(NULL)
    , m_pPrimitiveRenderer(NULL)
    , m_BufferDescriptorIndex(0)
    , m_SamplerDescriptorIndex(0)
    , m_FrameCount(0)
{
}

GraphicsSystem::~GraphicsSystem() NN_NOEXCEPT
{
}

void GraphicsSystem::SetApplicationHeap(nn::mem::StandardAllocator* pValue) NN_NOEXCEPT
{
    m_pApplicationHeap = pValue;
}

nn::gfx::Device& GraphicsSystem::GetDevice() NN_NOEXCEPT
{
    return m_Device;
}

nn::gfx::CommandBuffer& GraphicsSystem::GetCommandBuffer() NN_NOEXCEPT
{
    return m_CommandBuffer;
}

nn::gfx::util::DebugFontTextWriter& GraphicsSystem::GetDebugFont() NN_NOEXCEPT
{
    return m_DebugFont;
}

nns::gfx::PrimitiveRenderer::Renderer& GraphicsSystem::GetPrimitiveRenderer() const NN_NOEXCEPT
{
    return *m_pPrimitiveRenderer;
}

int GraphicsSystem::GetDisplayWidth() const NN_NOEXCEPT
{
    return m_DisplayWidth;
}

int GraphicsSystem::GetDisplayHeight() const NN_NOEXCEPT
{
    return m_DisplayHeight;
}

void GraphicsSystem::Initialize() NN_NOEXCEPT
{
    nn::vi::Initialize();
    this->InitializeDisplay();
    nn::gfx::Initialize();
    this->InitializeDevice();

#if NN_GFX_IS_TARGET_NVN
    nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData(&m_Device);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &m_TextureSlotCount);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &m_SamplerSlotCount);
#endif

    this->InitializeMemoryPool();
    this->InitializeMemoryPoolInvisible();
    this->InitializeSwapChain();
    this->InitializeQueue();
    this->InitializeFence();
    this->InitializeDepthStencilTexture();
    this->InitializeDepthStencilView();
    this->InitializeBufferDescriptorPool();
    this->InitializeTextureDescriptorPool();
    this->InitializeSamplerDescriptorPool();
    this->InitializeBlurBuffer();
    this->InitializeCommandBuffer();
    this->InitializeViewportScissorState();
    this->InitializeBlendState();
    this->InitializeDepthStencilState();
    this->InitializeRasterizerState();
    this->InitializePrimitiveRenderer();

    // 処理メータの初期化
    InitializeLoadMeter();

    InitializeColorBuffer(
        &m_ColorTextureView,
        &m_ColorTargetView,
        &m_ColorBuffer
    );

    m_ColorTextureDescriptorIndex = RegisterTextureViewSlot(
        &m_ColorDescriptorSlot,
        m_ColorTextureView
    );

    InitializeSampler(&m_Sampler);
    m_SamplerDescriptorIndex = RegisterSamplerSlot(&m_SamplerSlot, m_Sampler);

    NN_ASSERT(m_MemoryPoolOffset < static_cast<ptrdiff_t>(m_VisiblePoolMemorySize));
    NN_ASSERT(m_MemoryPoolInvisibleOffset < static_cast<ptrdiff_t>(m_InvisiblePoolMemorySize));

    m_FrameCount = 0;
}

void GraphicsSystem::Finalize() NN_NOEXCEPT
{
    // 処理メータ破棄
    FinalizeLoadMeter();

    FinalizeSampler(&m_Sampler);
    FinalizeColorBuffer(
        &m_ColorTextureView,
        &m_ColorTargetView,
        &m_ColorBuffer
    );

    this->FinalizePrimitiveRenderer();
    m_RasterizerState.Finalize(&m_Device);
    m_DepthStencilState.Finalize(&m_Device);
    this->FinalizeBlendState();
    this->FinalizeViewportScissorState();
    this->ResetCommandBuffer();
    m_CommandBuffer.Finalize(&m_Device);
    m_SamplerDescriptorPool.Finalize(&m_Device);
    m_TextureDescriptorPool.Finalize(&m_Device);
    m_BufferDescriptorPool.Finalize(&m_Device);
    m_BlurColorTextureView.Finalize(&m_Device);
    m_BlurColorTargetView.Finalize(&m_Device);
    m_BlurColorBuffer.Finalize(&m_Device);
    m_DepthStencilView.Finalize(&m_Device);
    m_DepthStencilTexture.Finalize(&m_Device);
    m_Fence.Finalize(&m_Device);
    m_Queue.Finalize(&m_Device);
    m_SwapChain.Finalize(&m_Device);
    this->FinalizeMemoryPoolInvisible();
    this->FinalizeMemoryPool();
    m_Device.Finalize();
    nn::gfx::Finalize();
    nn::vi::DestroyLayer(m_pLayer);
    nn::vi::CloseDisplay(m_pDisplay);
    nn::vi::Finalize();
}

void GraphicsSystem::BeginDraw() NN_NOEXCEPT
{
    NN_PERF_BEGIN_FRAME();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_BEGIN_MEASURE_NAME("MakeCommand");

    this->ResetCommandBuffer();
    this->AddCommandMemoryToCommandBuffer(&m_CommandBuffer);
    this->AddControlMemoryToCommandBuffer(&m_CommandBuffer);

    m_CommandBuffer.Begin();
    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Red());
    NN_PERF_BEGIN_MEASURE_GPU(&m_CommandBuffer);

    m_CommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Descriptor |
        nn::gfx::GpuAccess_ShaderCode);

    m_CommandBuffer.SetDescriptorPool(&m_BufferDescriptorPool);
    m_CommandBuffer.SetDescriptorPool(&m_TextureDescriptorPool);
    m_CommandBuffer.SetDescriptorPool(&m_SamplerDescriptorPool);

    m_CommandBuffer.ClearColor(
        &m_ColorTargetView,
        0.0f,
        0.0f,
        0.0f,
        1.0f,
        0);

    m_CommandBuffer.ClearDepthStencil(
        &m_DepthStencilView,
        1.0f,
        0,
        nn::gfx::DepthStencilClearMode_DepthStencil,
        0);

    nn::gfx::ColorTargetView* pColorTargetView = &m_ColorTargetView;
    m_CommandBuffer.SetRenderTargets(
        1,
        &pColorTargetView,
        &m_DepthStencilView);

    m_CommandBuffer.SetViewportScissorState(&m_ViewportScissorState);
    m_CommandBuffer.SetRasterizerState(&m_RasterizerState);

    m_CommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Texture
        | nn::gfx::GpuAccess_IndexBuffer
        | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer);

    // プリミティブレンダラの更新処理
    // 複数バッファで運用する場合は、 Update の引数で利用する
    static int s_BufferIndex = 0;
    m_pPrimitiveRenderer->Update(s_BufferIndex);
    s_BufferIndex = 1 - s_BufferIndex;
}

void GraphicsSystem::EndDraw() NN_NOEXCEPT
{
    NN_PERF_END_MEASURE_GPU(&m_CommandBuffer);
    m_CommandBuffer.FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    m_CommandBuffer.End();

    NN_PERF_END_MEASURE();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Blue());

    NN_PERF_BEGIN_MEASURE_NAME("ExecuteCommand");
    m_Queue.ExecuteCommand(&m_CommandBuffer, &m_Fence);
    NN_PERF_END_MEASURE();
    m_Queue.Flush();

    m_Queue.Present(&m_SwapChain, SwapInterval);
    m_Queue.Sync();
    NN_PERF_END_FRAME();

    ++m_FrameCount;
}

void GraphicsSystem::Synchronize(const nn::TimeSpan& timeout) NN_NOEXCEPT
{
    m_Fence.Sync(timeout);
}

void GraphicsSystem::AllocateTexture(
    nn::gfx::Texture* pTexture,
    const nn::gfx::Texture::InfoType* pInfo) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTexture);
    NN_ASSERT_NOT_NULL(pInfo);

    m_MemoryPoolInvisibleOffset = nn::util::align_up(
        m_MemoryPoolInvisibleOffset,
        nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, *pInfo)
    );
    size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, *pInfo);
    pTexture->Initialize(
        &m_Device,
        *pInfo,
        &m_MemoryPoolInvisible,
        m_MemoryPoolInvisibleOffset,
        size
    );
    m_MemoryPoolInvisibleOffset += size;
}

void GraphicsSystem::AllocateBuffer(
    nn::gfx::Buffer* pBuffer,
    const nn::gfx::Buffer::InfoType* pInfo
) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pBuffer);
    NN_ASSERT_NOT_NULL(pInfo);

    m_MemoryPoolOffset = nn::util::align_up(
        m_MemoryPoolOffset,
        nn::gfx::Buffer::GetBufferAlignment(&m_Device, *pInfo)
    );
    pBuffer->Initialize(&m_Device, *pInfo, &m_MemoryPool, m_MemoryPoolOffset, pInfo->GetSize());
    m_MemoryPoolOffset += pInfo->GetSize();
}

void GraphicsSystem::FreeTexture(nn::gfx::Texture* pTexture) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTexture);

    pTexture->Finalize(&m_Device);
}

void GraphicsSystem::FreeBuffer(nn::gfx::Buffer* pBuffer) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pBuffer);

    pBuffer->Finalize(&m_Device);
}

int GraphicsSystem::RegisterTextureViewSlot(nn::gfx::DescriptorSlot* pOutValue,
const nn::gfx::TextureView& textureView ) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutValue);

    const int Slot = m_TextureSlotCount;
    m_TextureDescriptorPool.BeginUpdate();
    m_TextureDescriptorPool.SetTextureView(Slot, &textureView);
    m_TextureDescriptorPool.EndUpdate();
    m_TextureDescriptorPool.GetDescriptorSlot(pOutValue, Slot);

    ++m_TextureSlotCount;
    return Slot;
}

int GraphicsSystem::RegisterSamplerSlot(nn::gfx::DescriptorSlot* pOutValue,
const nn::gfx::Sampler& sampler) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutValue);

    const int Slot = m_SamplerSlotCount;
    m_SamplerDescriptorPool.BeginUpdate();
    m_SamplerDescriptorPool.SetSampler(Slot, &sampler);
    m_SamplerDescriptorPool.EndUpdate();
    m_SamplerDescriptorPool.GetDescriptorSlot(pOutValue, Slot);

    ++m_SamplerSlotCount;
    return Slot;
}

void GraphicsSystem::OutOfCommandMemoryEventCallback(
nn::gfx::CommandBuffer* pCommandBuffer,
const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pCommandBuffer->GetUserPtr());
    NN_UNUSED(arg);

    reinterpret_cast<GraphicsSystem*>(
        pCommandBuffer->GetUserPtr()
        )->AddCommandMemoryToCommandBuffer(pCommandBuffer);
}

void GraphicsSystem::OutOfControlMemoryEventCallback(
nn::gfx::CommandBuffer* pCommandBuffer,
const nn::gfx::OutOfMemoryEventArg& arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pCommandBuffer->GetUserPtr());
    NN_UNUSED(arg);

    reinterpret_cast<GraphicsSystem*>(
        pCommandBuffer->GetUserPtr()
        )->AddControlMemoryToCommandBuffer(pCommandBuffer);
}

void* GraphicsSystem::AllocateFunction(size_t size,
size_t alignment,
void* pUserData) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pUserData);
    return reinterpret_cast<
        nn::mem::StandardAllocator*>(pUserData)->Allocate(size, alignment);
}

void GraphicsSystem::FreeFunction(void* ptr, void* pUserData) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pUserData);
    reinterpret_cast<nn::mem::StandardAllocator*>(pUserData)->Free(ptr);
}

void GraphicsSystem::InitializeDisplay() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&m_pDisplay));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::vi::CreateLayer(&m_pLayer,
        m_pDisplay));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::vi::SetLayerScalingMode(m_pLayer,
        nn::vi::ScalingMode_FitToLayer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::vi::GetNativeWindow(&m_NativeWindowHandle, m_pLayer));
}

void GraphicsSystem::InitializeDevice() NN_NOEXCEPT
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();

    m_Device.Initialize(info);
}

void GraphicsSystem::InitializeMemoryPool() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nn::gfx::MemoryPoolInfo info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached |
        nn::gfx::MemoryPoolProperty_GpuCached);

    m_VisiblePoolMemorySize = nn::util::align_up(MemoryPoolSize,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&m_Device, info));

    const size_t Aligment =
        nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);

    m_pMemoryPool = m_pApplicationHeap->Allocate(MemoryPoolSize, Aligment);

    info.SetPoolMemory(m_pMemoryPool, MemoryPoolSize);

    m_MemoryPool.Initialize(&m_Device, info);

    m_MemoryPoolOffset = 0;

    m_MemoryPoolCommandOffset = 0;
}

void GraphicsSystem::InitializeMemoryPoolInvisible() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nn::gfx::MemoryPoolInfo info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuInvisible |
        nn::gfx::MemoryPoolProperty_GpuCached |
        nn::gfx::MemoryPoolProperty_Compressible);

    m_InvisiblePoolMemorySize = nn::util::align_up(MemoryPoolSizeInvisible,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&m_Device, info));

    const size_t Aligment =
        nn::gfx::MemoryPool::GetPoolMemoryAlignment(&m_Device, info);

    m_pMemoryPoolInvisible =
        m_pApplicationHeap->Allocate(MemoryPoolSizeInvisible, Aligment);

    info.SetPoolMemory(m_pMemoryPoolInvisible, MemoryPoolSizeInvisible);

    m_MemoryPoolInvisible.Initialize(&m_Device, info);

    m_MemoryPoolInvisibleOffset = 0;
}

void GraphicsSystem::InitializeSwapChain() NN_NOEXCEPT
{
    nn::gfx::SwapChain::InfoType info;

    info.SetDefault();
    info.SetLayer(m_pLayer);
    info.SetWidth(m_DisplayWidth);
    info.SetHeight(m_DisplayHeight);
    info.SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);
    info.SetBufferCount(2);

    if (NN_STATIC_CONDITION(nn::gfx::SwapChain::IsMemoryPoolRequired))
    {
        m_MemoryPoolInvisibleOffset = nn::util::align_up(
            m_MemoryPoolInvisibleOffset,
            nn::gfx::SwapChain::GetScanBufferAlignment(&m_Device, info));

        size_t size = m_SwapChain.CalculateScanBufferSize(&m_Device, info);

        m_SwapChain.Initialize(&m_Device,
            info,
            &m_MemoryPoolInvisible,
            m_MemoryPoolInvisibleOffset,
            size);

        m_MemoryPoolInvisibleOffset += size;
    }
    else
    {
        m_SwapChain.Initialize(&m_Device, info, NULL, 0, 0);
    }
}

void GraphicsSystem::InitializeQueue() NN_NOEXCEPT
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability(nn::gfx::QueueCapability_Graphics);

    m_Queue.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeFence() NN_NOEXCEPT
{
    nn::gfx::Fence::InfoType info;
    info.SetDefault();

    m_Fence.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeDepthStencilTexture() NN_NOEXCEPT
{
    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetWidth(m_DisplayWidth);
    info.SetHeight(m_DisplayHeight);
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil);
    info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    info.SetImageFormat(nn::gfx::ImageFormat_D16_Unorm);
    info.SetMipCount(1);
    info.SetDepth(1);

    m_MemoryPoolInvisibleOffset = nn::util::align_up(
        m_MemoryPoolInvisibleOffset,
        nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, info));

    size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, info);

    m_DepthStencilTexture.Initialize(&m_Device,
        info,
        &m_MemoryPoolInvisible,
        m_MemoryPoolInvisibleOffset,
        size);

    m_MemoryPoolInvisibleOffset += size;
}

void GraphicsSystem::InitializeDepthStencilView() NN_NOEXCEPT
{
    nn::gfx::DepthStencilView::InfoType info;
    info.SetDefault();
    info.SetImageDimension(nn::gfx::ImageDimension_2d);
    info.SetTexturePtr(&m_DepthStencilTexture);

    m_DepthStencilView.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeBufferDescriptorPool() NN_NOEXCEPT
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_BufferView);
    info.SetSlotCount(DescriptorPoolSlotCount + 20);
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&m_Device, info);
    m_MemoryPoolOffset = nn::util::align_up(m_MemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&m_Device, info));
    m_BufferDescriptorPool.Initialize(
        &m_Device, info, &m_MemoryPool, m_MemoryPoolOffset, size
    );
    m_MemoryPoolOffset += size;
}

void GraphicsSystem::InitializeTextureDescriptorPool() NN_NOEXCEPT
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_TextureView);
    info.SetSlotCount(DescriptorPoolSlotCount + 20);

    m_MemoryPoolOffset = nn::util::align_up(
        m_MemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(
        &m_Device, info));

    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(
        &m_Device, info);

    m_TextureDescriptorPool.Initialize(&m_Device,
        info,
        &m_MemoryPool,
        m_MemoryPoolOffset,
        size);

    m_MemoryPoolOffset += size;
}

void GraphicsSystem::InitializeSamplerDescriptorPool() NN_NOEXCEPT
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_Sampler);
    info.SetSlotCount(DescriptorPoolSlotCount + 20);

    m_MemoryPoolOffset = nn::util::align_up(
        m_MemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(
        &m_Device, info));

    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(
        &m_Device, info);

    m_SamplerDescriptorPool.Initialize(&m_Device,
        info,
        &m_MemoryPool,
        m_MemoryPoolOffset,
        size);

    m_MemoryPoolOffset += size;
}

void GraphicsSystem::InitializeCommandBuffer() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    m_CommandBuffer.Initialize(&m_Device, info);
    m_CommandBuffer.SetUserPtr(this);
    m_CommandBuffer.SetOutOfCommandMemoryEventCallback(
        OutOfCommandMemoryEventCallback);
    m_CommandBuffer.SetOutOfControlMemoryEventCallback(
        OutOfControlMemoryEventCallback);

    m_MemoryPoolOffset = nn::util::align_up(
        m_MemoryPoolOffset,
        nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device));

    m_MemoryPoolCommandOffset = m_MemoryPoolOffset;
    m_MemoryPoolCommandOffset += CommandBufferCommandMemoryChunkSize;
}

void GraphicsSystem::InitializeViewportScissorState() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled(true);

    nn::gfx::ViewportStateInfo viewportStateInfo;
    viewportStateInfo.SetDefault();
    viewportStateInfo.SetWidth(static_cast<float>(m_DisplayWidth));
    viewportStateInfo.SetHeight(static_cast<float>(m_DisplayHeight));
    info.SetViewportStateInfoArray(&viewportStateInfo, 1);

    nn::gfx::ScissorStateInfo scissorStateInfo;
    scissorStateInfo.SetDefault();
    scissorStateInfo.SetWidth(m_DisplayWidth);
    scissorStateInfo.SetHeight(m_DisplayHeight);
    info.SetScissorStateInfoArray(&scissorStateInfo, 1);

    size_t memorySize =
        nn::gfx::ViewportScissorState::GetRequiredMemorySize(info);
    m_ViewportScissorState.SetMemory(
        m_pApplicationHeap->Allocate(memorySize), memorySize);

    m_ViewportScissorState.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeBlendState() NN_NOEXCEPT
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();

    nn::gfx::BlendTargetStateInfo targetInfo;
    targetInfo.SetDefault();
    targetInfo.SetChannelMask(nn::gfx::ChannelMask_Red |
        nn::gfx::ChannelMask_Green |
        nn::gfx::ChannelMask_Blue);
    targetInfo.SetBlendEnabled(true);
    targetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
    targetInfo.SetDestinationColorBlendFactor(
        nn::gfx::BlendFactor_OneMinusSourceAlpha);
    targetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
    info.SetBlendTargetStateInfoArray(&targetInfo, 1);

    size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize(info);
    m_BlendState.SetMemory(m_pApplicationHeap->Allocate(memorySize),
        memorySize);

    m_BlendState.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeDepthStencilState() NN_NOEXCEPT
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    info.SetDepthTestEnabled(false);
    info.SetDepthWriteEnabled(true);

    m_DepthStencilState.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeRasterizerState() NN_NOEXCEPT
{
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetCullMode(nn::gfx::CullMode_None);
    info.SetPrimitiveTopologyType(nn::gfx::PrimitiveTopologyType_Triangle);
    info.SetScissorEnabled(true);
    info.SetDepthClipEnabled(false);
    m_RasterizerState.Initialize(&m_Device, info);
}

void GraphicsSystem::InitializeDebugFont() NN_NOEXCEPT
{
    const int CharCountMax = 4096;
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(CharCountMax);
    info.SetUserMemoryPoolEnabled(false);

    size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(
        &m_Device,
        info
        );
    m_DebugFontHeap = nn::util::BytePtr(new uint8_t[debugFontHeapSize]);

    m_DebugFont.Initialize(
        &m_Device,
        info,
        m_DebugFontHeap.Get(),
        debugFontHeapSize,
        NULL,
        0,
        0
        );
    m_DebugFont.SetDisplayWidth(m_DisplayWidth);
    m_DebugFont.SetDisplayHeight(m_DisplayHeight);

    m_DebugFont.SetTextureDescriptor(&m_TextureDescriptorPool, m_TextureSlotCount);
    ++m_TextureSlotCount;
    m_DebugFont.SetSamplerDescriptor(&m_SamplerDescriptorPool, m_SamplerSlotCount);
    ++m_SamplerSlotCount;
}

void GraphicsSystem::InitializePrimitiveRenderer() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();

    // ダブルバッファで運用する場合は、SetMultiBufferQuantity で 2 を指定し、
    // プリミティブレンダラ内部のバッファをダブルにしたうえで、
    // pPrimitiveRenderer->Update(); で利用するバッファを選択( 0 -> 1 -> 0 -> 1 )する
    info.SetMultiBufferQuantity(2);
    info.SetAllocator(AllocateFunction, m_pApplicationHeap);

    // PrimitiveRendererのインスタンス
    m_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(&m_Device, info);

    m_pPrimitiveRenderer->SetScreenWidth(m_DisplayWidth);
    m_pPrimitiveRenderer->SetScreenHeight(m_DisplayHeight);
}

void GraphicsSystem::FinalizeMemoryPool() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    m_MemoryPool.Finalize(&m_Device);

    m_pApplicationHeap->Free(m_pMemoryPool);
}

void GraphicsSystem::FinalizeMemoryPoolInvisible() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    m_MemoryPoolInvisible.Finalize(&m_Device);

    m_pApplicationHeap->Free(m_pMemoryPoolInvisible);
}

void GraphicsSystem::FinalizeViewportScissorState() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    void* ptr = m_ViewportScissorState.GetMemory();

    m_ViewportScissorState.Finalize(&m_Device);

    m_pApplicationHeap->Free(ptr);
}

void GraphicsSystem::FinalizeBlendState() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    void* ptr = m_BlendState.GetMemory();

    m_BlendState.Finalize(&m_Device);

    m_pApplicationHeap->Free(ptr);
}

void GraphicsSystem::FinalizeDebugFont() NN_NOEXCEPT
{
    m_DebugFont.Finalize();
    delete[] reinterpret_cast<uint8_t*>(m_DebugFontHeap.Get());
    m_DebugFontHeap.Reset(NULL);
}

void GraphicsSystem::FinalizePrimitiveRenderer() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    nns::gfx::PrimitiveRenderer::DestroyRenderer(
        m_pPrimitiveRenderer,
        &m_Device,
        FreeFunction,
        m_pApplicationHeap
    );
}

void GraphicsSystem::ResetCommandBuffer() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    m_CommandBuffer.Reset();

    m_MemoryPoolOffset = m_MemoryPoolCommandOffset;

    const size_t Size = m_CommandBufferControlMemoryChunks.size();

    for (size_t i = 0; i < Size; ++i)
    {
        m_pApplicationHeap->Free(
            m_CommandBufferControlMemoryChunks[i]);
    }

    m_CommandBufferControlMemoryChunks.clear();
}

void GraphicsSystem::AddCommandMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);

    m_MemoryPoolOffset = nn::util::align_up(
        m_MemoryPoolOffset,
        nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&m_Device));

    pCommandBuffer->AddCommandMemory(&m_MemoryPool,
        m_MemoryPoolOffset,
        CommandBufferCommandMemoryChunkSize);

    m_MemoryPoolOffset += CommandBufferCommandMemoryChunkSize;
}

void GraphicsSystem::AddControlMemoryToCommandBuffer(nn::gfx::CommandBuffer* pCommandBuffer
) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(m_pApplicationHeap);

    void* pControlMemoryChunk = m_pApplicationHeap->Allocate(
        CommandBufferControlMemoryChunkSize,
        nn::gfx::CommandBuffer::GetControlMemoryAlignment(&m_Device));

    pCommandBuffer->AddControlMemory(pControlMemoryChunk,
        CommandBufferControlMemoryChunkSize);

    m_CommandBufferControlMemoryChunks.push_back(pControlMemoryChunk);
}

// カラーバッファを初期化
void GraphicsSystem::InitializeColorBuffer(
    nn::gfx::TextureView* pColorTextureView,
    nn::gfx::ColorTargetView* pColorTargetView,
    nn::gfx::Texture* pColorBuffer
) NN_NOEXCEPT
{
    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetWidth(m_DisplayWidth);
    info.SetHeight(m_DisplayHeight);
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
    info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    info.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
    info.SetMipCount(1);

    m_MemoryPoolInvisibleOffset = nn::util::align_up(
        m_MemoryPoolInvisibleOffset,
        nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, info)
    );
    size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, info);
    pColorBuffer->Initialize(&m_Device, info, &m_MemoryPoolInvisible,
        m_MemoryPoolInvisibleOffset, size);
    m_MemoryPoolInvisibleOffset += size;

    // カラーターゲットビューを初期化
    nn::gfx::ColorTargetView::InfoType targetInfo;
    targetInfo.SetDefault();
    targetInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    targetInfo.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
    targetInfo.SetTexturePtr(pColorBuffer);
    pColorTargetView->Initialize(&m_Device, targetInfo);

    // カラーバッファのテクスチャビューを初期化
    nn::gfx::TextureView::InfoType viewInfo;
    viewInfo.SetDefault();
    viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    viewInfo.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
    viewInfo.SetTexturePtr(pColorBuffer);
    pColorTextureView->Initialize(&m_Device, viewInfo);
}

void GraphicsSystem::FinalizeColorBuffer(
    nn::gfx::TextureView* pColorTextureView,
    nn::gfx::ColorTargetView* pColorTargetView,
    nn::gfx::Texture* pColorBuffer
) NN_NOEXCEPT
{
    pColorTextureView->Finalize(&m_Device);
    pColorTargetView->Finalize(&m_Device);
    pColorBuffer->Finalize(&m_Device);
}

void GraphicsSystem::DrawScreenQuarter(GraphicsSystem::DrawPosition position) NN_NOEXCEPT
{
    nn::gfx::DescriptorSlot samplerDescriptor;
    nn::gfx::DescriptorSlot colorDescriptor;

    nns::gfx::PrimitiveRenderer::Renderer::Model model;
    nns::gfx::PrimitiveRenderer::Renderer::SetDefault(&model);

    const nn::util::Vector3fType size
        = NN_UTIL_VECTOR_3F_INITIALIZER(1.f, -1.f, 2.f);
    nn::util::Vector3fType pos;
    switch (position)
    {
    case GraphicsSystem::DrawPosition_TopLeft:
        pos = NN_UTIL_VECTOR_3F_INITIALIZER(-1.0f, 1.f, 0.f);
        break;
    case GraphicsSystem::DrawPosition_TopRight:
        pos = NN_UTIL_VECTOR_3F_INITIALIZER(0.f, 1.f, 0.f);
        break;
    case GraphicsSystem::DrawPosition_BottomLeft:
        pos = NN_UTIL_VECTOR_3F_INITIALIZER(-1.0f, 0.f, 0.f);
        break;
    case GraphicsSystem::DrawPosition_BottomRight:
        pos = NN_UTIL_VECTOR_3F_INITIALIZER(0.f, 0.f, 0.f);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_SamplerDescriptorPool.GetDescriptorSlot(&samplerDescriptor,
        m_SamplerDescriptorIndex);
    m_TextureDescriptorPool.GetDescriptorSlot(&colorDescriptor,
        m_ColorTextureDescriptorIndex);
    m_pPrimitiveRenderer->DrawQuad(&m_CommandBuffer,
        pos,size,
        colorDescriptor,
        samplerDescriptor);
}

// サンプラを初期化
void GraphicsSystem::InitializeSampler(nn::gfx::Sampler* pSampler) NN_NOEXCEPT
{
    nn::gfx::Sampler::InfoType info;
    info.SetDefault();
    info.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
    info.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
    info.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
    info.SetAddressW(nn::gfx::TextureAddressMode_Mirror);
    pSampler->Initialize(&m_Device, info);
}

// サンプラを終了処理
void GraphicsSystem::FinalizeSampler(nn::gfx::Sampler* pSampler) NN_NOEXCEPT
{
    pSampler->Finalize(&m_Device);
}

nn::gfx::DescriptorPool* GraphicsSystem::GetTexureDescriptorPool() NN_NOEXCEPT
{
    return &m_TextureDescriptorPool;
}

nn::gfx::DescriptorPool* GraphicsSystem::GetBufferDescriptorPool() NN_NOEXCEPT
{
    return &m_BufferDescriptorPool;
}

nn::gfx::DescriptorPool* GraphicsSystem::GetSamplerDescriptorPool() NN_NOEXCEPT
{
    return &m_SamplerDescriptorPool;
}

int GraphicsSystem::GetBufferDescriptorIndex() const NN_NOEXCEPT
{
    return m_BufferDescriptorIndex;
}

int GraphicsSystem::GetSamplerDescriptorIndex() const NN_NOEXCEPT
{
    return m_SamplerDescriptorIndex;
}

nn::gfx::SwapChain* GraphicsSystem::GetSwapChain() NN_NOEXCEPT
{
    return &m_SwapChain;
}

nn::gfx::MemoryPool* GraphicsSystem::GetMemoryPoolInvisible() NN_NOEXCEPT
{
    return &m_MemoryPoolInvisible;
}

void GraphicsSystem::InitializeBlurBuffer() NN_NOEXCEPT
{
    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetWidth(m_DisplayWidth);
    info.SetHeight(m_DisplayHeight);
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
    info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    info.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
    info.SetMipCount(1);

    m_MemoryPoolInvisibleOffset = nn::util::align_up(m_MemoryPoolInvisibleOffset,
        nn::gfx::Texture::CalculateMipDataAlignment(&m_Device, info));
    size_t size = nn::gfx::Texture::CalculateMipDataSize(&m_Device, info);
    m_BlurColorBuffer.Initialize(
        &m_Device,
        info,
        GetMemoryPoolInvisible(),
        m_MemoryPoolInvisibleOffset, size);
    m_MemoryPoolInvisibleOffset += size;

    // カラーターゲットビューを初期化
    nn::gfx::ColorTargetView::InfoType targetInfo;
    targetInfo.SetDefault();
    targetInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    targetInfo.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
    targetInfo.SetTexturePtr(&m_BlurColorBuffer);
    m_BlurColorTargetView.Initialize(&m_Device, targetInfo);

    // カラーバッファのテクスチャビューを初期化
    nn::gfx::TextureView::InfoType viewInfo;
    viewInfo.SetDefault();
    viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    viewInfo.SetImageFormat(nn::gfx::ImageFormat_R16_G16_B16_A16_Float);
    viewInfo.SetTexturePtr(&m_BlurColorBuffer);
    m_BlurColorTextureView.Initialize(&m_Device, viewInfo);
}

// ---------------------------------------------------------------
// 負荷メーターの初期化
// ---------------------------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_WIN )
// windows 版では nn::os::SetThreadCoreMask による設定が効かない場合があるため
// コア番号は常に 0 を返すようにする。
int GetFixedCoreNumber()
{
    return 0;
}
#endif
void GraphicsSystem::InitializeLoadMeter() NN_NOEXCEPT
{
    nn::perf::LoadMeterCenterInfo info;
    info.SetCoreCount(1);
    info.SetCpuSectionCountMax(64);
    info.SetGpuSectionCountMax(64);

    size_t memorySize = NN_PERF_GET_BUFFER_SIZE(info);
    size_t memoryAlignment = NN_PERF_GET_BUFFER_ALIGNMENT();
    m_pMeterMemory = AllocateFunction(memorySize, memoryAlignment,
        &GetStandardAllocator());
    size_t memoryPoolSize = NN_PERF_GET_MEMORY_POOL_SIZE(&m_Device, info);
    m_MemoryPoolOffset = nn::util::align_up(m_MemoryPoolOffset,
        NN_PERF_GET_MEMORY_POOL_ALIGNMENT(&m_Device, info));
    NN_PERF_INITIALIZE_METER(
        &m_Device,
        info,
        m_pMeterMemory,
        memorySize,
        &m_MemoryPool,
        m_MemoryPoolOffset,
        memoryPoolSize
    );
    m_MemoryPoolOffset += memoryPoolSize;

#if defined( NN_BUILD_CONFIG_OS_WIN )
    NN_PERF_SET_GET_CORE_NUMBER_FUNCTION(GetFixedCoreNumber);
#endif
}

// ---------------------------------------------------------------
// 負荷メーターの破棄
// ---------------------------------------------------------------
void GraphicsSystem::FinalizeLoadMeter() NN_NOEXCEPT
{
    NN_PERF_FINALIZE_METER(&m_Device);
    FreeFunction(m_pMeterMemory, &GetStandardAllocator());
}
