﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <sstream>

#include <nn/nn_Assert.h>

#include "Demo1ClusteringModeState.h"
#include "Demo1Color.h"

class ObjectPixelCountMin : public ReadWriteBase
{
public:
    NN_IMPLICIT ObjectPixelCountMin(
        nn::irsensor::ClusteringProcessorConfig* pClusteringConfig
    ) NN_NOEXCEPT
        : m_pClusteringConfig(pClusteringConfig)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << m_pClusteringConfig->objectPixelCountMin;
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        m_pClusteringConfig->objectPixelCountMin += delta * 16;
        m_pClusteringConfig->objectPixelCountMin =
            std::min(
                std::max(m_pClusteringConfig->objectPixelCountMin, 0),
                m_pClusteringConfig->objectPixelCountMax - 1
            );
    }

private:
    nn::irsensor::ClusteringProcessorConfig* m_pClusteringConfig;
};

class ObjectPixelCountMax : public ReadWriteBase
{
public:
    NN_IMPLICIT ObjectPixelCountMax(
        nn::irsensor::ClusteringProcessorConfig* pClusteringConfig
    ) NN_NOEXCEPT
        : m_pClusteringConfig(pClusteringConfig)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << m_pClusteringConfig->objectPixelCountMax;
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        m_pClusteringConfig->objectPixelCountMax += delta * 16;
        m_pClusteringConfig->objectPixelCountMax =
            std::max(
                std::min(
                    m_pClusteringConfig->objectPixelCountMax,
                    nn::irsensor::ClusteringProcessorObjectPixelCountMax
                ),
                m_pClusteringConfig->objectPixelCountMin + 1
            );
    }

private:
    nn::irsensor::ClusteringProcessorConfig* m_pClusteringConfig;
};

class ObjectIntensityMin : public ReadWriteBase
{
public:
    NN_IMPLICIT ObjectIntensityMin(
        nn::irsensor::ClusteringProcessorConfig* pClusteringConfig
    ) NN_NOEXCEPT
        : m_pClusteringConfig(pClusteringConfig)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << m_pClusteringConfig->objectIntensityMin;
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        m_pClusteringConfig->objectIntensityMin += delta * 8;
        m_pClusteringConfig->objectIntensityMin =
            std::max(
                std::min(
                    m_pClusteringConfig->objectIntensityMin,
                    nn::irsensor::IrCameraIntensityMax),
                0
            );
    }

private:
    nn::irsensor::ClusteringProcessorConfig* m_pClusteringConfig;
};


class ExternalLightFilter : public ReadWriteBase
{
public:
    NN_IMPLICIT ExternalLightFilter(
        nn::irsensor::ClusteringProcessorConfig* pClusteringConfig
    ) NN_NOEXCEPT
        : m_pClusteringConfig(pClusteringConfig)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << (m_pClusteringConfig->isExternalLightFilterEnabled ? "Enabled" : "Disabled");
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
        m_pClusteringConfig->isExternalLightFilterEnabled =
            !m_pClusteringConfig->isExternalLightFilterEnabled;
    }

private:
    nn::irsensor::ClusteringProcessorConfig* m_pClusteringConfig;
};

class ObjectCount : public ReadWriteBase
{
public:
    NN_IMPLICIT ObjectCount(int8_t* pObjectCount) NN_NOEXCEPT
        : m_pObjectCount(pObjectCount)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << (*m_pObjectCount);
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
    }

private:
    int8_t* m_pObjectCount;
};

ClusteringModeState::ClusteringModeState(
    IrSensorMode* pNextProcessor,
    int* pMenuSelection,
    nn::irsensor::IrCameraHandle irCameraHandle) NN_NOEXCEPT
:   IrSensorModeState(pNextProcessor, pMenuSelection, irCameraHandle)
, m_ClusteringProcessorState()
, m_ClusteringProcessorConfig()
, m_pObjectPixelCountMin(NULL)
, m_pObjectPixelCountMax(NULL)
, m_pObjectIntensityMin(NULL)
, m_pExternalLightFilter(NULL)
, m_pObjectCount(NULL)
{
    nn::irsensor::GetClusteringProcessorDefaultConfig(&m_ClusteringProcessorConfig);
    nn::irsensor::ClusteringProcessorConfig* pClusteringConfig = &m_ClusteringProcessorConfig;
    AddCommonReadWriteMenu(&m_ReadWriteMenu,
        &pClusteringConfig->irCameraConfig,
        nn::irsensor::ClusteringProcessorExposureTimeMin,
        nn::irsensor::ClusteringProcessorExposureTimeMax
    );

    m_pObjectPixelCountMin = new ObjectPixelCountMin(pClusteringConfig);
    MenuItem objectPixelCountMinMenuItem("Object Pixel Count Min",
        m_pObjectPixelCountMin
    );
    m_ReadWriteMenu.push_back(objectPixelCountMinMenuItem);

    m_pObjectPixelCountMax = new ObjectPixelCountMax(pClusteringConfig);
    MenuItem objectIntensityMaxMenuItem("Object Pixel Count Max",
        m_pObjectPixelCountMax
    );
    m_ReadWriteMenu.push_back(objectIntensityMaxMenuItem);

    m_pObjectIntensityMin = new ObjectIntensityMin(pClusteringConfig);
    MenuItem objectIntensityMinMenuItem("Object Intensity Min",
        m_pObjectIntensityMin
    );
    m_ReadWriteMenu.push_back(objectIntensityMinMenuItem);

    m_pExternalLightFilter = new ExternalLightFilter(pClusteringConfig);
    MenuItem externalLightFilterMenuItem("External Light Filter",
        m_pExternalLightFilter
    );
    m_ReadWriteMenu.push_back(externalLightFilterMenuItem);

    AddCommonReadOnlyMenu(
        &m_ReadOnlyMenu,
        &m_ClusteringProcessorState.samplingNumber,
        &m_ClusteringProcessorState.ambientNoiseLevel
    );
    int8_t* pObjectCount = &m_ClusteringProcessorState.objectCount;
    m_pObjectCount = new ObjectCount(pObjectCount);
    MenuItem objectCountMenuItem("Object Count",
        m_pObjectCount
    );
    m_ReadOnlyMenu.push_back(objectCountMenuItem);
}

ClusteringModeState::~ClusteringModeState() NN_NOEXCEPT
{
    delete m_pObjectPixelCountMin;
    delete m_pObjectPixelCountMax;
    delete m_pObjectIntensityMin;
    delete m_pExternalLightFilter;
    delete m_pObjectCount;
}

void ClusteringModeState::Start() NN_NOEXCEPT
{
    nn::irsensor::RunClusteringProcessor(m_IrCameraHandle, m_ClusteringProcessorConfig);
}

void ClusteringModeState::Update() NN_NOEXCEPT
{
    int count;
    nn::irsensor::GetClusteringProcessorStates(
        &m_ClusteringProcessorState,
        &count,
        1,
        m_IrCameraHandle
    );
    m_ClusteringProcessorState.objectCount =
        (m_ClusteringProcessorState.objectCount >=
            nn::irsensor::ClusteringProcessorObjectCountMax) ?
        0 : m_ClusteringProcessorState.objectCount;
}

namespace
{

void WriteClusteringProcessorState(
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::irsensor::ClusteringProcessorState* pClusteringState,
    const int screenIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pClusteringState);

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // DepthStencilTypeをDepthStencilType_DepthNoWriteTestへ設定
    pPrimitiveRenderer->SetDepthStencilState(
        pCommandBuffer,
        nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest
    );

    // グリッドを描画
    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;
    pPrimitiveRenderer->SetLineWidth(10.f);

    float xStart;
    float xEnd;
    float yStart;
    float yEnd;
    if (screenIndex == 0)
    {
        xStart = 0.45f, xEnd = 0.0f;
        yStart = 0.2f, yEnd = 0.8f;
    }
    else
    {
        xStart = 0.95f, xEnd = 0.5f;
        yStart = 0.2f, yEnd = 0.8f;
    }

    pPrimitiveRenderer->SetLineWidth(10.f);
    pPrimitiveRenderer->SetColor(Color::White);

    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    nn::util::VectorSet(&end, xStart, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xEnd, yEnd, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&end, xEnd, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    nn::util::VectorSet(&begin, xStart, yStart, 0.0f);
    pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);

    // 各面を描画
    for (int8_t i = 0; i < pClusteringState->objectCount; ++i)
    {
        const nn::irsensor::ClusteringData* pObject = &pClusteringState->objects[i];
        pPrimitiveRenderer->SetColor(Color::Green);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) *
                pObject->bound.x / static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) *
                pObject->bound.y / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        nn::util::VectorSet(&end,
            xStart + (xEnd - xStart) * (pObject->bound.x + pObject->bound.width) /
                static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * pObject->bound.y /
                static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * (pObject->bound.x + pObject->bound.width) /
                static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * (pObject->bound.y + pObject->bound.height) /
                static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&end,
            xStart + (xEnd - xStart) * pObject->bound.x /
                static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * (pObject->bound.y + pObject->bound.height) /
                static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
        nn::util::VectorSet(&begin,
            xStart + (xEnd - xStart) * pObject->bound.x /
                static_cast<float>(nn::irsensor::IrCameraImageWidth),
            yStart + (yEnd - yStart) * pObject->bound.y /
                static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    }
}


}

void ClusteringModeState::Render(
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer, int index) NN_NOEXCEPT
{
    WriteClusteringProcessorState(
        pPrimitiveRenderer,
        pCommandBuffer,
        &m_ClusteringProcessorState, index
    );
}
