﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cmath>
#include <cstdlib>

#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/os/os_Event.h>

#include "Demo1NpadPluginBase.h"
#include "Demo1ScenePluginBase.h"

class SndDemo : public ScenePluginBase
{
    NN_DISALLOW_COPY(SndDemo);
    NN_DISALLOW_MOVE(SndDemo);
public:
    SndDemo() NN_NOEXCEPT
    {
        m_pActiveEvent = new nn::os::EventType();
        nn::os::InitializeEvent(m_pActiveEvent, false, nn::os::EventClearMode_ManualClear);
    }
    virtual ~SndDemo() NN_NOEXCEPT NN_OVERRIDE
    {
        delete m_pActiveEvent;
    }

    virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE;

    virtual void Start() NN_NOEXCEPT NN_OVERRIDE;

    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE;

    virtual void Draw() NN_NOEXCEPT NN_OVERRIDE;

    virtual void Wait() NN_NOEXCEPT NN_OVERRIDE;

    virtual void SetActive(bool isActive) NN_NOEXCEPT NN_OVERRIDE;

private:
    static const int SeCount = 4;
    static const int BgmCount = 1;
    static const size_t ThreadStackSize = 128 * 1024;

    struct VoiceData
    {
        nn::audio::VoiceType voiceSine;
        void* pDataSine;
        nn::audio::WaveBuffer waveBufferSine;
    };

    struct BgmData
    {
        nn::audio::VoiceType voiceBgm[BgmCount];
        void* pDataBgm[BgmCount];
        nn::audio::WaveBuffer waveBufferBgm;
    };

    struct SeData
    {
        nn::audio::VoiceType voiceSe[SeCount];
        nn::audio::WaveBuffer waveBufferSe[SeCount];
        nn::audio::AdpcmHeaderInfo* pHeader[SeCount];
        void* pDataSe[SeCount];
    };

    struct SoundData
    {
        VoiceData voice;
        BgmData bgm;
        SeData se;
        nn::audio::MemoryPoolType waveBufferMemoryPool;
    };

private:
    void InitializeRenderer(nn::audio::AudioRendererParameter* pParameter,
                            nn::audio::AudioRendererHandle* pHandle,
                            nn::os::SystemEvent* pSystemEvent,
                            nn::audio::AudioRendererConfig* pConfig,
                            void** ppConfigBuffer,
                            void** ppWorkBuffer) NN_NOEXCEPT;
    void LoadSoundData(SoundData* pSoundData,
                       nn::audio::SubMixType* pSubMix,
                       nn::audio::FinalMixType* pFinalMix,
                       const int8_t mainBus[2],
                       int8_t auxBusA[2],
                       nn::audio::AudioRendererConfig* pConfig) NN_NOEXCEPT;
    void PlaySound(nn::audio::AudioRendererHandle handle,
                   nn::os::SystemEvent* pSystemEvent,
                   SoundData* pSoundData,
                   nn::audio::FinalMixType* pFinalMix,
                   const int8_t mainBus[2],
                   nn::audio::AudioRendererConfig* pConfig) NN_NOEXCEPT;
    void SoundThreadFuncImpl() NN_NOEXCEPT;
    static void SoundThreadFunc(void* pArg) NN_NOEXCEPT;

    void MakeCommand(int64_t frame, const char* pName) NN_NOEXCEPT;

    void GetKeyState() NN_NOEXCEPT;
    template<typename BUTTON>
    bool IsButtonPressed() const NN_NOEXCEPT;

    int GetAnalogStickL_X() const NN_NOEXCEPT;
    int GetAnalogStickR_X() const NN_NOEXCEPT;

    void PlaySoundEffect(
        nn::audio::VoiceType voiceSe[SeCount],
        nn::audio::WaveBuffer waveBufferSe[SeCount]
    ) const NN_NOEXCEPT;
    void SetSineVolume(nn::audio::VoiceType* pVoiceSine) const NN_NOEXCEPT;

    void SetBiquadFilter(
        nn::audio::VoiceType voiceBgm[BgmCount],
        int index
    ) const NN_NOEXCEPT;
    void OnOffBGM(
        nn::audio::VoiceType voiceBgm[BgmCount],
        int index
    ) const NN_NOEXCEPT;
    void SetBgmVolumeMix(
        nn::audio::VoiceType voiceBgm[BgmCount],
        nn::audio::FinalMixType* pFinalMix,
        const int8_t mainBus[2],
        int index
    ) const NN_NOEXCEPT;

    void ChangeSinePitch(
        nn::audio::VoiceType* pVoiceSine
    ) const NN_NOEXCEPT;

private:
    nn::os::EventType* m_pActiveEvent;
    nn::os::ThreadType m_SoundThread;
    NpadCommonState    m_CurrentNpadButtonState[NN_ARRAY_SIZE(NpadIds)];
    NpadCommonState    m_OldNpadButtonState[NN_ARRAY_SIZE(NpadIds)];
    // サウンドスレッド
    char*  m_pThreadStack;
};
