﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <cstdlib>
#include <new>

#include <nn/audio.h>
#include <nn/fs.h>
#include <nn/gfx.h>
#include <nn/hid.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

#include <nn/hid/hid_KeyboardKey.h>
#include <nn/util/util_Constant.h>
#include <nn/util/util_Color.h>

#include <nns/audio/audio_WavFormat.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeshRes.h>

#include "Demo1.h"
#include "Demo1Audio.h"
#include "Demo1Color.h"
#include "Demo1PluginManager.h"

namespace {

SET_PLUGIN( "AudioRenderer", SndDemo, PluginProperty_Drawable );

// サンプリングレートの選択
const int RenderRate = 48000;
const int RenderCount = (RenderRate / 200);

// PLAYファイルを選択
const int BgmCount = 1;
const int SeCount = 4;

// ファイルリストへの追加・削除
const char* BgmFileNames[BgmCount] =
{
    "asset:/res/SampleBgm0-2ch.wav",
};

const char* SeFileNames[SeCount] =
{
    "asset:/AudioCommon/SampleSe0.adpcm",
    "asset:/AudioCommon/SampleSe1.adpcm",
    "asset:/AudioCommon/SampleSe2.adpcm",
    "asset:/AudioCommon/SampleSe3.adpcm",
};

NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_EffectBufferPoolMemory[12 * 1024 * 1024];

nn::mem::StandardAllocator* g_pWaveBufferAllocator;
nn::mem::StandardAllocator* g_pEffectBufferAllocator;

char* g_MountRomCacheBuffer = NULL;

}

std::size_t GenerateSineWave(
    void** ppData,
    int sampleRate,
    int frequency,
    int sampleCount) NN_NOEXCEPT
{
    // g_WaveBufferAllocator が管理するメモリ領域はすべて waveBufferMemoryPool メモリプールに
    // 追加されています。
    int16_t* p = static_cast<int16_t*>(g_pWaveBufferAllocator->Allocate(
        sampleCount * sizeof(int16_t), nn::audio::BufferAlignSize));
    NN_ABORT_UNLESS_NOT_NULL(p);
    const float Pi = nn::util::FloatPi;
    for (int i = 0; i < sampleCount; ++i)
    {
        p[i] = static_cast<int16_t>(
            std::numeric_limits<int16_t>::max() * sinf(2 * Pi * frequency * i / sampleRate)
        );
    }
    *ppData = p;
    return sampleCount * sizeof(int16_t);
}

std::size_t ReadAdpcmFile(
    nn::audio::AdpcmHeaderInfo* pHeader,
    void** ppAdpcmData,
    const char* pFilename) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, pFilename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *ppAdpcmData = g_pWaveBufferAllocator->Allocate(
        static_cast<std::size_t>(size) - sizeof(adpcmheader), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*ppAdpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *ppAdpcmData,
        static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(pHeader, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

std::size_t ReadWavFile(
    nns::audio::WavFormat* pFormat,
    void** ppData,
    const char* pFilename) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, pFilename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *ppData = g_pWaveBufferAllocator->Allocate(static_cast<std::size_t>(size),
        nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*ppData);

    // DATA チャンクを読む必要がありますが、ここではそれが 1024 バイト以内に見つかると仮定しています
    const std::size_t WavHeaderDataSize = 1024;

    result = nn::fs::ReadFile(handle, 0, *ppData, WavHeaderDataSize);
    NN_ABORT_UNLESS(result.IsSuccess());

    nns::audio::WavResult wavResult
        = nns::audio::ParseWavFormat(pFormat, *ppData, WavHeaderDataSize);
    NN_ABORT_UNLESS_EQUAL(wavResult, nns::audio::WavResult_Success);
    // このサンプルでは 16bit PCM を仮定しています
    NN_ABORT_UNLESS_EQUAL(pFormat->bitsPerSample, 16);

    result = nn::fs::ReadFile(handle,
        static_cast<std::size_t>(pFormat->dataOffset),
        *ppData, static_cast<std::size_t>(pFormat->dataSize));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    return static_cast<std::size_t>(pFormat->dataSize);
}

void InitializeFileSystem() NN_NOEXCEPT
{
    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem() NN_NOEXCEPT
{
    nn::fs::Unmount("asset");

    delete[] reinterpret_cast<unsigned char*>(g_MountRomCacheBuffer);
    g_MountRomCacheBuffer = NULL;
}

// キーの状態を取得する
void SndDemo::GetKeyState() NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(NpadIds); i++)
    {
        // 操作形態(NpadStyleSet)を取得
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadIds[i]);

        NpadPluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
        if(pNpad == NULL)
        {
            continue;
        }

        m_OldNpadButtonState[i] = m_CurrentNpadButtonState[i];
        m_CurrentNpadButtonState[i] = pNpad->GetNpadButtonState(NpadIds[i]);
    }
}

// ボタンが押されたか判断する
template<typename BUTTON>
bool SndDemo::IsButtonPressed() const NN_NOEXCEPT
{
    bool isButtonPressed = false;
    for (int i = 0; i < NN_ARRAY_SIZE(NpadIds); i++)
    {
        // 操作形態(NpadStyleSet)を取得
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadIds[i]);

        NpadPluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
        if(pNpad == NULL)
        {
            continue;
        }

        if (m_CurrentNpadButtonState[i].buttons.Test<BUTTON>() &&
            !m_OldNpadButtonState[i].buttons.Test<BUTTON>())
        {
            isButtonPressed = true;
        }

    }
    return isButtonPressed;
}

int SndDemo::GetAnalogStickL_X() const NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(NpadIds); i++)
    {
        //現在有効な操作形態(NpadStyleSet)を取得
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadIds[i]);

        NpadPluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
        if(pNpad == NULL)
        {
            continue;
        }

        // ボタンの入力状態を取得
        NpadCommonState state = pNpad->GetNpadButtonState(NpadIds[i]);

        return state.analogStickL.x;
    }
    return 0;
}

int SndDemo::GetAnalogStickR_X() const NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(NpadIds); i++)
    {
        //現在有効な操作形態(NpadStyleSet)を取得
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadIds[i]);

        NpadPluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
        if(pNpad == NULL)
        {
            continue;
        }

        // ボタンの入力状態を取得
        NpadCommonState state = pNpad->GetNpadButtonState(NpadIds[i]);

        return state.analogStickR.x;
    }
    return 0;
}


void AcquireFinalAndSubMix(
    nn::audio::AudioRendererConfig* pConfig,
    nn::audio::FinalMixType* pFinalMix,
    nn::audio::SubMixType* pSubMix0,
    nn::audio::SubMixType* pSubMix1,
    int sampleRate
) NN_NOEXCEPT
{
    nn::audio::AcquireFinalMix(pConfig, pFinalMix, 6);
    nn::audio::AcquireSubMix(pConfig, pSubMix0, sampleRate, 1);
    nn::audio::AcquireSubMix(pConfig, pSubMix1, sampleRate, 1);
}

void SetSubMix(
    nn::audio::AudioRendererConfig* pConfig,
    nn::audio::SubMixType* pSubMix0,
    nn::audio::SubMixType* pSubMix1,
    nn::audio::FinalMixType* pFinalMix,
    const int8_t mainBus[2]
) NN_NOEXCEPT
{
    // SubMix(0) を FinalMix へ接続します
    nn::audio::SetSubMixDestination(pConfig, pSubMix0, pFinalMix);

    // SubMix(0) の 0 番目のバッファを FinalMix の mainBus[0]、mainBus[1] へ
    // ボリューム 0.5f でミックスするように設定します
    nn::audio::SetSubMixMixVolume(pSubMix0, pFinalMix, 0.5f, 0, mainBus[0]);
    nn::audio::SetSubMixMixVolume(pSubMix0, pFinalMix, 0.5f, 0, mainBus[1]);

    // SubMix(1) を SubMix(0) へ接続します
    nn::audio::SetSubMixDestination(pConfig, pSubMix1, pSubMix0);

    // SubMix(1) の 0 番目のバッファを SubMix(0) の 0 番目のバッファへ
    // ボリューム 0.5f でミックスするように設定します
    nn::audio::SetSubMixMixVolume(pSubMix1, pSubMix0, 0.5f, 0, 0);
}

// Delay と Mixer を設定する
void SetDelayAndMixer(
    nn::audio::AudioRendererConfig* pConfig,
    void* pDelayBuffer,
    nn::audio::DelayType* pDelay,
    nn::audio::BufferMixerType* pMixer1,
    nn::audio::FinalMixType* pFinalMix,
    size_t delaySize,
    const nn::TimeSpan delayTimeMax,
    const int8_t mainBus[2],
    const int8_t auxBusA[2]
) NN_NOEXCEPT
{
    // delay エフェクトで利用しうる最大のディレイ時間を指定します。
    int channelCount = 2;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddDelay(pConfig, pDelay, pDelayBuffer,
        delaySize, pFinalMix, delayTimeMax, 2));
    nn::audio::SetDelayInputOutput(pDelay, auxBusA, auxBusA, channelCount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddBufferMixer(pConfig, pMixer1, pFinalMix));
    nn::audio::SetBufferMixerInputOutput(pMixer1, auxBusA, mainBus, channelCount);

    // Delay パラメータを設定します。
    nn::audio::SetDelayEnabled(pDelay, true);
    nn::audio::SetDelayTime(pDelay, nn::TimeSpan::FromMilliSeconds(100));
    nn::audio::SetDelayFeedbackGain(pDelay, 0.7f);
    nn::audio::SetDelayChannelSpread(pDelay, 0.256f);
    nn::audio::SetDelayLowPassAmount(pDelay, 0.9f);
    nn::audio::SetDelayDryGain(pDelay, 0.5f);
    nn::audio::SetDelayInGain(pDelay, 0.4f);

    // BufferMixer パラメータを設定します。
    nn::audio::SetBufferMixerVolume(pMixer1, 0, 1.0f);
    nn::audio::SetBufferMixerVolume(pMixer1, 1, 1.0f);
}

// Voice で再生する波形を準備します。
void GenerateAndAddSineWave(
    void** ppDataSine,
    nn::audio::AudioRendererConfig* pConfig,
    nn::audio::VoiceType* pVoiceSine,
    nn::audio::SubMixType* pSubMix1,
    nn::audio::WaveBuffer* pWaveBufferSine) NN_NOEXCEPT
{
    const int SineSampleRate = 32000;
    const int SineFrequency = 440;
    const int SineSampleCount = 32000;
    std::size_t size = GenerateSineWave(ppDataSine, SineSampleRate, SineFrequency, SineSampleCount);

    nn::audio::AcquireVoiceSlot(pConfig, pVoiceSine, SineSampleRate, 1,
        nn::audio::SampleFormat_PcmInt16,
        nn::audio::VoiceType::PriorityHighest, NULL, 0);
    nn::audio::SetVoiceDestination(pConfig, pVoiceSine, pSubMix1);

    pWaveBufferSine->buffer = *ppDataSine;
    pWaveBufferSine->size = size;
    pWaveBufferSine->startSampleOffset = 0;
    pWaveBufferSine->endSampleOffset = SineSampleCount;
    pWaveBufferSine->loop = 0;
    pWaveBufferSine->isEndOfStream = false;

    // 波形を Voice に追加します。
    nn::audio::AppendWaveBuffer(pVoiceSine, pWaveBufferSine);
    nn::audio::AppendWaveBuffer(pVoiceSine, pWaveBufferSine);
    nn::audio::AppendWaveBuffer(pVoiceSine, pWaveBufferSine);
    nn::audio::AppendWaveBuffer(pVoiceSine, pWaveBufferSine);
    nn::audio::SetVoicePlayState(pVoiceSine, nn::audio::VoiceType::PlayState_Play);
    nn::audio::SetVoiceMixVolume(pVoiceSine, pSubMix1, 0.707f / 2, 0, 0);
}

// パラメータの初期化
void InitAudioRendererParameter(nn::audio::AudioRendererParameter* parameter) NN_NOEXCEPT
{
    nn::audio::InitializeAudioRendererParameter(parameter);
    parameter->sampleRate = RenderRate;
    parameter->sampleCount = RenderCount;
    parameter->mixBufferCount = 6 + 2;
    parameter->voiceCount = 24;
    parameter->subMixCount = 2;
    parameter->sinkCount = 1;
    parameter->effectCount = 3;
    parameter->performanceFrameCount = 0;

    // パラメータがシステムでサポートされているかどうかを確認します。
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(*parameter),
        "Invalid AudioRendererParameter specified."
    );
}

void InitMainBusAndAuxBus(
    int8_t mainBus[2],
    int8_t auxBusA[2]
) NN_NOEXCEPT
{
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 4;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 5;
    auxBusA[nn::audio::ChannelMapping_FrontLeft] = 0;
    auxBusA[nn::audio::ChannelMapping_FrontRight] = 1;
}

// BGM を読み込む
void ReadBgm(
    nn::audio::AudioRendererConfig* pConfig,
    void* pDataBgm[BgmCount],
    nn::audio::VoiceType voiceBgm[BgmCount],
    nn::audio::FinalMixType* pFinalMix,
    nn::audio::WaveBuffer* waveBufferBgm,
    const int8_t mainBus[2]) NN_NOEXCEPT
{
    for (int i = 0; i < BgmCount; ++i)
    {
        nns::audio::WavFormat format;
        std::size_t dataSize = ReadWavFile(&format, &pDataBgm[i], BgmFileNames[i]);

        // 読み込んだ WAV 形式ファイルのチャンネル数を指定してボイスを取得します。
        // マルチチャンネルのデータを読み込んだ場合には、チャンネル数分のボイスを消費することに
        // 注意してください。
        nn::audio::AcquireVoiceSlot(pConfig, &voiceBgm[i], format.sampleRate,
            format.channelCount, nn::audio::SampleFormat_PcmInt16,
            nn::audio::VoiceType::PriorityHighest, NULL, 0);
        nn::audio::SetVoiceDestination(pConfig, &voiceBgm[i], pFinalMix);

        waveBufferBgm->buffer = pDataBgm[i];
        waveBufferBgm->size = dataSize;
        waveBufferBgm->startSampleOffset = 0;
        waveBufferBgm->endSampleOffset =
            static_cast<int32_t>(dataSize / sizeof(int16_t)) / format.channelCount;
        waveBufferBgm->loop = true;
        waveBufferBgm->isEndOfStream = false;
        waveBufferBgm->pContext = NULL;
        waveBufferBgm->contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceBgm[i], waveBufferBgm);
        nn::audio::SetVoicePlayState(&voiceBgm[i], nn::audio::VoiceType::PlayState_Play);
        // ボイスの 0 チャンネルを mainBus[0] へ、1 チャンネルを mainBus[1] へ
        // 出力するミックスボリュームを設定します。
        nn::audio::SetVoiceMixVolume(&voiceBgm[i], pFinalMix, 0.5f, 0, mainBus[0]);
        nn::audio::SetVoiceMixVolume(&voiceBgm[i], pFinalMix, 0.5f, 1, mainBus[1]);

        // カットオフ 2048Hz のローパスフィルタを設定します
        nn::audio::BiquadFilterParameter firstFilter = {
            true,{ 720, 1439, 720 },{ 22684, -9350 }
        };
        nn::audio::SetVoiceBiquadFilterParameter(&voiceBgm[i], 0, firstFilter);
        // カットオフ 1024 Hz のハイパスフィルタを無効状態で設定します。
        nn::audio::BiquadFilterParameter secondFilter = {
            false,{ 14041, -28083, 14041 },{ 29547, -13563 }
        };
        nn::audio::SetVoiceBiquadFilterParameter(&voiceBgm[i], 1, secondFilter);
    }
}

void ReadSoundEffect(
    nn::audio::AudioRendererConfig* pConfig,
    void* pDataSe[SeCount],
    nn::audio::VoiceType voiceSe[SeCount],
    nn::audio::FinalMixType* pFinalMix,
    nn::audio::AdpcmHeaderInfo* pHeader[SeCount],
    nn::audio::WaveBuffer waveBufferSe[SeCount],
    const int8_t auxBusA[2]
) NN_NOEXCEPT
{
    for (int i = 0; i < SeCount; ++i)
    {
        pHeader[i] = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(
            g_pWaveBufferAllocator->Allocate(sizeof(nn::audio::AdpcmHeaderInfo),
                NN_ALIGNOF(nn::audio::AdpcmHeaderInfo))
            );
        std::size_t dataSeSize = ReadAdpcmFile(pHeader[i], &pDataSe[i], SeFileNames[i]);
        nn::audio::AcquireVoiceSlot(pConfig, &voiceSe[i], pHeader[i]->sampleRate, 1,
            nn::audio::SampleFormat_Adpcm,
            nn::audio::VoiceType::PriorityHighest,
            &pHeader[i]->parameter,
            sizeof(nn::audio::AdpcmParameter));
        nn::audio::SetVoiceDestination(pConfig, &voiceSe[i], pFinalMix);

        waveBufferSe[i].buffer = pDataSe[i];
        waveBufferSe[i].size = dataSeSize;
        waveBufferSe[i].startSampleOffset = 0;
        waveBufferSe[i].endSampleOffset = pHeader[i]->sampleCount;
        waveBufferSe[i].loop = false;
        waveBufferSe[i].isEndOfStream = false;
        waveBufferSe[i].pContext = &pHeader[i]->loopContext;
        waveBufferSe[i].contextSize = sizeof(nn::audio::AdpcmContext);

        nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
        nn::audio::SetVoicePlayState(&voiceSe[i], nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], pFinalMix, 0.707f / 2, 0, auxBusA[0]);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], pFinalMix, 0.707f / 2, 0, auxBusA[1]);
    }
}

// SE を再生する
void SndDemo::PlaySoundEffect(
    nn::audio::VoiceType voiceSe[SeCount],
    nn::audio::WaveBuffer waveBufferSe[SeCount]
) const NN_NOEXCEPT
{
    if (nn::os::TryWaitEvent(m_pActiveEvent))
    {
        if (IsButtonPressed<nn::hid::NpadButton::A>())
        {
            if (nn::audio::GetReleasedWaveBuffer(&voiceSe[0]))
            {
                nn::audio::AppendWaveBuffer(&voiceSe[0], &waveBufferSe[0]);
            }
        }
        if (IsButtonPressed<nn::hid::NpadButton::B>())
        {
            if (nn::audio::GetReleasedWaveBuffer(&voiceSe[1]))
            {
                nn::audio::AppendWaveBuffer(&voiceSe[1], &waveBufferSe[1]);
            }
        }
        if (IsButtonPressed<nn::hid::NpadButton::X>())
        {
            if (nn::audio::GetReleasedWaveBuffer(&voiceSe[2]))
            {
                nn::audio::AppendWaveBuffer(&voiceSe[2], &waveBufferSe[2]);
            }
        }
        if (IsButtonPressed<nn::hid::NpadButton::Y>())
        {
            if (nn::audio::GetReleasedWaveBuffer(&voiceSe[3]))
            {
                nn::audio::AppendWaveBuffer(&voiceSe[3], &waveBufferSe[3]);
            }
        }
    }
}

// サイン波の音量を操作します。
void SndDemo::SetSineVolume(nn::audio::VoiceType* pVoiceSine) const NN_NOEXCEPT
{
    if (nn::os::TryWaitEvent(m_pActiveEvent))
    {
        float sineVolume = nn::audio::GetVoiceVolume(pVoiceSine) +
            0.01f * GetAnalogStickL_X() / (::nn::hid::AnalogStickMax + 1);

        if (IsButtonPressed<nn::hid::NpadButton::Right>())
        {
            sineVolume += 0.10f;
        }
        if (IsButtonPressed<nn::hid::NpadButton::Left>())
        {
            sineVolume -= 0.10f;
        }
        // Voice のボリューム・ミックスボリュームは nn::audio::VoiceType::GetVolumeMin()
        // 以上 nn::audio::VoiceType::GetVolumeMax() 以下の値で指定できます。
        // このサンプルではノイズにならぬよう (0.0f, 2.0f) もしくは (0.0f, 1.0f) の範囲に
        // 制限しています。
        if (sineVolume < 2.0f && sineVolume > 0.0f)
        {
            nn::audio::SetVoiceVolume(pVoiceSine, sineVolume);
        }
    }
}

// Biquad Filter を設定する
void SndDemo::SetBiquadFilter(
    nn::audio::VoiceType voiceBgm[BgmCount],
    int index
) const NN_NOEXCEPT
{
    if (nn::os::TryWaitEvent(m_pActiveEvent) && IsButtonPressed<nn::hid::NpadButton::L>())
    {
        // 2 つ目の Biquad フィルタの設定の有効・無効を切り替えます。
        nn::audio::BiquadFilterParameter biquadFilterParameter(
            nn::audio::GetVoiceBiquadFilterParameter(&voiceBgm[index], 1)
        );
        biquadFilterParameter.enable = !biquadFilterParameter.enable;
        nn::audio::SetVoiceBiquadFilterParameter(&voiceBgm[index], 1, biquadFilterParameter);
    }
}

// BGM を On/Off する
void SndDemo::OnOffBGM(
    nn::audio::VoiceType voiceBgm[BgmCount],
    int index
) const NN_NOEXCEPT
{
    if (nn::os::TryWaitEvent(m_pActiveEvent) && IsButtonPressed<nn::hid::NpadButton::R>())
    {
        // BGM を ON/OFF します。
        switch (nn::audio::GetVoicePlayState(&voiceBgm[index]))
        {
        case nn::audio::VoiceType::PlayState_Play:
            nn::audio::SetVoicePlayState(&voiceBgm[index], nn::audio::VoiceType::PlayState_Pause);
            break;
        case nn::audio::VoiceType::PlayState_Pause:
        case nn::audio::VoiceType::PlayState_Stop:
            nn::audio::SetVoicePlayState(&voiceBgm[index], nn::audio::VoiceType::PlayState_Play);
            break;
        default:
            NN_ABORT("Unexpected play state\n");
            break;
        }
    }
}

// BGM のボリュームを設定する
void SndDemo::SetBgmVolumeMix(
    nn::audio::VoiceType voiceBgm[BgmCount],
    nn::audio::FinalMixType* pFinalMix,
    const int8_t mainBus[2],
    int index
) const NN_NOEXCEPT
{
    if (nn::os::TryWaitEvent(m_pActiveEvent))
    {
        // 左チャンネル
        float bgmLeftVolume = nn::audio::GetVoiceMixVolume(
            &voiceBgm[index], pFinalMix, 0, mainBus[0]);
        if (IsButtonPressed<nn::hid::NpadButton::ZL>())
        {
            bgmLeftVolume += 0.01f;
        }

        if (IsButtonPressed<nn::hid::NpadButton::ZR>())
        {
            bgmLeftVolume -= 0.01f;
        }

        if (bgmLeftVolume < 1.0f && bgmLeftVolume > 0.0f)
        {
            nn::audio::SetVoiceMixVolume(&voiceBgm[index], pFinalMix, bgmLeftVolume, 0, mainBus[0]);
        }

        // 右チャンネル
        float bgmRightVolume = 1.0f - bgmLeftVolume;

        if (bgmRightVolume < 1.0f && bgmRightVolume > 0.0f)
        {
            nn::audio::SetVoiceMixVolume(
                &voiceBgm[index], pFinalMix, bgmRightVolume, 1, mainBus[1]);
        }
    }
}

// サイン波のピッチを操作します。
void SndDemo::ChangeSinePitch(
    nn::audio::VoiceType* pVoiceSine
) const NN_NOEXCEPT
{
    if (nn::os::TryWaitEvent(m_pActiveEvent))
    {
        float sinePitch = nn::audio::GetVoicePitch(pVoiceSine) +
            0.01f * GetAnalogStickR_X() / (::nn::hid::AnalogStickMax + 1);
        if (IsButtonPressed<nn::hid::NpadButton::Up>())
        {
            sinePitch += 0.10f;
        }
        if (IsButtonPressed<nn::hid::NpadButton::Down>())
        {
            sinePitch -= 0.10f;
        }

        if (sinePitch < nn::audio::VoiceType::GetPitchMax() &&
            sinePitch > nn::audio::VoiceType::GetPitchMin())
        {
            nn::audio::SetVoicePitch(pVoiceSine, sinePitch);
        }
    }
}

void AddReleasedBuffer(nn::audio::VoiceType* pVoiceSine) NN_NOEXCEPT
{
    if (const nn::audio::WaveBuffer* pWaveBuffer = nn::audio::GetReleasedWaveBuffer(pVoiceSine))
    {
        nn::audio::AppendWaveBuffer(pVoiceSine, pWaveBuffer);
    }
}

void FreeBufferEffect(void *& pBuffer) NN_NOEXCEPT
{
    if (pBuffer)
    {
        g_pEffectBufferAllocator->Free(pBuffer);
        pBuffer = NULL;
    }
}

void FreeBufferWave(void *& pBuffer) NN_NOEXCEPT
{
    if (pBuffer)
    {
        g_pWaveBufferAllocator->Free(pBuffer);
        pBuffer = NULL;
    }
}

void FreeBufferWave(nn::audio::AdpcmHeaderInfo *& pBuffer) NN_NOEXCEPT
{
    if (pBuffer)
    {
        g_pWaveBufferAllocator->Free(pBuffer);
        pBuffer = NULL;
    }
}

void FreeBufferAllocator(void *& pBuffer) NN_NOEXCEPT
{
    if (pBuffer)
    {
        GetStandardAllocator().Free(pBuffer);
        pBuffer = NULL;
    }
}

void FreeBgm(void* pDataBgm[BgmCount]) NN_NOEXCEPT
{
    for (int i = 0; i < BgmCount; ++i)
    {
        FreeBufferWave(pDataBgm[i]);
    }
}

void FreeDataSeAndHeader(
    void* pDataSe[SeCount],
    nn::audio::AdpcmHeaderInfo* pHeader[SeCount]) NN_NOEXCEPT
{
    for (int i = 0; i < SeCount; ++i)
    {
        FreeBufferWave(pDataSe[i]);
        FreeBufferWave(pHeader[i]);
    }
}

void SndDemo::InitializeRenderer(nn::audio::AudioRendererParameter* pParameter,
                                 nn::audio::AudioRendererHandle* pHandle,
                                 nn::os::SystemEvent* pSystemEvent,
                                 nn::audio::AudioRendererConfig* pConfig,
                                 void** ppConfigBuffer,
                                 void** ppWorkBuffer) NN_NOEXCEPT
{
    // レンダラのパラメータを指定します。
    InitAudioRendererParameter(pParameter);

    // レンダラのワークバッファを準備します。
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(*pParameter);
    *ppWorkBuffer = GetStandardAllocator().Allocate(
        workBufferSize, nn::os::MemoryPageSize
    );
    NN_ABORT_UNLESS_NOT_NULL(*ppWorkBuffer);

    // レンダラを初期化します。
    NN_ABORT_UNLESS( nn::audio::OpenAudioRenderer(pHandle, pSystemEvent, *pParameter,
                      *ppWorkBuffer, workBufferSize).IsSuccess(), "Failed to open AudioRenderer"
    );

    // AudioRendererConfig を初期化します。
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(*pParameter);
    *ppConfigBuffer = GetStandardAllocator().Allocate(
        configBufferSize, nn::os::MemoryPageSize
    );
    NN_ABORT_UNLESS_NOT_NULL(*ppConfigBuffer);
    nn::audio::InitializeAudioRendererConfig(pConfig, *pParameter, *ppConfigBuffer, configBufferSize);

    return;
}

void SndDemo::LoadSoundData(SoundData* pSoundData,
                            nn::audio::SubMixType* pSubMix,
                            nn::audio::FinalMixType* pFinalMix,
                            const int8_t mainBus[2],
                            int8_t auxBusA[2],
                            nn::audio::AudioRendererConfig* pConfig) NN_NOEXCEPT
{
    // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
    NN_ABORT_UNLESS(AcquireMemoryPool(pConfig, &pSoundData->waveBufferMemoryPool, g_WaveBufferPoolMemory,
                           sizeof(g_WaveBufferPoolMemory)));
    NN_ABORT_UNLESS(RequestAttachMemoryPool(&pSoundData->waveBufferMemoryPool));

    // Voice で再生する波形を準備します。
    GenerateAndAddSineWave(&pSoundData->voice.pDataSine,
                           pConfig,
                           &pSoundData->voice.voiceSine,
                           pSubMix,
                           &pSoundData->voice.waveBufferSine);

    // BGM を読み込みます。
    ReadBgm(pConfig,
            pSoundData->bgm.pDataBgm,
            pSoundData->bgm.voiceBgm,
            pFinalMix,
            &pSoundData->bgm.waveBufferBgm,
            mainBus);

    // SE を読み込みます。
    ReadSoundEffect(pConfig,
                    pSoundData->se.pDataSe,
                    pSoundData->se.voiceSe,
                    pFinalMix,
                    pSoundData->se.pHeader,
                    pSoundData->se.waveBufferSe,
                    auxBusA);

    return;
}

void SndDemo::PlaySound(nn::audio::AudioRendererHandle handle,
                        nn::os::SystemEvent* pSystemEvent,
                        SoundData* pSoundData,
                        nn::audio::FinalMixType* pFinalMix,
                        const int8_t mainBus[2],
                        nn::audio::AudioRendererConfig* pConfig) NN_NOEXCEPT
{
    // キーの状態を取得します。
    GetKeyState();

    // 波形再生が終わるまで待ちつつ、パラメータを更新します。
    while (!nn::os::TryWaitEvent(&GetGlobalEvent()))
    {
        pSystemEvent->Wait();

        // キーの状態を取得します。
        GetKeyState();

        // SE を再生します（同じ SE を多重再生しません）。
        PlaySoundEffect(pSoundData->se.voiceSe, pSoundData->se.waveBufferSe);

        // サイン波の音量を操作します。
        SetSineVolume(&pSoundData->voice.voiceSine);

        for (int i = 0; i < BgmCount; ++i)
        {
            // BiquadFilter を設定する
            SetBiquadFilter(pSoundData->bgm.voiceBgm, i);

            // BGM を On/Off する
            OnOffBGM(pSoundData->bgm.voiceBgm, i);

            // BGM のボリュームを設定する
            SetBgmVolumeMix(pSoundData->bgm.voiceBgm, pFinalMix, mainBus, i);
        }

        // サイン波のピッチを操作します。
        ChangeSinePitch(&pSoundData->voice.voiceSine);

        // 空きバッファを取得して追加する
        AddReleasedBuffer(&pSoundData->voice.voiceSine);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handle, pConfig));
    }
}

void SndDemo::SoundThreadFuncImpl() NN_NOEXCEPT
{
    g_pWaveBufferAllocator->Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    g_pEffectBufferAllocator->Initialize(
        g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));

    InitializeFileSystem();

    // レンダラを初期化します。
    nn::audio::AudioRendererParameter parameter;
    nn::audio::AudioRendererHandle handle;
    nn::os::SystemEvent systemEvent;
    nn::audio::AudioRendererConfig config;
    void* pConfigBuffer;
    void* pWorkBuffer;
    InitializeRenderer(&parameter,&handle,&systemEvent,&config,&pConfigBuffer,&pWorkBuffer);

    // ミックスバッファとオーディオバスの関係を定義します。
    int8_t mainBus[2];
    int8_t auxBusA[2];
    InitMainBusAndAuxBus(mainBus, auxBusA);

    nn::audio::FinalMixType finalMix;
    nn::audio::SubMixType subMix0;
    nn::audio::SubMixType subMix1;
    AcquireFinalAndSubMix(&config, &finalMix, &subMix0, &subMix1, parameter.sampleRate);

    int channelCount = 2;
    // レンダラの出力先を用意します。
    nn::audio::DeviceSinkType deviceSink;
    // オーディオ出力デバイスへの入力を設定します。
    // mainBus に指定したミックスバッファのインデックスに応じて出力チャンネルが決定されます。
    // mainBus[nn::audio::ChannelMapping_FrontLeft] が L チャンネルに、
    // mainBus[nn::audio::ChannelMapping_FrontRight] が R チャンネルにそれぞれ出力されます。
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix,
                                                 mainBus, channelCount, "MainAudioOut"));

    // SubMix の接続
    SetSubMix(&config, &subMix0, &subMix1, &finalMix, mainBus);

    // エフェクトバッファ用のメモリプールを準備します。
    // MemoryPool は nn::audio::AcquireMemoryPool() で渡される address 、 size の領域を
    // 書き換えることはありません。
    // そのため g_EffectBufferPoolMemory として準備した領域をすべて、メモリプールに
    // 利用することができます。
    // またこのサンプルでは WaveBuffer, Effect とメモリの用途別にアロケータを準備しています。
    // MemoryPool はその領域全体を「アタッチ」「デタッチ」します。
    // MemoryPool の操作に必要な粒度に応じてメモリ領域を分けておくことをおすすめします。
    // 詳しくは @link ../../../Samples/Sources/Applications/AudioMemoryPool
    // Samples/Sources/Applications/AudioMemoryPool @endlink を参照してください。
    nn::audio::MemoryPoolType effectBufferPool;
    NN_ABORT_UNLESS(AcquireMemoryPool(&config, &effectBufferPool, g_EffectBufferPoolMemory,
                                sizeof(g_EffectBufferPoolMemory)));
    // メモリプールをアタッチします。設定は次の nn::audio::RequestUpdateAudioRenderer() が
    // 呼ばれた以降に反映されます。
    NN_ABORT_UNLESS(RequestAttachMemoryPool(&effectBufferPool));

    // Delay と BufferMixer を用意し、Delay, BufferMixer の順に登録します。
    nn::audio::DelayType delay;

    // delay エフェクトで利用しうる最大のディレイ時間を指定します。
    const nn::TimeSpan DelayTimeMax = nn::TimeSpan::FromSeconds(1);
    size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(DelayTimeMax,
        parameter.sampleRate, 2);

    // エフェクトバッファを取得します。g_EffectBufferAllocator が管理する全領域は
    // effectBufferPool メモリプールに登録してあります。
    void* pDelayBuffer = g_pEffectBufferAllocator->Allocate(delaySize, nn::audio::BufferAlignSize);
    nn::audio::BufferMixerType mixer1;

    // DELAY と Mixer を設定します。
    SetDelayAndMixer(&config, pDelayBuffer, &delay, &mixer1, &finalMix, delaySize,
        DelayTimeMax, mainBus, auxBusA );

    // 設定したパラメータをレンダラに反映させます。
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handle, &config));

    // レンダリングを開始します。
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handle));

    // サウンドデータを読み込みます。
    SoundData soundData;
    LoadSoundData(&soundData, &subMix1, &finalMix, mainBus, auxBusA, &config);

    // サウンドを再生します。
    PlaySound(handle, &systemEvent, &soundData, &finalMix, mainBus, &config);

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);

    // メモリを解放します。
    FreeBufferEffect(pDelayBuffer);
    FreeBufferWave(soundData.voice.pDataSine);
    FreeBgm(soundData.bgm.pDataBgm);
    FreeDataSeAndHeader(soundData.se.pDataSe, soundData.se.pHeader);
    FreeBufferAllocator(pConfigBuffer);
    FreeBufferAllocator(pWorkBuffer);
    FinalizeFileSystem();
}

void SndDemo::SoundThreadFunc(void* pArg) NN_NOEXCEPT
{
    g_pWaveBufferAllocator = new nn::mem::StandardAllocator();
    g_pEffectBufferAllocator = new nn::mem::StandardAllocator();

    SndDemo * pSndDemo = reinterpret_cast<SndDemo*>(pArg);
    pSndDemo->SoundThreadFuncImpl();

    g_pWaveBufferAllocator->Finalize();
    g_pEffectBufferAllocator->Finalize();
    delete g_pWaveBufferAllocator;
    delete g_pEffectBufferAllocator;
}

void SndDemo::Initialize() NN_NOEXCEPT
{
    m_pThreadStack = reinterpret_cast<char*>(
        GetStandardAllocator().Allocate(
            ThreadStackSize,
            nn::os::ThreadStackAlignment
        )
    );

    // スレッドを作成
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &m_SoundThread, SoundThreadFunc, this,
        m_pThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
}

void SndDemo::Start() NN_NOEXCEPT
{
    // スレッドをStart
    nn::os::StartThread(&m_SoundThread);
}

void SndDemo::Finalize() NN_NOEXCEPT
{
    // サウンドスレッドの破棄
    nn::os::DestroyThread(&m_SoundThread);

    GetStandardAllocator().Free(m_pThreadStack);
}

void SndDemo::Wait() NN_NOEXCEPT
{
    nn::os::WaitThread(&m_SoundThread);
}

void SndDemo::MakeCommand(int64_t frame, const char* pName) NN_NOEXCEPT
{
    NN_UNUSED(frame);

    nn::gfx::util::DebugFontTextWriter& debugFontTextWriter = m_pGraphicsSystem->GetDebugFont();
    nn::gfx::CommandBuffer& commandBuffer = m_pGraphicsSystem->GetCommandBuffer();

    debugFontTextWriter.SetScale(NormalFontScaleX, NormalFontScaleY);

    // テスト用にテキスト出力
    float displayWidth = static_cast<float>(m_pGraphicsSystem->GetDisplayWidth());
    float displayHeight = static_cast<float>(m_pGraphicsSystem->GetDisplayHeight());
    float lineStep = displayHeight / 40.0f;
    //62文字
    float widthOffset = displayWidth / 2.0f + 70.0f;
    float heightOffset = displayHeight / 2.0f + lineStep;

    debugFontTextWriter.SetTextColor(nn::util::Color4u8::White());

    const char * Descriptions[] = {
        "                                    Demo1 AudioRenderer",
        "-----------------------------------------------------------------------------",
        "[A]                     StartSound                                (SampleSe0)",
        "[B]                     StartSound                                (SampleSe1)",
        "[X]                     StartSound                                (SampleSe2)",
        "[Y]                     StartSound                                (SampleSe3)",
        "[L]                     Toggle Biquad filter enabled         (SampleBgm0-2ch)",
        "[R]                     Toggle BGM Play state               (SampleBgm0-2ch)",
        "[AnalogStickL]     ControlVolume                            (SineWave)",
        "[Left][Right]        ControlVolume                            (SineWave)",
        "[AnalogStickR]     ControlPitch                               (SineWave)",
        "[Up][Down]         ControlPitch                               (SineWave)",
        "[ZLeft/U]            ControlVolume                            (SampleBgm0-2ch)",
        "[ZRight/V]           ControlVolume                            (SampleBgm0-2ch)",
        "[Plus + Minus]     Shut down sample program",
        "-----------------------------------------------------------------------------"
    };

    for( int i = 0; i < NN_ARRAY_SIZE(Descriptions); i++ )
    {
        debugFontTextWriter.SetCursor(widthOffset, heightOffset + lineStep * (i + 1));
        debugFontTextWriter.Print(Descriptions[i]);
    }

    //!< 共通操作説明を描画します。
    WriteCommonGuide(&debugFontTextWriter, pName);

    // 負荷メータを表示する
    DrawLoadMeter();

    // テキストを描画
    debugFontTextWriter.Draw(&commandBuffer);
}

void SndDemo::Draw() NN_NOEXCEPT
{
    static int64_t s_Frame = 0;
    MakeCommand(s_Frame, m_Name.c_str());
    s_Frame++;
}

void SndDemo::SetActive(bool isActive) NN_NOEXCEPT
{
    if (isActive)
    {
        nn::os::SignalEvent(m_pActiveEvent);
    }
    else
    {
        nn::os::ClearEvent(m_pActiveEvent);
    }
}

