﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{DebugFontSimple.cpp,PageSampleDebugFontSimple}
 *
 * @brief
 *  デバッグフォント描画クラスの使い方を示したサンプルプログラム
 */

/**
 * @page PageSampleDebugFontSimple DebugFontSimple
 * @tableofcontents
 *
 * @image html Applications\DebugFontSimple\debugfontsimple.png
 *
 * @brief
 *  デバッグフォント描画クラスの使い方を示したサンプルプログラムです。
 *
 * @section PageSampleDebugFontSimple_SectionBrief 概要
 *  DebugFontTextWriter クラスを使用してフォント表示を行うシンプルなサンプルです。
 *
 * @section PageSampleDebugFontSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/DebugFontSimple Samples/Sources/Applications/DebugFontSimple @endlink 以下にあります。
 *
 * @section PageSampleDebugFontSimple_SectionNecessaryEnvironment 必要な環境
 *  画面表示が利用可能である必要があります。
 *
 * @section PageSampleDebugFontSimple_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleDebugFontSimple_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleDebugFontSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleDebugFontSimple_SectionDetail 解説
 * DebugFontTextWriter クラスを使用してフォント表示を行う最も基本的なデモです。
 *
 * このサンプルプログラムの処理の流れは以下の通りです。
 *
 * - グラフィックスフレームワークを初期化
 *     - VI レイヤを初期化
 *     - デバイスを初期化
 *     - メモリプールを初期化
 *     - スワップチェーンを初期化
 *     - キューを初期化
 *     - ビューポートシザーを初期化
 *     - コマンドバッファを初期化
 *     - デスクリプタプールを初期化
 * - デバッグフォント描画クラスを初期化
 * - ループ開始
 * - フォント文字を描画
 * - コマンドリストを作成
 * - コマンドリストを実行
 * - ディスプレイへプレゼンテーション
 * - ループ開始に戻る
 * - 各種オブジェクトを破棄
 */

// NintendoSDK のヘッダファイルをインクルードする前に、NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET マクロを
// 定義することで、DebugFontWriter::Print() の入力文字コードを Windows のロケールのデフォルト
// (日本語の場合、CP932)に変更できます。
#define NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET

#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

//------------------------------------------------------------------------------
//  プリント 関数
//------------------------------------------------------------------------------
void Print(nn::gfx::util::DebugFontTextWriter& writer, int frame)
{
    nn::util::Color4u8Type color0 = { { 255, 255, 255, 255 } };
    nn::util::Color4u8Type color1 = { { 255, 128, 0, 255 } };
    nn::util::Color4u8Type color2 = { { 0, 255, 255, 255 } };

    writer.SetScale(1.0f, 1.0f);

    writer.SetTextColor(color0);
    writer.SetCursor(0, 0);
    writer.Print("DebugFontSimple");

    if ((frame % 60) < 30) {
        writer.Print("■");
    }

    writer.SetTextColor(color1);
    writer.SetCursor(100, 100);
    writer.Print("あいうえおかきくけこ\n");
    writer.Print("さしすせそたちつてと\n");
    writer.Print("なにぬねのはひふへほ\n");
    writer.Print("まみむめもやゆよ\n");
    writer.Print("らりるれろわゐゑをん\n");

    writer.SetTextColor(color0);
    writer.SetCursor(200, 250);
    writer.Print("ABCDEFGHIJKLMNOPQRSTUVWXYZ\n");
    writer.Print("abcdefghijklmnopqrstuvwxyz 0123456789");

    writer.SetFixedWidthEnabled(true);
    writer.SetFixedWidth(18);
    writer.SetCursor(200, 320);
    writer.Print("ABCDEFGHIJKLMNOPQRSTUVWXYZ\n");
    writer.Print("abcdefghijklmnopqrstuvwxyz 0123456789");
    writer.SetFixedWidthEnabled(false);

    writer.SetTextColor(color2);
    writer.SetCursor(100, 400);
    writer.SetScale(0.5f, 0.5f);
    writer.Print("Scale: %1.1f\n", 0.5f);
    writer.SetScale(1.0f, 1.0f);
    writer.Print("Scale: %1.1f\n", 1.0f);
    writer.SetScale(2.0f, 2.0f);
    writer.Print("Scale: %1.1f\n", 2.0f);
    writer.SetScale(4.0f, 4.0f);
    writer.Print("Scale: %1.1f\n", 4.0f);
}

//------------------------------------------------------------------------------
//  コマンド生成
//------------------------------------------------------------------------------
struct MakeCommandUserData
{
    nn::gfx::util::DebugFontTextWriter* pWriter;
    int frame;
};

void MakeCommand(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    MakeCommandUserData* pData = reinterpret_cast<MakeCommandUserData*>(pUserData);

    // フォント表示
    Print(*pData->pWriter, pData->frame);

    // コマンド生成
    pGraphicsFramework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

        // レンダーターゲット、ビューポートシザー設定
        nn::gfx::ColorTargetView* target = pGraphicsFramework->GetColorTargetView();
        rootCommandBuffer->ClearColor(target, 0.1f, 0.1f, 0.1f, 1.0f, nullptr);
        rootCommandBuffer->SetRenderTargets(1, &target, nullptr);
        rootCommandBuffer->SetViewportScissorState(pGraphicsFramework->GetViewportScissorState());

        // デバッグフォント用のコマンド生成
        pData->pWriter->Draw(rootCommandBuffer);
    }
    pGraphicsFramework->EndFrame(bufferIndex, true);
}

//------------------------------------------------------------------------------
//  メイン 関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    // FS を使用する場合はフレームワークよりも前に初期化します

    // フレームワーク初期化
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(GraphicsSystemMemorySize);

    const int BufferCount = 2;
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetBufferCount(BufferCount);
    fwInfo.SetSwapChainBufferCount(BufferCount);
    nns::gfx::GraphicsFramework gfw;
    gfw.Initialize(fwInfo);

    // デバッグフォント初期化
    const bool userMemoryPoolEnable = true;    // true にすると、ユーザーのメモリプールを使用します
    const int charCountMax = 1024;
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(charCountMax);
    info.SetBufferCount(BufferCount);
    info.SetUserMemoryPoolEnabled(userMemoryPoolEnable);

    size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(gfw.GetDevice(), info);
    nn::util::BytePtr debugFontHeap(new uint8_t[debugFontHeapSize]);

    size_t debugFontMemoryPoolSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemoryPoolSize(gfw.GetDevice(), info);

    nns::gfx::GraphicsFramework::MemoryPoolType type = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
    nn::gfx::MemoryPool* pMemoryPool = gfw.GetMemoryPool(type);
    ptrdiff_t memoryPoolOffset = nn::gfx::util::MemoryPoolAllocator::InvalidOffset;
    nn::gfx::util::DebugFontTextWriter writer;
    if (NN_STATIC_CONDITION(userMemoryPoolEnable)) {
        memoryPoolOffset = gfw.AllocatePoolMemory(type, debugFontMemoryPoolSize, 1);

        writer.Initialize(gfw.GetDevice(), info, debugFontHeap.Get(), debugFontHeapSize,
            pMemoryPool, memoryPoolOffset, debugFontMemoryPoolSize);
    }
    else {
        writer.Initialize(gfw.GetDevice(), info, debugFontHeap.Get(), debugFontHeapSize, nullptr, 0, 0);
    }

    int textureDescriptorIndex = gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
    int samplerDescriptorIndex = gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);

    writer.SetDisplayWidth(gfw.GetDisplayWidth());
    writer.SetDisplayHeight(gfw.GetDisplayHeight());
    writer.SetTextureDescriptor(gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView), textureDescriptorIndex);
    writer.SetSamplerDescriptor(gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler), samplerDescriptorIndex);


    // GraphicsFramework によるフレーム処理用の設定
    MakeCommandUserData userData;
    userData.pWriter = &writer;
    userData.frame = 0;

    gfw.SetMakeCommandCallback(MakeCommand, &userData);


    // 毎フレームのレンダリング
    for (int frame = 0; frame < 60 * 10; ++frame)
    {
        gfw.ProcessFrame();
        userData.frame++;
    }
    gfw.QueueFinish();


    // デバッグフォント終了
    writer.Finalize();
    delete[] reinterpret_cast<uint8_t*>(debugFontHeap.Get());
    debugFontHeap.Reset(nullptr);

    if (NN_STATIC_CONDITION(userMemoryPoolEnable)) {
        gfw.FreePoolMemory(type, memoryPoolOffset);
    }
    gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, textureDescriptorIndex);
    gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, samplerDescriptorIndex);

    // フレームワーク終了
    gfw.Finalize();
}
