﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nns/console/console_ConsoleManager.h>
#include <nns/console/console_InputEventListener.h>
#include <nns/console/console_InputManager.h>
#include <nns/console/console_SimpleConsole.h>

// USB Keyboard の入力をそのまま画面に表示するプログラム
class TypeWriter
    : nns::console::PadEventListenerBase
    , nns::console::KeyboardEventListenerBase
{
private:
    bool m_IsShifted;
    bool m_IsCapsLocked;

    nns::console::SimpleConsole m_StatusConsole;
    nns::console::SimpleConsole m_LogConsole;

    // 画面表示の更新
    void PrintStatus() NN_NOEXCEPT;
    void Echo(const nn::hid::KeyboardKeySet& key) NN_NOEXCEPT;
    void Reset() NN_NOEXCEPT;

    // 内部状態の変更
    void EnableShift() NN_NOEXCEPT;
    void DisableShift() NN_NOEXCEPT;
    void EnableCapsLock() NN_NOEXCEPT;
    void DisableCapsLock() NN_NOEXCEPT;

    // 入力イベントのハンドラ
    virtual void OnKeyPressedImpl(const nn::hid::KeyboardKeySet& key, const nn::hid::KeyboardModifierSet& modifier) NN_NOEXCEPT final NN_OVERRIDE;
    virtual void OnKeyReleasedImpl(const nn::hid::KeyboardKeySet& key, const nn::hid::KeyboardModifierSet& modifier) NN_NOEXCEPT final NN_OVERRIDE;
    virtual void OnKeyRepeatedImpl(const nn::hid::KeyboardKeySet& key, const nn::hid::KeyboardModifierSet& modifier) NN_NOEXCEPT final NN_OVERRIDE;
    virtual void OnButtonReleasedImpl(const nns::console::PadButtonSet& buttons) NN_NOEXCEPT final NN_OVERRIDE;

public:
    TypeWriter() NN_NOEXCEPT;

    // 引数で受け取った ConsoleManager に Console を配置する
    template <typename Settings>
    void Initialize(nns::console::ConsoleManager<Settings>* pConsoleManager, nns::console::DefaultInputManager* pInputManager) NN_NOEXCEPT
    {
        const auto Margin = 40;
        const auto StatusWidth = 320;
        const auto LogWidth = static_cast<int>(Settings::ScreenWidth) - StatusWidth - Margin;
        NN_SDK_ASSERT(LogWidth > 0);
        const auto Height = static_cast<int>(Settings::ScreenHeight);

        // 画面上にコンソールオブジェクトを配置する
        // {x, y, width, height} の順。単位は px。
        pConsoleManager->Register(&m_LogConsole, 0, 0, LogWidth, Height);
        pConsoleManager->Register(&m_StatusConsole, LogWidth + Margin, 0, StatusWidth, Height);

        // 入力イベントのハンドラを登録
        pInputManager->SetPadEventListener(this);
        pInputManager->SetKeyboardEventListener(this);

        // 初期状態を表示
        PrintStatus();
    }
};
