﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{CodecAacDecoder.cpp,PageSampleCodecAacDecoder}
 *
 * @brief
 * Aac デコーダのサンプルプログラム
 */

/**
 * @page PageSampleCodecAacDecoder Aac デコーダ
 * @tableofcontents
 *
 * @brief
 * Aac デコーダのサンプルプログラムの解説です。
 *
 * @section PageSampleCodecAacDecoder_SectionBrief 概要
 * Aac デコーダを用いて、Aac データをデコードしながらオーディオ再生を行うサンプルです。
 * オーディオ再生にはオーディオレンダラを利用します。
 *
 * @section PageSampleCodecAacDecoder_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/CodecAacDecoder
 * Samples/Sources/Applications/CodecAacDecoder @endlink 以下にあります。
 *
 * @section PageSampleCodecAacDecoder_SectionNecessaryEnvironment 必要な環境
 * オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleCodecAacDecoder_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、一定の時間 BGM を再生し、自動的に終了します。
 *
 * @section PageSampleCodecAacDecoder_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleCodecAacDecoder_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleCodecAacDecoder_SectionDetail 解説
 *
 * @subsection PageSampleCodecAacDecoder_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  CodecAacDecoder.cpp
 *  @includelineno CodecAacDecoder.cpp
 *
 * @subsection PageSampleCodecAacDecoder_SectionSampleDetail サンプルプログラムの解説
 *
 * このサンプルプログラムは、あらかじめエンコードされた Aac データをデコードしながら、オーディオ再生を行うものです。
 * このサンプルでは m4a フォーマットデータと adts フォーマットデータのデコード例を示しており、それぞれを ADTS というマクロで切り替えることが可能です。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - ファイルシステムの初期化
 * - Aac データの読み込み
 * - Aac デコーダの初期化
 * - オーディオレンダラの初期化、および開始
 * - Aac デコーダの実行と、デコード結果の再生
 * - オーディオレンダラの終了
 * - Aac デコーダの終了
 *
 * 最初にファイルシステムの初期化を行います。
 * リソースファイルはビルド時にコンバートされるものを利用します。
 * リソースディレクトリの場所は適宜変更して利用してください。
 *
 * Aac データをファイルから読み込みます。
 *
 * ヘッダ情報をもとに Aac デコーダを初期化します。
 *
 * 初期化関数 nn::codec::AacDecoder::Initialize() にはサンプルレートとチャンネル数、Mpeg-4 Audio Object Type, さらにそれらによってサイズが決まるワークバッファを与える必要があります。
 *
 * ワークバッファのサイズは nn::codec::AacDecoder::GetWorkBufferSize() により取得可能です。
 *
 * オーディオ出力を準備します。
 * オーディオ出力の詳細に関しては @link ../../../Samples/Sources/Applications/AudioOut
 * Samples/Sources/Applications/AudioOut @endlink サンプルおよび関連する API リファレンスを参照してください。
 *
 * メインループでは、ストリーム再生のような処理を行います。
 * 複数の PCM バッファを用意し、 nn::audio::AudioOut 経由で再生しながら、
 * 再生が終わったバッファに対して順次 Aac データのデコードを行います。
 *
 * Aac データをすべてデコードし、再生を終えると、オーディオ出力を停止し、
 * オーディオ出力および Aac デコーダを終了し、メモリの破棄を行います。
 */

#include <cstdlib>  // std::malloc, std::free
#include <new>  // std::nothrow

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/audio.h>
#include <nn/codec.h>
#include <nn/fs.h>
#include <nn/mem.h>

#include <nns/nns_Log.h>

// #define ADTS

#if defined(ADTS)
#include "AdtsHeaderReader.h"
#else  // defined(ADTS)
#include "Mp4BoxHeaderReader.h"
#include "Mp4BoxStsdReader.h"
#endif  // defined(ADTS)

namespace {

char g_HeapBuffer[32 * 1024 * 1024];

char* g_MountRomCacheBuffer = NULL;

void InitializeFileSystem()
{
    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

}  // namespace anonymous

extern "C" void nnMain()
{
    nn::mem::StandardAllocator allocator(g_HeapBuffer, sizeof(g_HeapBuffer));

    // AacDecoder 用のワークバッファを初期化します。
    auto aacWorkBufferSize = nn::codec::GetAacDecoderWorkBufferSize(48000, 2, nn::codec::Mpeg4AudioObjectType_AacLc);
    auto aacWorkBuffer = allocator.Allocate(aacWorkBufferSize);

#if defined(ADTS)
    // AAC ファイルを読み込みます。
    uint8_t* aacData;
    int64_t size;
    {
        InitializeFileSystem();
        const char Filename[] = "asset:/SampleBgm0-2ch.adts.aac";
        nn::fs::FileHandle handle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, Filename, nn::fs::OpenMode_Read));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&size, handle));
        aacData = static_cast<uint8_t*>(allocator.Allocate(static_cast<size_t>(size)));
        NN_ASSERT(aacData);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, aacData, static_cast<std::size_t>(size)));
        nn::fs::CloseFile(handle);
        FinalizeFileSystem();
    }

    // ADTS フォーマットを解析し、AacDecoder を初期化します。
    AdtsHeaderReader header(aacData, 16);
    nn::codec::AacDecoderType decoder;
    auto result = nn::codec::InitializeAacDecoder(&decoder, header.GetSampleRate(), header.GetChannelCount(), nn::codec::Mpeg4AudioObjectType_AacLc, aacWorkBuffer, aacWorkBufferSize);
    NN_ASSERT(result == nn::codec::AacDecoderResult_Success);
#else  // defined(ADTS)
    // AAC ファイルを読み込みます。
    uint8_t* aacData = nullptr;
    int64_t size = 0;
    int sampleRate = 0;
    int channelCount = 0;
    {
        InitializeFileSystem();
        const char Filename[] = "asset:/SampleBgm0-2ch.m4a";
        nn::fs::FileHandle handle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, Filename, nn::fs::OpenMode_Read));
        int64_t filesize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&filesize, handle));
        // RAW データを探します。
        int64_t offset = 0;
        while (offset < filesize)
        {
            uint8_t buffer[8];
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, offset, buffer, sizeof(buffer)));
            Mp4BoxHeaderReader header(buffer, sizeof(buffer));
//             NNS_LOG("%s: %d\n", header.GetType().c_str(), header.GetSize());
            if (header.GetType() == "mdat")
            {
                size = header.GetSize() - 8;
                aacData = static_cast<uint8_t*>(allocator.Allocate(static_cast<size_t>(size)));
                NN_ASSERT(aacData);
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, offset + 8, aacData, static_cast<std::size_t>(size)));
            }
            if (header.GetType() == "stsd")
            {
                uint8_t stsdBuffer[128];
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, offset + 8, stsdBuffer, header.GetSize() - 8));
                Mp4BoxStsdReader stsd(stsdBuffer, header.GetSize());
                sampleRate = stsd.GetSampleRate();
                channelCount = stsd.GetChannelCount();
            }
            if (header.HasChild())
            {
                offset += 8;
            }
            else
            {
                offset += header.GetSize();
            }
        }
        nn::fs::CloseFile(handle);
        FinalizeFileSystem();
    }

    nn::codec::AacDecoderType decoder;
    auto result = nn::codec::InitializeAacDecoder(&decoder, sampleRate, channelCount, nn::codec::Mpeg4AudioObjectType_AacLc, aacWorkBuffer, aacWorkBufferSize);
    NN_ASSERT(result == nn::codec::AacDecoderResult_Success);
#endif  // defined(ADTS)

    // オーディオ再生用に AudioOut を初期化、開始します。
    nn::audio::AudioOut audioOut;
    nn::os::SystemEvent systemEvent;
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    parameter.sampleRate = 48000;
    parameter.channelCount = 2;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&audioOut, &systemEvent, parameter));
    nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioOutSampleFormat(&audioOut);
    NN_ASSERT(nn::audio::GetSampleByteSize(sampleFormat) == sizeof(int16_t));
    const size_t dataSize = 1024 * nn::codec::GetAacDecoderChannelCount(&decoder) * nn::audio::GetSampleByteSize(sampleFormat);
    const size_t bufferSize = nn::util::align_up(dataSize, nn::audio::AudioOutBuffer::SizeGranularity);
    const int bufferCount = 4;
    nn::audio::AudioOutBuffer audioOutBuffer[bufferCount];
    void* outBuffer[bufferCount];
    for (int i = 0; i < bufferCount; ++i)
    {
        outBuffer[i] = allocator.Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
        NN_ASSERT(outBuffer[i]);
        memset(outBuffer[i], 0, dataSize);
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], outBuffer[i], bufferSize, dataSize);
        nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioOut(&audioOut));

    // メインループ
    int64_t dec = 0;
    while (dec < size)
    {
        systemEvent.Wait();
        nn::audio::AudioOutBuffer* pAudioOutBuffer = nullptr;
        pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
        while (pAudioOutBuffer)
        {
            void* pOutBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);

#if defined(ADTS)
            // ADTS ヘッダを読み込みます。
            uint8_t* data = aacData + dec;
            AdtsHeaderReader reader(data, 16);
            NN_ASSERT(reader.GetSampleRate() == nn::codec::GetAacDecoderSampleRate(&decoder));
            NN_ASSERT(reader.GetChannelCount() == nn::codec::GetAacDecoderChannelCount(&decoder));
            auto headerSize = reader.GetHeaderSize();
            auto frameLength = reader.GetFrameLength();
            data += headerSize;
            dec += headerSize;
#else  // defined(ADTS)
            uint32_t frameLength = 512;
            uint8_t* data = aacData + dec;
#endif  // defined(ADTS)

            // デコード処理を行います。
            size_t consumed;
            int sampleCount;
            int16_t pcm[2 * 1024 * 4];
            result = nn::codec::DecodeAacInterleaved(&decoder, &consumed, &sampleCount, pcm, sizeof(pcm), data, frameLength);
            NN_ASSERT(result == nn::codec::AacDecoderResult_Success);
            dec += consumed;

            // デコード結果を AudioOutBuffer にコピーし、再生します。
            memcpy(pOutBuffer, pcm, sizeof(int16_t) * nn::codec::GetAacDecoderChannelCount(&decoder) * sampleCount);
            nn::audio::AppendAudioOutBuffer(&audioOut, pAudioOutBuffer);

            pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
        }
    }

    nn::audio::StopAudioOut(&audioOut);
    nn::audio::CloseAudioOut(&audioOut);
    for (int i = 0; i < bufferCount; ++i)
    {
        allocator.Free(outBuffer[i]);
    }

    nn::codec::FinalizeAacDecoder(&decoder);
    allocator.Free(aacData);
    allocator.Free(aacWorkBuffer);
}  // NOLINT(readability/fn_size)
