﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{CapstoneSimple.cpp,PageSampleCapstoneSimple}
 *
 * @brief
 * capstone による逆アセンブリ
 */

/**
 * @page PageSampleCapstoneSimple capstone による逆アセンブリ
 * @tableofcontents
 *
 * @brief
 * capstone による逆アセンブリのサンプルプログラムの解説です。
 *
 * @section PageSampleCapstoneSimple_SectionBrief 概要
 * 指定したメモリに書かれている命令を逆アセンブリする capstone ライブラリのサンプルになります。
 *
 * @section PageSampleCapstoneSimple_SectionFileStoructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/CapstoneSimple
 * Samples/Sources/Applications/CapstoneSimple @endlink 以下にあります。
 *
 * @section PageSampleCapstoneSimple_SectionNecessaryEnvironment 必要な環境
 * 本機能を利用するには、libcapstone.a をリンクする必要があります。
 * libcapstone.a は Capstone パッケージで配布されていますので、別途NDIで取得する必要があります。
 *
 * @section PageSampleCapstoneSimple_SectionHowToOperate 操作方法
 * 特にありません。
 *
 * @section PageSampleCapstoneSimple_SectionPrecaution 注意事項
 * 本ライブラリを製品に含める場合、権利表記の追加が必要となります。
 * 詳細は SLIM をご参照ください。
 *
 * @section PageSampleCapstoneSimple_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleCapstoneSimple_SectionDetail 解説
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - capstone ライブラリを利用するためのメモリの設定をします。
 * - 逆アセンブリを行う命令を ARM の AArch64(NX64) であると設定します。
 * - cs_disasm を利用して、命令を一括して逆アセンブリして表示します。
 * - cs_disasm_iter を利用して、命令を1つずつ逆アセンブリして表示します。
 * - nn::diag::GetBacktrace を利用して取得したバックトレースを逆アセンブリして表示します。
 *
 * このサンプルプログラムの実行結果を以下に示します。
 *
 * @verbinclude CapstoneSimple_Output.txt
 */

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/diag/diag_Backtrace.h>

#include <capstone/capstone.h>

extern "C" void nnMain()
{
    csh handle;
    cs_insn *insn;
    size_t count;

    cs_err csError;
    cs_opt_mem csOptMem;
    csOptMem.malloc = &malloc;
    csOptMem.calloc = &calloc;
    csOptMem.realloc = &realloc;
    csOptMem.free = &free;
    csOptMem.vsnprintf = &vsnprintf;

    csError = cs_option(0, CS_OPT_MEM, reinterpret_cast<size_t>(&csOptMem));
    if (csError != CS_ERR_OK)
    {
        NN_LOG("cs_open error = %d\n", csError);
        return;
    }
    csError = cs_open(CS_ARCH_ARM64, CS_MODE_ARM, &handle);
    if (csError != CS_ERR_OK)
    {
        NN_LOG("cs_open error = %d\n", csError);
        return;
    }

    NN_LOG("=== cs_disasm ===\n");

    size_t backtraceSize;
    uintptr_t backtraceAddress[16];

    backtraceSize = nn::diag::GetBacktrace(backtraceAddress, 16);
    for(int i = 0; i < backtraceSize; i++)
    {
        NN_LOG("## trace[%d] ##\n", i);
        count = cs_disasm(handle, reinterpret_cast<uint8_t *>(backtraceAddress[i]), 64, backtraceAddress[i], 64, &insn);
        if (count > 0)
        {
            size_t j;
            for (j = 0; j < count; j++)
            {
                NN_LOG("0x%P :\t%s\t\t%s\n", insn[j].address, insn[j].mnemonic, insn[j].op_str);
            }
            cs_free(insn, count);
        }
        else
        {
            NN_LOG("ERROR: Failed to disassemble given code!\n");
        }
    }

    NN_LOG("=== cs_disasm_iter ===\n");
    cs_insn* insn_for_iter = cs_malloc(handle);
    for(int i = 0; i < backtraceSize; i++)
    {
        const unsigned char *code = reinterpret_cast<uint8_t *>(backtraceAddress[i]);
        size_t size = 64;
        uint64_t address = backtraceAddress[i];

        NN_LOG("## trace[%d] ##\n", i);

        while( cs_disasm_iter(handle, &code, &size, &address, insn_for_iter) )
        {
            NN_LOG("0x%P :\t%s\t\t%s\n", insn_for_iter->address, insn_for_iter->mnemonic, insn_for_iter->op_str);
        }
    }
    cs_free(insn_for_iter, 1);

    cs_close(&handle);
    return;
}
