﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @page BtmSimple Bluetooth 機能の制御
    @tableofcontents

    @brief
    Bluetooth 機能を制御するためのサンプルプログラムの解説です。

    @section BtmSimple_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。あらかじめペアリングしておく必要があります。
    - BT デバイス探索イベントの登録
    - 接続状態変更イベントの登録
    - NX 本体の Bluetooth デバイス情報の取得
    - BluetoothMode 変更
    - WlanMode 変更
    - 接続したデバイス情報の取得
    - 接続したデバイスの RetransmissionMode 変更
    - HID report の受信
    - A を押すと接続したデバイスの SlotMode を変更します
    - X または ^ を押すと接続を切ります
 */

#include <cstdlib>
#include <cstdio>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/bluetooth/bluetooth_Api.h>
#include <nn/btm/btm.h>
#include <nn/btm/btm_Result.h>

namespace
{
    const char ProgramName[] = "Btm Sample Application";

    const char ProfileStr[2][16] = {
        "NONE",
        "HID"
    };

    const char SniffModeStr[4][16] = {
        "5ms",
        "10ms",
        "15ms",
        "ACTIVE"
    };

    const char SlotSizeStr[6][16] = {
        "2", "4", "6", "ACTIVE"
    };

    const char WlanModeStr[3][16] = {
        "Local4", "Local8", "NONE"
    };

    const char BluetoothModeStr[2][16] = {
        "Active", "Auto"
    };

    enum Condition
    {
        Condition_Disconnected,
        Condition_Connected,
        Condition_StateChanged,

        Condition_None,
    };

    enum AsyncTask
    {
        AsyncTask_Idle,
        AsyncTask_SetSlotMode,
        AsyncTask_SetBurstMode,
        AsyncTask_SetWlanMode,
        AsyncTask_SetRetransMode,
    };

    enum State
    {
        State_Idle,
        State_Working,
    };

    State g_State;

    const size_t QueueSize = 3;
    uintptr_t    g_MessageBuffer[QueueSize];
    nn::os::MessageQueueType g_MessageQueue;

    const size_t ThreadStackSize = 8192;
    NN_ALIGNAS(4096) char g_ThreadStack[ThreadStackSize];
    nn::os::ThreadType    g_HidThread;
    bool                  g_ExitThreadFlag;

    nn::os::SystemEventType g_ConnectionEvent;
    nn::os::SystemEventType g_HidReportEvent;

    nn::btm::BdAddress          g_TargetDeviceAddress;
    nn::btm::DeviceSlotModeList g_SlotModeList;
    struct BurstModeConfig
    {
        nn::btm::BdAddress addr;
        bool mode;
    };
    BurstModeConfig g_BurstModeConfig;
    nn::btm::WlanMode      g_WlanMode;

} // anonymous-namespace

char* BdaddrToString(const nn::btm::BdAddress &bdaddr)
{
    static char strBdaddr[18];
    snprintf(strBdaddr, sizeof(strBdaddr), "%02X:%02X:%02X:%02X:%02X:%02X",
        bdaddr.address[0], bdaddr.address[1], bdaddr.address[2], bdaddr.address[3], bdaddr.address[4], bdaddr.address[5]);
    return strBdaddr;
}

void SetLed(nn::bluetooth::BluetoothAddress &bdaddr, int number)
{
    nn::bluetooth::HidData hidData;
    memset(&hidData, 0, sizeof(hidData));
    hidData.length = 12;
    hidData.data[0] = 0x01;
    hidData.data[10] = 0x30;
    hidData.data[11] = number;
    nn::bluetooth::HidSendData(&bdaddr, &hidData);
}

Condition GetCondition(uint8_t newDeviceCount)
{
    static uint8_t currentCount = 0;

    Condition condition = Condition_None;

    if (newDeviceCount > currentCount)
    {
        NN_LOG("Connected new device\n");
        condition = Condition_Connected;
    }
    else if (newDeviceCount < currentCount)
    {
        NN_LOG("Disconnected a device\n");
        condition = Condition_Disconnected;
    }
    else
    {
        NN_LOG("Connection state changed\n");
        condition = Condition_StateChanged;
    }
    currentCount = newDeviceCount;

    return condition;
}

void AddTask(AsyncTask task)
{
    static AsyncTask taskBuffer[3] = { AsyncTask_Idle };
    static int index = 0;

    taskBuffer[index] = task;
    if (nn::os::TrySendMessageQueue(&g_MessageQueue, reinterpret_cast<uintptr_t>(&taskBuffer[index])) == false)
    {
        NN_LOG("[ERR] Dropped SendMessageQueue\n");
    }
    index = (index + 1) % 3;
}

nn::Result SetRetransMode(nn::btm::BdAddress &bdaddr)
{
    // Zero Retransmission の変更。0x10, 0x11, 0x12, 0x13 だけ ZeroRetran 有効
    nn::btm::ZeroRetransmissionList zeroRetransmission;
    zeroRetransmission.enabledReportIdCount = 0;

    uint8_t reportIds[] = { 0x10, 0x11, 0x12, 0x13 };
    zeroRetransmission.enabledReportIdCount = sizeof(reportIds);
    for (int i = 0; i < zeroRetransmission.enabledReportIdCount; i++)
    {
        zeroRetransmission.enabledReportId[i] = reportIds[i];
    }

    return nn::btm::HidSetRetransmissionMode(&bdaddr, &zeroRetransmission);
}

void ChangeSlotMode(nn::btm::BdAddress &bdaddr, nn::btm::SlotMode slot)
{
    g_SlotModeList.deviceCount = 1;
    memcpy(&g_SlotModeList.device[0].bdAddress, &bdaddr, nn::btm::SIZE_OF_BDADDRESS);
    g_SlotModeList.device[0].slotMode = slot;
    AddTask(AsyncTask_SetSlotMode);
}

void ChangeWlanLocal(nn::btm::WlanMode mode)
{
    g_WlanMode = g_WlanMode == mode ? nn::btm::WlanMode_Local8 : mode;
    // g_WlanMode = g_WlanMode == mode ? nn::btm::WlanMode_None : mode;
    AddTask(AsyncTask_SetWlanMode);
}

bool Update()
{
    nn::Result result = nn::ResultSuccess();

    uintptr_t task;
    if (nn::os::TryReceiveMessageQueue(&task, &g_MessageQueue) == false || g_State == State_Working)
    {
        return true;
    }

    g_State = State_Working;
    switch (*reinterpret_cast<AsyncTask*>(task))
    {
    case AsyncTask_SetSlotMode:
        {
            NN_LOG("@@@ SetSlotMode\n");
            result = nn::btm::SetSlotMode(&g_SlotModeList);
        }
        break;

    case AsyncTask_SetBurstMode:
        {
            result = nn::btm::SetBurstMode(&g_BurstModeConfig.addr, g_BurstModeConfig.mode);
        }
        break;

    case AsyncTask_SetWlanMode:
        {
            NN_LOG("@@@ SetWlanMode (%s)\n", WlanModeStr[g_WlanMode]);
            result = nn::btm::SetWlanMode(g_WlanMode);
        }
        break;

    case AsyncTask_SetRetransMode:
        {
            NN_LOG("@@@ HidSetRetransmissionMode (%s)\n", BdaddrToString(g_TargetDeviceAddress));
            result = SetRetransMode(g_TargetDeviceAddress);
        }

    case AsyncTask_Idle:
        break;

    default:
            NN_LOG("%d\n",*reinterpret_cast<AsyncTask*>(task));
            NN_UNEXPECTED_DEFAULT;
    }

    if (!result.IsSuccess() && !nn::btm::ResultBusy::Includes(result))
    {
        NN_LOG("[ERR] result = %0X\n", result.GetInnerValueForDebug());
        g_State = State_Idle;
        // NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    return result.IsSuccess() ? true : false;
}

void UpdateHidReport()
{
    static bool button = false; //[TORIAEZU] true: pressed down, false: up

    nn::bluetooth::HidEventType eventType;
    uint8_t buffer[nn::bluetooth::BUFFER_SIZE_OF_HID_OUT];
    nn::bluetooth::HidGetReportEventInfo(&eventType, buffer, nn::bluetooth::BUFFER_SIZE_OF_HID_OUT);

    switch (eventType)
    {
    case nn::bluetooth::EventFromGetReportCallback:
        {
            nn::bluetooth::InfoFromGetReportCallback* pInfo = reinterpret_cast<nn::bluetooth::InfoFromGetReportCallback*>(buffer);
            nn::bluetooth::HidReport* pReport = &pInfo->rptData;

            if (pReport->reportData.data[0] >= 0x30 && pReport->reportData.data[0] <= 0x33 )
            {
                if ((pReport->reportData.data[3] & 0x02) > 0 || (pReport->reportData.data[5] & 0x02) > 0)
                {
                    NN_LOG("@@@ Detach a controller\n");
                    nn::btm::HidDisconnect(&pReport->bluetoothAddress);
                }
                if ((pReport->reportData.data[3] & 0x08) > 0)
                {
                    if (!button)
                    {
                        ChangeSlotMode(pReport->bluetoothAddress, nn::btm::SlotMode_4);
                    }
                    button = true;
                }
                else if ((pReport->reportData.data[3] & 0x04) > 0)
                {
                    if (!button)
                    {
                        ChangeSlotMode(pReport->bluetoothAddress, nn::btm::SlotMode_Active);
                    }
                    button = true;
                }
                else
                    button = false;
            }
        }
        break;
    case nn::bluetooth::EventFromSetReportStatusCallback:
    case nn::bluetooth::EventFromGetReportStatusCallback:
        break;
    default:
        NN_LOG("[ERR] Unhandled event: 0x%02X\n", eventType);
        NN_UNEXPECTED_DEFAULT;
    }
}

static void HidThreadFunc(void *arg)
{
    NN_UNUSED(arg);

    while (!g_ExitThreadFlag)
    {
        if (nn::os::TryWaitSystemEvent(&g_HidReportEvent))
        {
            UpdateHidReport();
        }
    }
}

extern "C" void nnMain()
{
    NN_LOG("%s Start.\n", ProgramName);

    nn::Result result;
    int32_t waitCount = 0;
    int32_t testLoop = 0;

    nn::os::InitializeMessageQueue(&g_MessageQueue, g_MessageBuffer, QueueSize);

    nn::btm::InitializeBtmInterface();

    nn::bluetooth::InitializeBluetoothDriver();
    nn::bluetooth::RegisterHidReportEvent(&g_HidReportEvent);

    // DeviceCondition 変更通知用のシステムイベントを登録
    nn::btm::RegisterSystemEventForConnectedDeviceCondition(&g_ConnectionEvent);

    // NX 本体の Bluetooth デバイス情報を取得
    nn::btm::HostDeviceProperty property;
    nn::btm::GetHostDeviceProperty(&property);
    NN_LOG("-- Host Device Property --\n");
    NN_LOG("    BD_ADDR: %s\n", BdaddrToString(property.bdAddress));
    // Major Service Classes, Major Device Classes, Minor Device Class
    NN_LOG("    COD: %02X %02X %02X\n", property.classOfDevice.cod[0], property.classOfDevice.cod[1], property.classOfDevice.cod[2]);
    NN_LOG("    Name: %s\n", property.bdName.name);
    NN_LOG("    FeatureSet: %02X\n", property.featureSet);
    NN_LOG("-- -------------------- --\n");

    g_ExitThreadFlag = false;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_HidThread, HidThreadFunc, NULL, g_ThreadStack,
                                                        ThreadStackSize, nn::os::HighestThreadPriority));
    nn::os::StartThread(&g_HidThread);

    g_State = State_Idle;

    while (testLoop < 100)
    {
        // BT デバイスの接続、切断、DeviceCondition 変更に関する操作
        if (nn::os::TryWaitSystemEvent(&g_ConnectionEvent))
        {
            NN_LOG("! Connection state changed\n");
            nn::btm::DeviceConditionList list;
            // 現在の DeviceCondition 情報を取得
            nn::btm::GetConnectedDeviceCondition(&list);

            NN_LOG("-- Bluetooth Device Condition --\n");
            NN_LOG("    WlanMode:%s\n", WlanModeStr[list.wlanMode]);
            NN_LOG("    BluetoothMode:%s\n", BluetoothModeStr[list.bluetoothMode]);
            NN_LOG("-- -------------------------- --\n");
            for (int i = 0; i < list.deviceCount; ++i)
            {
                NN_LOG("Dev:%d\n", i);
                NN_LOG("    BD_ADDR: %s\n", BdaddrToString(list.device[i].bdAddress));
                NN_LOG("    Profile:%s\n", ProfileStr[list.device[i].profile]);
                NN_LOG("    Name:%s\n", list.device[i].bdName.name);

                if (list.device[i].profile == nn::btm::Profile_Hid)
                {
                    NN_LOG("    SlotMode:%s\n", SlotSizeStr[list.device[i].hidDeviceCondition.slotMode]);
                    NN_LOG("    SniffMode:%s, BurstMode:%s, Zero retx:%s\n",
                            SniffModeStr[list.device[i].hidDeviceCondition.sniffMode],
                            list.device[i].hidDeviceCondition.isBurstMode ? "ON" : "OFF",
                            list.device[i].hidDeviceCondition.zeroRetransmissionList.enabledReportIdCount > 0 ? "ON" : "OFF");
                }
            }
            NN_LOG("-- -------------------------- --\n");

            Condition condition = GetCondition(list.deviceCount);
            if (condition == Condition_StateChanged)
            {
                g_State = State_Idle;
            }
            else if (condition == Condition_Connected)
            {
                SetLed(list.device[list.deviceCount - 1].bdAddress, list.deviceCount);
                memcpy(&g_TargetDeviceAddress, &list.device[list.deviceCount - 1].bdAddress, nn::btm::SIZE_OF_BDADDRESS);
                AddTask(AsyncTask_SetRetransMode);
            }
        }

        Update();

        if (waitCount <= 0)
        {
            waitCount = 4000;
            testLoop++;
            NN_LOG("##### itr = %d\n", testLoop);
            // WlanMode, BluetoothMode を切り替えます
            ChangeWlanLocal(nn::btm::WlanMode_Local4);
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        --waitCount;
    }

    g_ExitThreadFlag = true;
    nn::os::DestroyThread(&g_HidThread);

    nn::bluetooth::FinalizeBluetoothDriver();

    nn::os::DestroySystemEvent(&g_ConnectionEvent);
    nn::os::DestroySystemEvent(&g_HidReportEvent);

    nn::btm::FinalizeBtmInterface();

    NN_LOG("%s Done\n", ProgramName);
}

