﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include "BleSimple_Color.h"
#include "BleSimple_ScanScene.h"
#include "BleSimple_Uuid.h"

ScanScene::ScanScene(std::string title, uint16_t scanParameterId)
    : SceneBase(title)
{
    m_IsScanningGeneral = false;
    m_IsScanningSmartDevice = false;
    m_ScanParameterId = scanParameterId;
}

void ScanScene::Draw(nn::gfx::util::DebugFontTextWriter* pTextWriter)
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    pTextWriter->SetCursor(20, 20);
    pTextWriter->SetTextColor(Color::White);
    pTextWriter->Print("%s", m_Title.c_str());

    pTextWriter->SetCursor(20, 40);
    pTextWriter->Print("SCAN STATE: %s", (m_IsScanningGeneral | m_IsScanningSmartDevice) ? "SCANNING" : "NOT SCANNING");

    pTextWriter->SetCursor(20, 65);
    pTextWriter->Print("General Scan Results:");

    float offsetY = 65;
    for (int i = 0; i < NN_ARRAY_SIZE(m_ScanResults); ++i)
    {
        if (m_Cursor.posY == i)
        {
            pTextWriter->SetTextColor(Color::Orange);
        }

        offsetY += 25;

        pTextWriter->SetCursor(40, offsetY);
        pTextWriter->Print("#%d    Address: %02X:%02X:%02X:%02X:%02X:%02X    RSSI: %d",
            i,
            m_ScanResults[i].address.address[0], m_ScanResults[i].address.address[1], m_ScanResults[i].address.address[2],
            m_ScanResults[i].address.address[3], m_ScanResults[i].address.address[4], m_ScanResults[i].address.address[5],
            m_ScanResults[i].rssi);

        pTextWriter->SetTextColor(Color::White);
    }

    pTextWriter->SetCursor(650, 65);
    pTextWriter->Print("Advertise Packet Content:");

    offsetY = 65;

    nn::bluetooth::BleScanResult selectedResult = m_ScanResults[m_Cursor.posY];

    for (int i = 0; i < selectedResult.adStructureNum; ++i)
    {
        offsetY += 30;
        pTextWriter->SetCursor(670, offsetY);
        pTextWriter->Print("AD Type: %d    Length: %d", selectedResult.adStructures[i].adType, selectedResult.adStructures[i].length);

        offsetY += 25;
        pTextWriter->SetCursor(670, offsetY);

        int offsetX = 0;
        for (int j = 0; j < selectedResult.adStructures[i].length - 1; ++j)
        {
            pTextWriter->SetCursorX(670 + offsetX * pTextWriter->GetFontWidth() * 2);
            pTextWriter->Print("%02X ", selectedResult.adStructures[i].data[j]);

            offsetX++;

            if (j % 16 == 15)
            {
                offsetX = 0;
                offsetY += 20;
                pTextWriter->SetCursorY(offsetY);
            }
        }
    }

    pTextWriter->SetCursor(20, 690);
    pTextWriter->Print("X: Start Scan    Y: Stop Scan    Up/Down: Select Scanned Device    B: Refresh Scan Result   A: Connect    +: Exit");
}

void ScanScene::ProcessControllerInput(nns::hid::ButtonSet buttons)
{
    if ((buttons & nns::hid::Button::X::Mask).IsAnyOn())
    {
        NN_LOG("Start BLE Scan\n");

        StartBleScanGeneral();
        StartBleScanSmartDevice();
    }
    else if ((buttons & nns::hid::Button::Y::Mask).IsAnyOn())
    {
        NN_LOG("Stop BLE Scan\n");

        StopBleScanGeneral();
        StopBleScanSmartDevice();
    }
    else if ((buttons & nns::hid::Button::B::Mask).IsAnyOn())
    {
        NN_LOG("Refresh BLE Scan\n");

        UpdateScanResult();
    }
    else if ((buttons & nns::hid::Button::Up::Mask).IsAnyOn())
    {
        if (m_Cursor.posY > 0)
        {
            m_Cursor.posY--;
        }
    }
    else if ((buttons & nns::hid::Button::Down::Mask).IsAnyOn())
    {
        if (m_Cursor.posY < NN_ARRAY_SIZE(m_ScanResults) - 1)
        {
            m_Cursor.posY++;
        }
    }
    else if ((buttons & nns::hid::Button::A::Mask).IsAnyOn())
    {
        NN_LOG("Connect to GATT Server\n");

        if (!StopBleScanGeneral() || !StopBleScanSmartDevice())
        {
            return;
        }

        nn::bluetooth::BleScanResult selectedResult = m_ScanResults[m_Cursor.posY];

        auto result = nn::bluetooth::ConnectToGattServer(selectedResult.address);
        //!< 既に接続が存在します。
        //!< 本サンプルでは、GATT Client Scene で接続を確認できます。
        if (nn::bluetooth::ResultBleAlreadyConnected().Includes(result))
        {
            NN_LOG("Connection already exist.\n");
        }
        //!< 接続数が最大に達しています。
        //!< 本サンプルでは、GATT Client Scene でX ボタンを押して切断できます。
        else if (nn::bluetooth::ResultBleConnectionFull().Includes(result))
        {
            NN_LOG("Connection full. Disconnect existing connection and try again.\n");
        }
        //!< Bluetooth がOFF になっています。
        else if (nn::bluetooth::ResultBluetoothOff().Includes(result))
        {
            NN_LOG("Bluetooth is OFF.\n");
        }
        //!< BLE 機能を使える状態にありません。必要なコントローラーのみを接続してください。
        else if (nn::bluetooth::ResultInvalidWirelessState().Includes(result))
        {
            NN_LOG("You must disconnect unnecessary controllers.\n");
        }
    }
}

bool ScanScene::StartBleScanGeneral()
{
    nn::bluetooth::BleAdvertisePacketParameter param;
    nn::bluetooth::GetBleScanParameter(&param, m_ScanParameterId);

    auto result = nn::bluetooth::StartBleScanGeneral(param);
    if (result.IsFailure())
    {
        NN_LOG("Failed to start BLE Scan General.\n");

        //!< システムが他のタスクを実行中なので、少し待ってから再試行してください。
        //!< 本サンプルでは、X ボタンを押すことで再試行します。
        if (nn::bluetooth::ResultSystemBusy().Includes(result))
        {
            NN_LOG("Wait a little and try again .\n");
        }
        //!< システムで何らかのエラーが発生しました。再試行することで成功することがあります。
        //!< 本サンプルでは、X ボタンを押すことで再試行します。
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            NN_LOG("Wait a little and trye again.\n");
        }
        //!< BLE スキャンの対象が上限に達しています。不要な既存のBLE スキャンを停止してから再試行してください。
        //!< 本サンプルでは、Y ボタンを押すことでBLE スキャンを停止します。
        else if (nn::bluetooth::ResultBleScanFull().Includes(result))
        {
            NN_LOG("Stop on going BLE scan and try again.\n");
        }
        //!< Bluetooth がOFF になっています。
        else if (nn::bluetooth::ResultBluetoothOff().Includes(result))
        {
            NN_LOG("Bluetooth is OFF.\n");
        }
        return false;
    }
    else
    {
        ClearScanResult();
        m_IsScanningGeneral = true;
    }

    return true;
}

bool ScanScene::StopBleScanGeneral()
{
    auto result = nn::bluetooth::StopBleScanGeneral();
    if (result.IsFailure())
    {
        NN_LOG("Failed to stop BLE Scan General.\n");

        //!< システムが他のタスクを実行中なので、少し待ってからリトライしてください
        if (nn::bluetooth::ResultSystemBusy().Includes(result))
        {
            NN_LOG("Wait a little and try again .\n");
        }
        //!< システムで何らかのエラーが発生しました。再試行することで成功することがあります。
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            NN_LOG("Wait a little and trye again.\n");
        }
        //!< Bluetooth がOFF になっています。
        else if (nn::bluetooth::ResultBluetoothOff().Includes(result))
        {
            NN_LOG("Bluetooth is OFF.\n");
        }

        return false;
    }

    m_IsScanningGeneral = false;

    return true;
}

bool ScanScene::StartBleScanSmartDevice()
{
    auto result = nn::bluetooth::StartBleScanSmartDevice(ADV_SERVICE_UUID);
    if (result.IsFailure())
    {
        NN_LOG("Failed to start BLE Scan Smart Device.\n");

        //!< システムが他のタスクを実行中なので、少し待ってから再試行してください。
        //!< 本サンプルでは、X ボタンを押すことで再試行します。
        if (nn::bluetooth::ResultSystemBusy().Includes(result))
        {
            NN_LOG("Wait a little and try again .\n");
        }
        //!< システムで何らかのエラーが発生しました。再試行することで成功することがあります。
        //!< 本サンプルでは、X ボタンを押すことで再試行します。
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            NN_LOG("Wait a little and trye again.\n");
        }
        //!< BLE スキャンの対象が上限に達しています。不要な既存のBLE スキャンを停止してから再試行してください。
        //!< 本サンプルでは、Y ボタンを押すことでBLE スキャンを停止します。
        else if (nn::bluetooth::ResultBleScanFull().Includes(result))
        {
            NN_LOG("Stop on going BLE scan and try again.\n");
        }
        //!< Bluetooth がOFF になっています。
        else if (nn::bluetooth::ResultBluetoothOff().Includes(result))
        {
            NN_LOG("Bluetooth is OFF.\n");
        }
        return false;
    }
    else
    {
        ClearScanResult();
        m_IsScanningSmartDevice = true;
    }

    return true;
}

bool ScanScene::StopBleScanSmartDevice()
{
    auto result = nn::bluetooth::StopBleScanSmartDevice();
    if (result.IsFailure())
    {
        NN_LOG("Failed to stop BLE Scan Smart Device.\n");

        //!< システムが他のタスクを実行中なので、少し待ってからリトライしてください
        if (nn::bluetooth::ResultSystemBusy().Includes(result))
        {
            NN_LOG("Wait a little and try again .\n");
        }
        //!< システムで何らかのエラーが発生しました。再試行することで成功することがあります。
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            NN_LOG("Wait a little and trye again.\n");
        }
        //!< Bluetooth がOFF になっています。
        else if (nn::bluetooth::ResultBluetoothOff().Includes(result))
        {
            NN_LOG("Bluetooth is OFF.\n");
        }
        return false;
    }

    m_IsScanningSmartDevice = false;

    return true;
}

void ScanScene::UpdateScanResult()
{
    //!< スキャン結果を取得
    nn::bluetooth::GetBleScanResult(m_ScanResults, NN_ARRAY_SIZE(m_ScanResults));
}

void ScanScene::ClearScanResult()
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_ScanResults); ++i)
    {
        memset(&m_ScanResults[i], 0x00, sizeof(nn::bluetooth::BleScanResult));
    }
}
