﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{BcatSimple.cpp,PageSampleBcatSimple}
 *
 * @brief
 *  データ配信のサンプルプログラム
 */

/**
 * @page PageSampleBcatSimple データ配信機能の紹介
 * @tableofcontents
 *
 * @brief
 *  データ配信機能を紹介するサンプルプログラムです。
 *
 * @section PageSampleBcatSimple_SectionBrief 概要
 *  ここでは、データ配信ライブラリの API を一通り紹介します。
 *
 * @section PageSampleBcatSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/BcatSimple Samples/Sources/Applications/BcatSimple @endlink 以下にあります。
 *
 * @section PageSampleBcatSimple_SectionNecessaryEnvironment 必要な環境
 *  ネットワークサービスアカウントが紐付いたユーザーアカウントを作成する必要があります。@n
 *  また、ネットワーク接続が可能な環境が必要です。
 *
 * @section PageSampleBcatSimple_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleBcatSimple_SectionPrecaution 注意事項
 *  開発中の機能であるため、データ配信ライブラリの利用に関して制限が掛かっている場合があります。@n
 *  利用に関する制限は、 NintendoSDK ドキュメントを参照してください。
 *
 * @section PageSampleBcatSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleBcatSimple_SectionDetail 解説
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - 即時同期を行う。
 *  - データ配信キャッシュストレージをマウントする。
 *  - 受信済みのファイルを列挙する。
 */

#include <nn/bcat.h>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/nifm.h>

// ダウンロードが完了するまで待機します。
void WaitForDone(nn::bcat::DeliveryCacheProgress& progress) NN_NOEXCEPT
{
    NN_LOG("Start!\n");

    // 完了するまで待機します。
    while (progress.GetStatus() != nn::bcat::DeliveryCacheProgressStatus_Done)
    {
        progress.Update();

        switch (progress.GetStatus())
        {
        case nn::bcat::DeliveryCacheProgressStatus_Queued:
            NN_LOG("Progress: status = queued\n");
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Connect:
            NN_LOG("Progress: status = connect\n");
            break;
        case nn::bcat::DeliveryCacheProgressStatus_ProcessList:
            NN_LOG("Progress: status = process list\n");
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Download:
            NN_LOG("Progress: status = download, current = (%s/%s, %lld/%lld), whole = (%lld/%lld)\n",
                progress.GetCurrentDirectoryName().value, progress.GetCurrentFileName().value,
                progress.GetCurrentDownloaded(), progress.GetCurrentTotal(),
                progress.GetWholeDownloaded(), progress.GetWholeTotal());
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Commit:
            NN_LOG("Progress: status = commit, current = (%s), whole = (%lld/%lld)\n",
                progress.GetCurrentDirectoryName().value,
                progress.GetWholeDownloaded(), progress.GetWholeTotal());
            break;
        case nn::bcat::DeliveryCacheProgressStatus_Done:
            NN_LOG("Progress: status = done, whole = (%lld/%lld), e = %08x\n",
                progress.GetWholeDownloaded(), progress.GetWholeTotal(),
                progress.GetResult().GetInnerValueForDebug());
            break;
        default:
            break;
        }

        // 過剰なログ出力を抑えるため、 500ms 待機します。
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
    }

    NN_LOG("Done!\n");
}

// データ配信キャッシュストレージをマウントします。
bool MountDeliveryCacheStorage() NN_NOEXCEPT
{
    nn::Result result = nn::bcat::MountDeliveryCacheStorage();

    // バックグラウンド同期中にアプリを起動した場合、バックグラウンド同期が完了するまでアプリがマウントすることはできません。
    while (nn::bcat::ResultLocked::Includes(result))
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        // 一定期間待機した後、再度マウントを試みます。
        result = nn::bcat::MountDeliveryCacheStorage();
    }

    if (nn::bcat::ResultServiceUnavailable::Includes(result))
    {
        NN_LOG("[Warning] Please link Nintendo account or register Network service account.\n");
    }

    return result.IsSuccess();
}

// 受信したファイルをすべて列挙します。
void EnumerateAll() NN_NOEXCEPT
{
    static nn::bcat::DirectoryName s_DirNames[nn::bcat::DeliveryCacheDirectoryCountMax] = {};
    int dirCount = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bcat::EnumerateDeliveryCacheDirectory(&dirCount, s_DirNames, NN_ARRAY_SIZE(s_DirNames)));

    NN_LOG("----------------------------------------------------------------------------------------------------\n");
    NN_LOG("DirectoryCount = %d\n", dirCount);

    for (int d = 0; d < dirCount; d++)
    {
        NN_LOG("----------------------------------------------------------------------------------------------------\n");
        NN_LOG("Directory[%3d] = %s\n", d, s_DirNames[d].value);

        nn::bcat::DeliveryCacheDirectory directory;

        NN_ABORT_UNLESS_RESULT_SUCCESS(directory.Open(s_DirNames[d]));

        static nn::bcat::DeliveryCacheDirectoryEntry s_Entries[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};
        int entryCount = 0;

        NN_ABORT_UNLESS_RESULT_SUCCESS(directory.Read(&entryCount, s_Entries, NN_ARRAY_SIZE(s_Entries)));

        NN_LOG("    EntryCount = %d\n", entryCount);

        for (int e = 0; e < entryCount; e++)
        {
            NN_LOG("    Entry[%3d] = %32s, %016llx%016llx, %lld\n", e,
                s_Entries[e].name.value, s_Entries[e].digest.value[0], s_Entries[e].digest.value[1], s_Entries[e].size);

            nn::bcat::DeliveryCacheFile file;

            NN_ABORT_UNLESS_RESULT_SUCCESS(file.Open(s_DirNames[d], s_Entries[e].name));

            nn::Bit8 data[8] = {};
            size_t read;

            NN_ABORT_UNLESS_RESULT_SUCCESS(file.Read(&read, 0, data, NN_ARRAY_SIZE(data)));

            NN_LOG("               = ");

            for (int r = 0; r < static_cast<int>(read); r++)
            {
                char c = static_cast<char>(data[r]);

                NN_LOG("%c(%02X) ", (c >= 0x20 && c <= 0x7E) ? c : '?', data[r]);
            }

            NN_LOG("\n");
        }
    }

    NN_LOG("----------------------------------------------------------------------------------------------------\n");
}

extern "C" void nnMain()
{
    nn::bcat::Initialize();
    nn::nifm::Initialize();

    // 本サンプルプログラムは、起動直後にデータ配信キャッシュの即時同期を行います。

    NN_LOG("SubmitNetworkRequestAndWait ...\n");

    nn::nifm::SubmitNetworkRequestAndWait();

    NN_LOG("SubmitNetworkRequestAndWait done!\n");

    if (nn::nifm::IsNetworkAvailable())
    {
        nn::bcat::DeliveryCacheProgress progress;

        // インターネットに接続できた場合のみ、即時同期を行います。
        nn::Result result = nn::bcat::RequestSyncDeliveryCache(&progress);

        if (result.IsSuccess())
        {
            WaitForDone(progress);
        }
        else
        {
            NN_LOG("RequestSyncDeliveryCache failed. e = %08x\n", result.GetInnerValueForDebug());
        }
    }
    else
    {
        NN_LOG("The network is not available.\n");
    }

    if (MountDeliveryCacheStorage())
    {
        EnumerateAll();

        nn::bcat::UnmountDeliveryCacheStorage();
    }

    NN_LOG("End.\n");
}
