﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{BcatPractical.cpp,PageSampleBcatPractical}
 *
 * @brief
 *  データ配信のサンプルプログラム
 */

/**
 * @page PageSampleBcatPractical データ配信機能の実践的な利用
 * @tableofcontents
 *
 * @brief
 *  データ配信機能の実践的な利用について解説するサンプルプログラムです。
 *
 * @section PageSampleBcatPractical_SectionBrief 概要
 *  ここでは、データ配信機能の実践的な利用について解説します。
 *
 * @section PageSampleBcatPractical_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/BcatPractical Samples/Sources/Applications/BcatPractical @endlink 以下にあります。
 *
 * @section PageSampleBcatPractical_SectionNecessaryEnvironment 必要な環境
 *  ネットワークサービスアカウントが紐付いたユーザーアカウントを作成する必要があります。@n
 *  また、ネットワーク接続が可能な環境が必要です。
 *
 * @section PageSampleBcatPractical_SectionHowToOperate 操作方法
 *  Npad または DebugPad を使用します。
 *
 * @section PageSampleBcatPractical_SectionPrecaution 注意事項
 *  開発中の機能であるため、データ配信ライブラリの利用に関して制限が掛かっている場合があります。@n
 *  利用に関する制限は、 NintendoSDK ドキュメントを参照してください。
 *
 * @section PageSampleBcatPractical_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。@n
 *  繰り返し即時同期の処理を確認したい場合、サンプルプログラムを終了した後、データ配信キャッシュストレージを削除してください。
 *
 * @section PageSampleBcatPractical_SectionDetail 解説
 *  即時同期や配信データへのアクセスを行うサンプルプログラムです。
 *
 *  エラーハンドリングや即時同期を行う際の注意点が記載されています。@n
 *  詳しくは、各シーン関数内のコメントを参照してください。
 */

#include <nn/bcat.h>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/nifm.h>
#include <nn/err.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nns/console/console_ConsoleManager.h>

#include "BcatPractical.h"
#include "Input.h"

#define CHANGE_SCENE(nextSceneId) \
    do                                  \
    {                                   \
        SetNextSceneId(nextSceneId);    \
        return;                         \
    }                                   \
    while (NN_STATIC_CONDITION(false))

namespace
{
    nns::console::DefaultConsoleManagerHolder g_ConsoleManagerHolder = NNS_CONSOLE_CONSOLE_MANAGER_HOLDER_INITIALIZER;

    NN_ALIGNAS(4096) nn::Bit8 g_ThreadStack[64 * 1024];
}

SampleProgram::SampleProgram() NN_NOEXCEPT :
    m_SceneId(SceneId_Top),
    m_IsRunning(false)
{
}

void SampleProgram::Start(void* stack, size_t stackSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, Main, this, stack, stackSize, nn::os::DefaultThreadPriority));
    nn::os::StartThread(&m_Thread);
}

void SampleProgram::SetNextSceneId(SceneId sceneId) NN_NOEXCEPT
{
    m_SceneId = sceneId;
}

void SampleProgram::SceneTop() NN_NOEXCEPT
{
    nns::console::SimpleConsole::Clear();

    Log("--------------------------------------------------------------------------------\n");
    Log("SceneTop\n");
    Log("--------------------------------------------------------------------------------\n");

    Log("[A]: Access to delivery cache storage\n");
    Log("[X]: Sync delivery cache\n");

    while (NN_STATIC_CONDITION(true))
    {
        nn::hid::NpadButtonSet buttons = input::GetTriggerButtons();

        if (buttons.Test<nn::hid::NpadButton::A>())
        {
            CHANGE_SCENE(SceneId_Access);
        }
        else if (buttons.Test<nn::hid::NpadButton::X>())
        {
            CHANGE_SCENE(SceneId_Sync);
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }
}

void SampleProgram::SceneAccess() NN_NOEXCEPT
{
    nns::console::SimpleConsole::Clear();

    Log("--------------------------------------------------------------------------------\n");
    Log("SceneAccess\n");
    Log("--------------------------------------------------------------------------------\n");

    nn::Result result = nn::bcat::MountDeliveryCacheStorage();

    // データ配信キャッシュストレージのマウントに失敗した場合、エラービューアを表示します。
    if (result.IsFailure())
    {
        nn::err::ShowError(result);
        CHANGE_SCENE(SceneId_Top);
    }

    NN_UTIL_SCOPE_EXIT
    {
        nn::bcat::UnmountDeliveryCacheStorage();
    };

    // サンプルではディレクトリ・ファイルを列挙します。

    nn::bcat::DirectoryName dirNames[nn::bcat::DeliveryCacheDirectoryCountMax] = {};
    int dirCount = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bcat::EnumerateDeliveryCacheDirectory(&dirCount, dirNames, NN_ARRAY_SIZE(dirNames)));

    for (int d = 0; d < dirCount; d++)
    {
        nn::bcat::DeliveryCacheDirectory directory;

        NN_ABORT_UNLESS_RESULT_SUCCESS(directory.Open(dirNames[d]));

        Log("- %s\n", dirNames[d].value);

        nn::bcat::DeliveryCacheDirectoryEntry entries[nn::bcat::DeliveryCacheFileCountMaxPerDirectory] = {};
        int entryCount = 0;

        NN_ABORT_UNLESS_RESULT_SUCCESS(directory.Read(&entryCount, entries, NN_ARRAY_SIZE(entries)));

        for (int e = 0; e < entryCount; e++)
        {
            Log("  - %32s, %016llx%016llx, %lld\n",
                entries[e].name.value, entries[e].digest.value[0], entries[e].digest.value[1], entries[e].size);
        }
    }

    Log("[B]: Back\n");

    while (NN_STATIC_CONDITION(true))
    {
        nn::hid::NpadButtonSet buttons = input::GetTriggerButtons();

        if (buttons.Test<nn::hid::NpadButton::B>())
        {
            CHANGE_SCENE(SceneId_Top);
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }
}

void SampleProgram::SceneSync() NN_NOEXCEPT
{
    nns::console::SimpleConsole::Clear();

    Log("--------------------------------------------------------------------------------\n");
    Log("SceneSync\n");
    Log("--------------------------------------------------------------------------------\n");

    /*
        即時同期を複数回呼び出した場合、前回の即時同期が完了するまで実行待ち状態（status = queued）になります。
        実行待ち状態では、パーセンテージを計算して適切な進捗を表示することができなくなります。
        また、実行待ち状態の即時同期要求が溜まり続けると、即時同期要求時に nn::bcat::ResultTooManySyncRequests が返るようになります。

        そのため、進捗オブジェクトはグローバルに 1 つだけ保持するようにし、処理が完了するまで新規に即時同期要求を行わないようにすることを推奨します。
    */
    if (m_IsRunning)
    {
        m_Progress.Update();

        if (m_Progress.GetStatus() == nn::bcat::DeliveryCacheProgressStatus_Done)
        {
            m_IsRunning = false;
        }
    }

    Log("[B]: Cancel\n");

    if (!m_IsRunning)
    {
        if (!Connect())
        {
            CHANGE_SCENE(SceneId_Top);
        }

        Log("Sync...\n");

        nn::Result result = nn::bcat::RequestSyncDeliveryCache(&m_Progress);

        if (result.IsFailure())
        {
            nn::err::ShowError(result);
            CHANGE_SCENE(SceneId_Top);
        }

        m_IsRunning = true;
    }

    if (!WaitForDone())
    {
        CHANGE_SCENE(SceneId_Top);
    }

    Log("[A]: OK\n");

    while (NN_STATIC_CONDITION(true))
    {
        nn::hid::NpadButtonSet buttons = input::GetTriggerButtons();

        if (buttons.Test<nn::hid::NpadButton::A>())
        {
            CHANGE_SCENE(SceneId_Top);
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }
}

bool SampleProgram::Connect() NN_NOEXCEPT
{
    Log("Connecting...\n");

    while (NN_STATIC_CONDITION(true))
    {
        nn::nifm::SubmitNetworkRequest();

        while (nn::nifm::IsNetworkRequestOnHold())
        {
            nn::hid::NpadButtonSet buttons = input::GetTriggerButtons();

            if (buttons.Test<nn::hid::NpadButton::B>())
            {
                return false;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        }
        if (nn::nifm::IsNetworkAvailable())
        {
            break;
        }

        if (nn::nifm::ResultErrorHandlingCompleted::Includes(nn::nifm::HandleNetworkRequestResult()))
        {
            // 利用要求を再提出することで受理される可能性があります。
        }
        else
        {
            return false;
        }
    }

    Log("Connected!\n");

    return true;
}

bool SampleProgram::WaitForDone() NN_NOEXCEPT
{
    nn::bcat::DeliveryCacheProgressStatus status = nn::bcat::DeliveryCacheProgressStatus_None;
    int prevPercent = 0;

    int count = 0;

    while (status != nn::bcat::DeliveryCacheProgressStatus_Done)
    {
        nn::hid::NpadButtonSet buttons = input::GetTriggerButtons();

        if (buttons.Test<nn::hid::NpadButton::B>())
        {
            // 同期要求をキャンセルします。
            // 本サンプルは、キャンセル後にデータ配信キャッシュストレージにアクセスできることを保証するため、完了するまで待機します。
            nn::bcat::CancelSyncDeliveryCacheRequest();
        }

        m_Progress.Update();

        status = m_Progress.GetStatus();

        if (status == nn::bcat::DeliveryCacheProgressStatus_Download ||
            status == nn::bcat::DeliveryCacheProgressStatus_Commit ||
            status == nn::bcat::DeliveryCacheProgressStatus_Done)
        {
            int64_t downloaded = m_Progress.GetWholeDownloaded();
            int64_t total = m_Progress.GetWholeTotal();

            if (total > 0)
            {
                int percent = static_cast<int>(downloaded * 100 / total);

                if (percent != prevPercent)
                {
                    Log("%d %% (%lld / %lld)\n", percent, downloaded, total);
                    prevPercent = percent;
                }
            }
        }
        if ((++count) % 50 == 0)
        {
            // 進捗の状態をデバッグ出力にのみダンプしておきます。
            switch (status)
            {
            case nn::bcat::DeliveryCacheProgressStatus_Queued:
                NN_LOG("Progress: status = queued\n");
                break;
            case nn::bcat::DeliveryCacheProgressStatus_Connect:
                NN_LOG("Progress: status = connect\n");
                break;
            case nn::bcat::DeliveryCacheProgressStatus_ProcessList:
                NN_LOG("Progress: status = process list\n");
                break;
            case nn::bcat::DeliveryCacheProgressStatus_Download:
                NN_LOG("Progress: status = download\n");
                break;
            case nn::bcat::DeliveryCacheProgressStatus_Commit:
                NN_LOG("Progress: status = commit\n");
                break;
            case nn::bcat::DeliveryCacheProgressStatus_Done:
                NN_LOG("Progress: status = done\n");
                break;
            default:
                break;
            }
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

    m_IsRunning = false;

    Log("Done!\n");

    nn::Result result = m_Progress.GetResult();

    // 即時同期に失敗した場合、ユーザー操作によるキャンセルでなければエラービューアを表示します。
    if (result.IsFailure() && !nn::bcat::ResultCanceledByUser::Includes(result))
    {
        nn::err::ShowError(result);
        return false;
    }

    return true;
}

void SampleProgram::Log(const char* format, ...) NN_NOEXCEPT
{
    char buffer[256];

    va_list args;

    va_start(args, format);
    {
        nn::util::VSNPrintf(buffer, sizeof (buffer), format, args);

        nns::console::SimpleConsole::Puts(buffer);
        NN_VLOG(format, args);
    }
    va_end(args);
}

void SampleProgram::Main(void* param) NN_NOEXCEPT
{
    SampleProgram* pThis = reinterpret_cast<SampleProgram*>(param);

    while (NN_STATIC_CONDITION(true))
    {
        switch (pThis->m_SceneId)
        {
        case SceneId_Top:
            pThis->SceneTop();
            break;
        case SceneId_Access:
            pThis->SceneAccess();
            break;
        case SceneId_Sync:
            pThis->SceneSync();
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        // シーンが切り替わる前に、Trigger/Release 等の入力差分をクリアしておきます。
        input::Update();
    }
}

extern "C" void nnMain()
{
    nn::bcat::Initialize();
    nn::nifm::Initialize();

    input::Initialize();

    SampleProgram program;

    auto& manager = g_ConsoleManagerHolder.GetManager();
    manager.Register(&program, 0, 0, manager.Settings.ScreenWidth, manager.Settings.ScreenHeight);

    program.Start(g_ThreadStack, sizeof (g_ThreadStack));

    while (NN_STATIC_CONDITION(true))
    {
        manager.Update();
        manager.Draw();

        input::Update();
    }
}
