﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.AudioTool;
using System;
using System.Diagnostics;
using System.Linq;

namespace SystemAudioCaptureSample
{
    /// <summary>
    ///  開発機のシステムミキサーの出力相当のオーディオデータをキャプチャし、 host PC 上で処理を行うサンプルプログラムです。
    /// </summary>
    /// <remarks>
    ///   <para>
    ///     ファイル構成:
    ///     本サンプルプログラムは Samples/Sources/Applications/AudioToolUtilities/SystemAudioCapture/ 以下にあります。
    ///   </para>
    ///   <para>
    ///     必要な環境:
    ///     プログラムの起動前に、デフォルトに指定されているターゲットとの接続を行う必要があります。
    ///   </para>
    ///   <para>
    ///     操作方法:
    ///     サンプルプログラムを実行すると、開発機からキャプチャした波形の MeanSquare 値を表示します。
    ///     キーボードによる入力を行うことで、キャプチャを終了します。
    ///   </para>
    ///   <para>
    ///     注意事項:
    ///     特にありません。
    ///   </para>
    ///   <para>
    ///     実行手順:
    ///     サンプルプログラムをビルドし、実行してください。
    ///   </para>
    ///   <para>
    ///     解説:
    ///     このサンプルプログラムは、開発機からシステムミキサーの出力データをキャプチャし、 host PC 上で処理を行うものです。
    ///     サンプルプログラムの流れは、以下のようになります。
    ///     <list type="bullet">
    ///       <item>
    ///         <description>SystemAudioCapture クラスの生成</description>
    ///         <description>SystemAudioCapture に、波形キャプチャ時、停止時のイベントハンドラを登録する</description>
    ///         <description>キャプチャの開始</description>
    ///         <description>停止の要求があるまで、キャプチャを行い続ける</description>
    ///         <description>キャプチャの停止</description>
    ///       </item>
    ///     </list>
    ///
    ///     はじめに SystemAudioCapture クラスを生成します。
    ///     この際にターゲット名を指定することで、波形をキャプチャするターゲットを指定することができます。
    ///     ターゲット名を指定しない場合、デフォルトのターゲットの波形をキャプチャします。
    ///
    ///     次に、波形キャプチャ時、停止時のイベントハンドラを登録したうえで、キャプチャを開始します。
    ///     このサンプルでは、波形のキャプチャ時に MeanSquare 値の計算を、
    ///     停止時に例外が起きていればログの出力を行うイベントハンドラを登録しています。
    ///     キャプチャ処理は、 SystemAudioCapture クラスで波形のキャプチャの停止を行うか、
    ///     キャプチャ時に例外が発生するまで続きます。
    ///
    ///     その後、キーボードの入力が行われるか、ターゲットとの接続が切れるなどの例外が発生すると、
    ///     キャプチャの停止が行われ、プログラムが終了します。
    ///   </para>
    /// </remarks>
    class Program
    {
        class ConsoleApplication
        {
            private int _channelCount;
            private int _bitDepth;
            private bool _isCaptureTerminated = false;

            private int ByteSizePerSample
            {
                get
                {
                    return _bitDepth / 8;
                }
            }

            private void OnDataAvailable(object sender, SystemAudioCaptureEventArgs e)
            {
                if (e.Buffer != null)
                {
                    var frontLeftData = GetChannelData(e.Buffer, e.BytesCaptured, 0);
                    var frontRightData = GetChannelData(e.Buffer, e.BytesCaptured, 1);
                    Console.Write("MeanSquare(FL): {0:f2} [dB]\t\t MeanSquare(FR):{1:f2} [dB]\r", 10 * Math.Log10(GetMeanSquare(frontLeftData)), 10 * Math.Log10(GetMeanSquare(frontRightData)));
                }
            }

            private double[] GetChannelData(byte[] buffer, int length, int channel)
            {
                Debug.Assert(channel >= 0 && channel < _channelCount);
                double[] data = new double[length / _channelCount / ByteSizePerSample];
                switch (_bitDepth)
                {
                    case 16:
                        return data.Select((value, index) => value = (double)BitConverter.ToInt16(buffer, ByteSizePerSample * (index * _channelCount + channel)) / (1 << (_bitDepth - 1))).ToArray();
                    default:
                        // サンプルでは、ビット深度 16bit のもののみ扱います。
                        Debug.Assert(true);
                        return null;
                }
            }

            private double GetMeanSquare(double[] data)
            {
                int length = data.Length;
                if(length <= 0)
                {
                    return double.NegativeInfinity;
                }
                double sum = 0;
                foreach (var value in data)
                {
                    sum += value * value;
                }
                return sum / length;
            }

            private void OnCaptureStopped(object sender, SystemAudioCaptureStoppedEventArgs e)
            {
                var exception = e.Exception;
                if (exception != null)
                {
                    _isCaptureTerminated = true;
                    Console.WriteLine("{0}", exception.Message);
                }
            }

            public bool Run(string[] args)
            {
                // デフォルトのターゲットの波形をキャプチャします。
                // キャプチャを開始するためには、事前にターゲットと接続を確立する必要があります。
                SystemAudioCapture capture = new SystemAudioCapture();
                var format = capture.Format;

                // キャプチャの設定を取得します。
                _bitDepth = capture.Format.BitDepth;
                _channelCount = capture.Format.ChannelCount;

                // 波形キャプチャ時、停止時のイベントハンドラを登録します。
                capture.DataAvailable += OnDataAvailable;
                capture.CaptureStopped += OnCaptureStopped;

                // キャプチャをスタートします。
                while (true)
                {
                    try
                    {
                        capture.Start();
                        break;
                    }
                    catch (Exception e)
                    {
                        Console.WriteLine("{0}", e.Message);
                        Console.WriteLine("Press Enter to retry, or Esc to exit.");
                        ConsoleKey key;
                        while ((key = Console.ReadKey().Key) != ConsoleKey.Enter)
                        {
                            if (key == ConsoleKey.Escape)
                            {
                                return false;
                            }
                        }
                        continue;
                    }
                }

                // キーが入力されるまでの間、キャプチャを続けます。
                Console.WriteLine("Capture started.");
                while (!Console.KeyAvailable && !_isCaptureTerminated)
                {
                    System.Threading.Thread.Sleep(50);
                }
                capture.Stop();

                return true;
            }
        }

        static int Main(string[] args)
        {
            return (new ConsoleApplication().Run(args) ? 0 : 1);
        }
    }
}
