﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AudioSink.cpp,PageSampleAudioAudioSink}
 *
 * @brief
 * オーディオレンダラの Sink についてのサンプルプログラム
 */

/**
 * @page PageSampleAudioAudioSink オーディオレンダラの Sink についての解説
 * @tableofcontents
 *
 * @brief
 *  オーディオレンダラの Sink について解説するサンプルプログラムです。
 *
 * @section PageSampleAudioAudioSink_SectionBrief 概要
 * オーディオレンダラのレンダリング結果を出力する先となる各 Sink について解説するサンプルです。
 * とくに CircularBufferSinkType の使い方にフォーカスした解説が含まれています。
 *
 * @section PageSampleAudioAudioSink_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/AudioSink
 * Samples/Sources/Applications/AudioSink @endlink 以下にあります。
 *
 * @section PageSampleAudioAudioSink_SectionNecessaryEnvironment 必要な環境
 * オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleAudioAudioSink_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、以下に説明する操作方法が出力され、すべての SE が一度再生されます。
 *
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> [A]             </td><td> 左チャンネルから SampleSe0 を一度再生 </td></tr>
 * <tr><td> [B]             </td><td> 右チャンネルから SampleSe1 を一度再生 </td></tr>
 * <tr><td> [X]             </td><td> 左チャンネルから SampleSe2 を一度再生 </td></tr>
 * <tr><td> [Y]             </td><td> 右チャンネルから SampleSe3 を一度再生 </td></tr>
 * <tr><td> [Start/Space]   </td><td> プログラムの終了                      </td></tr>
 * </table>
 * </p>
 * A / B / X / Y のボタンを押すことで、対応する SE が左右どちらかのチャンネルから再生されます。
 * この際、それぞれのチャンネルの出力レベルが操作方法の下にログ出力されます。
 *
 * @section PageSampleAudioAudioSink_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAudioAudioSink_SectionDetail 解説
 * オーディオレンダラは波形のレンダリング処理を行い、最終的に FinalMixType のミックスバッファにそのサンプルデータが出力されます。@n
 * 各 Sink は FinalMixType のミックスバッファに含まれるサンプルデータを受け取り、それぞれの Sink の機能に応じた処理を行います。
 *
 * このサンプルでは主に CircularBufferSinkType の解説を行います。
 * CircularBufferSinkType はオーディオレンダラのレンダリング結果をメモリ上に書き出す機能を提供する Sink です。
 * この機能を用いて録音や描画等、サンプルデータを利用したデバッグ、ツール開発に利用することができます。
 * このサンプルでは CircularBufferSinkType から得られたサンプルデータを処理して、
 * オーディオレンダラが出力している音圧レベルを表示するプログラムの解説をしています。
 *
 * このプログラムで解説する内容の要点は以下の通りです。
 * - オーディオレンダラの初期化時に、必要な nn::audio::AudioRendererParameter.sinkCount に必要な数を指定する
 * - CircularBufferSinkType が必要とするメモリのサイズを取得する
 * - CircularBufferSinkType を FinalMixType に追加する
 * - CircularBufferSinkType からサンプルデータを読み込む
 */

#include <cstdlib>
#include <cmath>
#include <algorithm>

#include <nns/nns_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/nn_TimeSpan.h>

#include <nn/audio.h>
#include <nns/audio/audio_HidUtilities.h>
#include <nns/audio/audio_WavFormat.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>

#include <nn/settings/settings_DebugPad.h>

namespace {

// choose rendering engine sample rate
//const int RenderRate = 48000;
const int RenderRate = 32000;
const int RenderCount = (RenderRate / 200);

// choose number of files to play
const int SeCount = 4;

const char Title[] = "AudioSink";

const char* g_SeFileNames[SeCount] =
{
    "asset:/AudioCommon/SampleSe0.adpcm",
    "asset:/AudioCommon/SampleSe1.adpcm",
    "asset:/AudioCommon/SampleSe2.adpcm",
    "asset:/AudioCommon/SampleSe3.adpcm",
};

NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_CircularBufferPoolMemory[12 * 1024 * 1024];

nn::mem::StandardAllocator g_Allocator;
nn::mem::StandardAllocator g_WaveBufferAllocator;
nn::mem::StandardAllocator g_CircularBufferAllocator;

char* g_MountRomCacheBuffer = NULL;

}

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *adpcmData = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

std::size_t ReadWavFile(nns::audio::WavFormat* format, void** data, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *data = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*data);

    // DATA チャンクを読む必要がありますが、ここではそれが 1024 バイト以内に見つかると仮定しています
    const std::size_t WavHeaderDataSize = 1024;

    result = nn::fs::ReadFile(handle, 0, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS(result.IsSuccess());

    nns::audio::WavResult wavResult = nns::audio::ParseWavFormat(format, *data, WavHeaderDataSize);
    NN_ABORT_UNLESS_EQUAL(wavResult, nns::audio::WavResult_Success);
    NN_ABORT_UNLESS_EQUAL(format->channelCount, 1);    // このサンプルでは 1ch を仮定しています
    NN_ABORT_UNLESS_EQUAL(format->bitsPerSample, 16);  // このサンプルでは 16bit PCM を仮定しています

    result = nn::fs::ReadFile(handle, static_cast<std::size_t>(format->dataOffset), *data, static_cast<std::size_t>(format->dataSize));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    return static_cast<std::size_t>(format->dataSize);
}

void InitializeFileSystem()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA     = nn::hid::KeyboardKey::A::Index;
    map.buttonB     = nn::hid::KeyboardKey::B::Index;
    map.buttonX     = nn::hid::KeyboardKey::X::Index;
    map.buttonY     = nn::hid::KeyboardKey::Y::Index;
    map.buttonL     = nn::hid::KeyboardKey::L::Index;
    map.buttonR     = nn::hid::KeyboardKey::R::Index;
    map.buttonLeft  = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp    = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown  = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

nn::hid::NpadButtonSet GetPadButtonDown()
{
    nn::hid::NpadButtonSet npadButtonDown = {};

    // Npad の入力を取得します。
    if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
    {
        static nn::hid::NpadFullKeyState npadFullKeyState = {};
        nn::hid::NpadFullKeyState state;
        nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
        npadButtonDown |= state.buttons & ~npadFullKeyState.buttons;
        npadFullKeyState = state;
    }
    if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
    {
        static nn::hid::NpadHandheldState npadHandheldState = {};
        nn::hid::NpadHandheldState state;
        nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
        npadButtonDown |= state.buttons & ~npadHandheldState.buttons;
        npadHandheldState = state;
    }

    // DebugPad の入力を取得します。
    {
        static nn::hid::DebugPadState debugPadState = {};
        nn::hid::DebugPadButtonSet debugPadButtonDown = {};
        nn::hid::DebugPadState state;
        nn::hid::GetDebugPadState(&state);
        debugPadButtonDown |= state.buttons & ~debugPadState.buttons;
        debugPadState = state;
        nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonDown, debugPadButtonDown);
    }

    return npadButtonDown;
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("[A]             StartSound (Left ch.)   (SampleSe0)\n");
    NNS_LOG("[B]             StartSound (Right ch.)  (SampleSe1)\n");
    NNS_LOG("[X]             StartSound (Left ch.)   (SampleSe2)\n");
    NNS_LOG("[Y]             StartSound (Right ch.)  (SampleSe3)\n");
    NNS_LOG("[Start/Space]   Shut down sample program\n");
    NNS_LOG("-------------------------------------------------------\n");
}

extern "C" void nnMain()
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    g_CircularBufferAllocator.Initialize(g_CircularBufferPoolMemory, sizeof(g_CircularBufferPoolMemory));

    InitializeFileSystem();
    InitializeHidDevices();

    // オーディオレンダラの基本的な解説は @link ../../../Samples/Sources/Applications/AudioRenderer Samples/Sources/Applications/AudioRenderer @endlink サンプルを参照してください。
    // ここでは 2 つのミックスバッファを持つ FinalMixType に、 4 つの VoiceType を直接入力する構造を準備します。
    // その過程で FinalMixType に対して CircularBufferSinkType を追加します。
    // CircularBufferSinkType は DeviceSinkType と同様に SinkType の 1 つです。
    // DeviceSinkType と併用、つまり音をデバイスから再生させながら CircularBufferSinkType も利用する場合は、
    // nn::audio::AudioRendererParameter.sinkCount に 2 を指定する必要があります。
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 2; // FinalMixType(2)
    parameter.voiceCount = 4;
    parameter.subMixCount = 0;
    parameter.sinkCount = 2;  // DeviceSinkType(1) + CircularBufferSinkType(1)
    parameter.effectCount = 0;
    parameter.performanceFrameCount = 0;

    const int channelCount = 2;
    int8_t mainBus[2];
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 0;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 1;
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
    );

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::os::SystemEvent systemEvent;
    nn::audio::AudioRendererHandle handle;
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize).IsSuccess(),
        "Failed to open AudioRenderer"
    );

    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);
    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 2);

    // オーディオレンダラのレンダリング結果をメモリに書き出すための Sink である CircularBufferSinkType を FinalMixType に追加します。
    // 追加にあたってはまず  CircularBufferSinkType に必要なメモリを準備する必要があります。
    // CircularBufferSinkType で必要となるメモリは nn::audio::GetRequiredBufferSizeForCircularBufferSink() でそのサイズを取得できます。
    // nn::audio::GetRequiredBufferSizeForCircularBufferSink() への入力として、「入力チャンネル数」 「オーディオフレーム数」の 2 点を事前に決定しておく必要があります。
    // - 「入力チャンネル数」 について
    //   CircularBufferSinkType は MixBufferCountMax 以下の任意の数のミックスバッファを入力にとることができます。
    //   このサンプルでは FinalMixType が持つ 2 つのミックスバッファを CircularBufferSinkType への入力とします。
    const int circularBufferSinkInputCount = channelCount;
    const int8_t circularBufferInputs[channelCount] = { 0, 1 };
    // - 「オーディオフレーム数」について
    //   CircularBufferSinkType が管理するメモリへのオーディオレンダラからの書き出しは、オーディオフレーム単位で行われます。
    //   つまり (nn::audio::AudioRendererParameter.sampleCount) * (CircularBufferSinkType への入力チャンネル数) のサンプル数単位でサンプルデータが書き出されます。
    //   CircularBufferSinkType に保持されたサンプルデータの読み込みが遅延し、 読み込まれていないサンプルデータサイズが CircularBufferSinkType が保持するバッファサイズを上回った場合、 古いサンプルデータは破棄され、新しいサンプルデータでバッファが更新されます。
    //   そのため CircularBufferSinkType に持たせるオーディオフレーム数は自由に設定することができますが、ここフレーム数が CircularBufferSinkType が保持できる最大サンプルデータ数を決定することに注意してください。
    //   もし読み込まれるサンプルデータに音途切れがある場合は、オーディオフレーム数を増やすか、読み込み頻度を上げることを検討してください。
    const int circularBufferSinkFrameCount = 10;
    size_t circularBufferSinkBufferSize = nn::audio::GetRequiredBufferSizeForCircularBufferSink(&parameter, circularBufferSinkInputCount, circularBufferSinkFrameCount, nn::audio::SampleFormat_PcmInt16);

    // CircularBufferSinkType で用いるメモリは、メモリプールに含まれている必要があります。
    // そのためここでは、メモリプールで必要なメモリアドレス・サイズに合わせてアライメントを取って、バッファを確保します。
    void* circularBufferSinkBuffer = g_CircularBufferAllocator.Allocate(nn::util::align_up(circularBufferSinkBufferSize, nn::audio::MemoryPoolType::SizeGranularity), nn::audio::MemoryPoolType::AddressAlignment);
    NN_ABORT_UNLESS_NOT_NULL(circularBufferSinkBuffer);
    nn::audio::MemoryPoolType circularBufferMemoryPool;
    auto rt = nn::audio::AcquireMemoryPool(&config, &circularBufferMemoryPool, circularBufferSinkBuffer, nn::util::align_up(circularBufferSinkBufferSize, nn::audio::MemoryPoolType::SizeGranularity));
    NN_ABORT_UNLESS(rt);
    rt = nn::audio::RequestAttachMemoryPool(&circularBufferMemoryPool);
    NN_ABORT_UNLESS(rt);

    // CircularBufferSinkType の追加は nn::audio::AddCircularBufferSink() で行います。
    // - オーディオレンダラはこの関数の引数 buffer に指定するメモリに対して、引数 bufferSize の大きさを上限に、レンダリング結果を出力します。
    //   ここではそれぞれ circularBufferSinkBuffer および circularBufferSinkBufferSize を指定しています。
    // - buffer に対して出力される内容は、引数 input で指定するミックスバッファインデックスの内容です。 出力は input[0] 、 input[1] 、 ... の順にブロックインターリーブされて出力されます。(出力結果の詳細な読み込み手順は後述します。)
    //   ここでは circularBufferInputs を指定しています。
    // - 引数 sampleFormat には buffer に出力するサンプルフォーマットを指定します。現在のリリースでは nn::audio::SampleFormat_PcmInt16 のみサポートされています。
    nn::audio::CircularBufferSinkType circularBufferSink;
    nn::Result result = nn::audio::AddCircularBufferSink(&config, &circularBufferSink, &finalMix, circularBufferInputs, circularBufferSinkInputCount, circularBufferSinkBuffer, circularBufferSinkBufferSize, nn::audio::SampleFormat_PcmInt16);
    NN_ABORT_UNLESS(result.IsSuccess());

    // 音声をデバイスに出力するための Sink である DeviceSinkType を FinalMixType に追加します。
    // DeviceSinkType と CircularBufferSinkType は併用が可能です。ただし併用する場合は nn::audio::AudioRendererParameter.sinkCount に 2 以上の値を設定してオーディオレンダラを取得してください。
    nn::audio::DeviceSinkType deviceSink;
    result = nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut");
    NN_ABORT_UNLESS(result.IsSuccess());

    // 設定したパラメータをレンダラに反映しレンダリングを開始します。
    result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
    NN_ABORT_UNLESS(result.IsSuccess());
    result = nn::audio::StartAudioRenderer(handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    // 4 つの効果音 VoiceType をオーディオレンダラに追加します。
    // このサンプルでは、これらの 4 つの効果音をすべてを FinalMix に直接流し込みます。
    // 効果音は Left ないしは Right のどちらか片方から再生されるように nn::audio::SetVoiceMixVolume() でボリュームを調整しています。
    nn::audio::MemoryPoolType waveBufferMemoryPool;
    rt = nn::audio::AcquireMemoryPool(&config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = nn::audio::RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(rt);
    nn::audio::VoiceType voiceSe[SeCount];
    nn::audio::WaveBuffer waveBufferSe[SeCount];
    nn::audio::AdpcmHeaderInfo* header[SeCount];
    void* dataSe[SeCount];

    for (int i = 0; i < SeCount; ++i)
    {
        header[i] = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(g_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
        std::size_t dataSeSize = ReadAdpcmFile(header[i], &dataSe[i], g_SeFileNames[i]);
        nn::audio::AcquireVoiceSlot(&config, &voiceSe[i], header[i]->sampleRate, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &header[i]->parameter, sizeof(nn::audio::AdpcmParameter));
        nn::audio::SetVoiceDestination(&config, &voiceSe[i], &finalMix);

        waveBufferSe[i].buffer = dataSe[i];
        waveBufferSe[i].size = dataSeSize;
        waveBufferSe[i].startSampleOffset = 0;
        waveBufferSe[i].endSampleOffset = header[i]->sampleCount;
        waveBufferSe[i].loop = false;
        waveBufferSe[i].isEndOfStream = false;
        waveBufferSe[i].pContext = &header[i]->loopContext;
        waveBufferSe[i].contextSize = sizeof(nn::audio::AdpcmContext);

        nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
        nn::audio::SetVoicePlayState(&voiceSe[i], nn::audio::VoiceType::PlayState_Play);
        int ch = (i % 2);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 1.0f, 0, mainBus[ch]);
    }


    // CircularBufferSinkType のメモリへの出力結果を、読み込む先のバッファを準備します。読み込み処理は後述する nn::audio::ReadCircularBufferSink() を用いて行います。
    // 読み込み先バッファには任意のサイズのバッファを取ることができますが、上述の通り CircularBufferSinkType が管理するメモリへのオーディオレンダラからの書き出しは、オーディオフレーム単位で行われるため
    // (nn::audio::AudioRendererParameter.sampleCount) * (CircularBufferSinkType への入力チャンネル数) * (CircularBufferSinkType が出力する smpleFormat のサイズ) の整数倍となる大きさに指定すると、
    // ブロックインターリーブの途中でデータが途切れることが無いため、取扱いが簡単になります。
    // nn::audio::GetRequiredBufferSizeForCircularBufferSink() の返り値は、この条件を満たしているため、読み込み先バッファのサイズの指定にも利用することができます。
    // ここでは circularBufferSink が保持しているバッファと同サイズの読み込み先バッファを準備します。
    size_t resultDataSize = circularBufferSinkBufferSize;
    int16_t* resultData = reinterpret_cast<int16_t*>(g_CircularBufferAllocator.Allocate(resultDataSize));
    NN_ABORT_UNLESS_NOT_NULL(resultData);


    PrintUsage();

    for (;;)
    {
        systemEvent.Wait();

        auto buttonDown = GetPadButtonDown();

        //SE を再生します（同じ SE を多重再生しません）。
        for (int i = 0; i < SeCount; ++i)
        {
            // SE番号 {0, 1, 2, 3, 4, 5} が、ボタン {A, B, X, Y, L, R} に対応します。
            if(buttonDown.Test(i))
            {
                if (nn::audio::GetReleasedWaveBuffer(&voiceSe[i]))
                {
                    nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
                }
            }
        }

        // CircularBufferSinkType によってメモリに書き出されたレンダリングを読み込みます。
        // 読み込みには nn::audio::ReadCircularBufferSink() を利用します。
        // この関数は CircularBufferSinkType が保持する読み込まれていないデータを pOutBuffer 引数に書き出し、書き出したサイズを返り値として返します。
        // ここでは pOutBuffer 引数に resultData を指定しています。
        size_t leftSize = nn::audio::ReadCircularBufferSink(&circularBufferSink, resultData, resultDataSize);

        // resultData に読み込まれたデータは、ブロックインターリーブされています。
        // 1 ブロックには 1 オーディオフレーム 1 ミックスバッファ分のサンプルデータが含まれています。
        // つまり nn::audio::AudioRendererParameter.sampleCount で指定したサンプル数のデータが含まれています。
        const int sampleCountPerBlock = parameter.sampleCount;
        while (leftSize > 0)
        {
            // resultData に読み込まれたデータを用いて、波形表示やファイルへの書き出し等、任意の処理を行うことができます。
            // このサンプルでは、各入力バッファ毎のボリュームレベルを計算し表示します。
            float squareMean[circularBufferSinkInputCount] = { 0 };

            // nn::audio::AddCircularBufferSink() での追加時に inputCount 引数に circularBufferSinkInputCount = 2 を指定したので、
            // 読み込み結果には 2 つのブロックインタリーブされたデータが含まれています。
            for (int i = 0; i < circularBufferSinkInputCount; ++i)
            {
                int64_t tmp = 0;
                int16_t* buf = &resultData[sampleCountPerBlock * i];
                for (int j = 0; j < sampleCountPerBlock; ++j)
                {
                    tmp += static_cast<int64_t>(buf[j]) * static_cast<int64_t>(buf[j]);
                }
                squareMean[i] = static_cast<float>(tmp) / (1 << 15) / (1 << 15) / sampleCountPerBlock;
            }
            NNS_LOG("MixBuffer0:%8.2f [dB], MixBuffer1:%8.2f [dB]\r", 10 * std::log10(squareMean[0]), 10 * std::log10(squareMean[1]));

            leftSize -= circularBufferSinkInputCount * sampleCountPerBlock * nn::audio::GetSampleByteSize(nn::audio::SampleFormat_PcmInt16);
        }

        result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
        NN_ABORT_UNLESS(result.IsSuccess());

        // サンプルを終了します。
        if (buttonDown.Test< ::nn::hid::NpadButton::Plus >())
        {
            NNS_LOG("\nClosing program.\n");
            break;
        }
    }

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    // メモリを解放します。
    for (int i = 0; i < SeCount; ++i)
    {
        if (dataSe[i])
        {
            g_WaveBufferAllocator.Free(dataSe[i]);
            dataSe[i] = nullptr;
            g_WaveBufferAllocator.Free(header[i]);
            header[i] = nullptr;
        }
    }
    if (configBuffer)
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if (workBuffer)
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }

    FinalizeFileSystem();
}  // NOLINT(readability/fn_size)
