﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AudioCpuRenderer.cpp,PageSampleAudioAudioCpuRenderer}
 *
 * @brief
 * オーディオ CPU レンダラのサンプルプログラム
 */

/**
 * @page PageSampleAudioAudioCpuRenderer オーディオ CPU レンダラ
 * @tableofcontents
 *
 * @brief
 * オーディオ CPU レンダラのサンプルプログラムの解説です。
 *
 * @section PageSampleAudioAudioCpuRenderer_SectionBrief 概要
 * オーディオ CPU レンダラを利用して、オーディオ再生を行うサンプルです。
 *
 * @section PageSampleAudioAudioCpuRenderer_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/AudioCpuRenderer
 * Samples/Sources/Applications/AudioCpuRenderer @endlink 以下にあります。
 *
 * @section PageSampleAudioAudioCpuRenderer_SectionNecessaryEnvironment 必要な環境
 * オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleAudioAudioCpuRenderer_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、自動的にサイン波が再生されます。
 * 再生中はキーボードや DebugPad による入力を用いて、再生される SE、サイン波を操作できます。
 *
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> [A]             </td><td> SampleSe0 を一度再生                           </td></tr>
 * <tr><td> [B]             </td><td> SampleSe1 を一度再生                           </td></tr>
 * <tr><td> [X]             </td><td> SampleSe2 を一度再生                           </td></tr>
 * <tr><td> [Y]             </td><td> SampleSe3 を一度再生                           </td></tr>
 * <tr><td> [AnalogStickL]  </td><td> サイン波の音量を大小操作                       </td></tr>
 * <tr><td> [AnalogStickR]  </td><td> サイン波のピッチを上下操作                     </td></tr>
 * <tr><td> [Start/Space]   </td><td> プログラムの終了                               </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleAudioAudioCpuRenderer_SectionPrecaution 注意事項
 * 本サンプルはサンプルレート 32000 Hz もしくは 48000 Hz に対応した音声出力デバイスが搭載された環境でのみ実行可能です。
 *
 * @section PageSampleAudioAudioCpuRenderer_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAudioAudioCpuRenderer_SectionDetail 解説
 * このサンプルプログラムは、アプリケーションが動作する CPU 上でオーディオレンダリングを行うものです。
 * オーディオレンダリングした結果は CircularBufferSink により取り出し、オーディオ出力に流し再生を行います。
 *
 * 本サンプルは、アプリケーションが動作する CPU 上でオーディオレンダリングを行うこと以外は、
 * AudioRenderer サンプル・AudioSink サンプル・AudioOut サンプルで説明されている機能を用いています。
 * オーディオレンダラやオーディオ出力の基本的な使い方については、これらのサンプルをご確認ください。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - ファイルシステムの初期化
 * - オーディオレンダラ・オーディオ出力の初期化・準備
 * - オーディオ出力の初期化・準備
 * - オーディオレンダリングの開始
 * - パラメータ変更、状態取得の実施
 * - オーディオレンダリング処理の実行
 * - レンダリング結果をオーディオ出力に流す
 * - レンダリングを終了
 * - オーディオレンダラ・オーディオ出力をクローズ
 * - 入力波形を破棄
 *
 * 最初にファイルシステムの初期化を行います。
 * リソースファイルはビルド時にコンバートされるものを利用します。
 * リソースディレクトリの場所は適宜変更して利用してください。
 *
 * オーディオレンダラを初期化します。
 * 本サンプルではレンダリング処理を実行するデバイスに CPU (nn::audio::AudioRendererRenderingDevice_Cpu)、
 * レンダリング実行モードに手動実行モード (nn::audio::AudioRendererExecutionMode_ManualExecution) を指定しています。
 * この 2 つのオプションを指定して初期化されたオーディオレンダラは、nn::audio::ExecuteAudioRendererRendering() を呼び出すことで、
 * アプリケーションが動作する CPU 上でオーディオレンダリングを行うことが出来ます。レンダリング処理は呼び出した API 内部で実行されます。
 * 上記機能を機能を有効したオーディオレンダラには DeviceSink を利用することが出来ない制限があります。
 * レンダリング結果を再生するためには、CircularBufferSink でレンダリング結果を取得し、
 * それをオーディオ出力・もしくは別のオーディオレンダラに流す必要があります。
 * 本サンプルではオーディオ出力を用いて、レンダリング結果を再生します。
 *
 * オーディオ出力を初期化します。
 * ダウンミックス処理などを省略するため、オーディオ出力はオーディオレンダラに準備した
 * CircularBufferSink と同じチャンネル数で初期化することを推奨します。
 *
 * メインループでは、オーディオレンダラの更新処理とオーディオレンダリング処理、オーディオ出力への書き込みを行います。
 * nn::audio::ExecuteAudioRendererRendering() を呼び出しオーディオレンダリング処理を行った後、
 * その結果を CircularBufferSink の出力結果であるブロックインターリーブド形式からインターリーブド形式に変換した上で、
 * オーディオ出力に書き込みます。
 *
 * プログラムの終了が要求されると、上記の繰り返し処理を抜けて、再生を停止し、
 * オーディオレンダラ・オーディオ出力をクローズし、メモリの破棄を行います。
 */

#include <cstdlib>
#include <cmath>
#include <new>
#include <algorithm>

#include <nns/nns_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/nn_TimeSpan.h>

#include <nn/audio.h>
#include <nns/audio/audio_HidUtilities.h>
#include <nns/audio/audio_WavFormat.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>

#include <nn/settings/settings_DebugPad.h>

namespace {

// choose number of files to play
const int SeCount = 4;

const char Title[] = "AudioCpuRenderer";

// - and add / remove them to / from the files lists

const char* g_SeFileNames[SeCount] =
{
    "asset:/AudioCommon/SampleSe0.adpcm",
    "asset:/AudioCommon/SampleSe1.adpcm",
    "asset:/AudioCommon/SampleSe2.adpcm",
    "asset:/AudioCommon/SampleSe3.adpcm",
};

NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_CircularBufferPoolMemory[12 * 1024 * 1024];

nn::mem::StandardAllocator g_Allocator;
nn::mem::StandardAllocator g_WaveBufferAllocator;
nn::mem::StandardAllocator g_CircularBufferAllocator;

char* g_MountRomCacheBuffer = NULL;

}

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

std::size_t GenerateSineWave(void** data, int sampleRate, int frequency, int sampleCount)
{
    // g_WaveBufferAllocator が管理するメモリ領域はすべて waveBufferMemoryPool メモリプールに追加されています。
    int16_t* p = static_cast<int16_t*>(g_WaveBufferAllocator.Allocate(sampleCount * sizeof(int16_t), nn::audio::BufferAlignSize));
    NN_ABORT_UNLESS_NOT_NULL(p);
    const float Pi = 3.1415926535897932384626433f;
    for (auto i = 0; i < sampleCount; ++i)
    {
        p[i] = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(2 * Pi * frequency * i / sampleRate));
    }
    *data = p;
    return sampleCount * sizeof(int16_t);
}

std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *adpcmData = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader), nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

void InitializeFileSystem()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA     = nn::hid::KeyboardKey::A::Index;
    map.buttonB     = nn::hid::KeyboardKey::B::Index;
    map.buttonX     = nn::hid::KeyboardKey::X::Index;
    map.buttonY     = nn::hid::KeyboardKey::Y::Index;
    map.buttonL     = nn::hid::KeyboardKey::L::Index;
    map.buttonR     = nn::hid::KeyboardKey::R::Index;
    map.buttonZL    = nn::hid::KeyboardKey::U::Index;
    map.buttonZR    = nn::hid::KeyboardKey::V::Index;
    map.buttonLeft  = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp    = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown  = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("[A]             StartSound                   (SampleSe0)\n");
    NNS_LOG("[B]             StartSound                   (SampleSe1)\n");
    NNS_LOG("[X]             StartSound                   (SampleSe2)\n");
    NNS_LOG("[Y]             StartSound                   (SampleSe3)\n");
    NNS_LOG("[AnalogStickL]  ControlVolume                (SineWave)\n");
    NNS_LOG("[Left][Right]   ControlVolume                (SineWave)\n");
    NNS_LOG("[AnalogStickR]  ControlPitch                 (SineWave)\n");
    NNS_LOG("[Up][Down]      ControlPitch                 (SineWave)\n");
    NNS_LOG("[Start/Space]   Shut down sample program\n");
    NNS_LOG("-------------------------------------------------------\n");
}

template<typename T>
size_t ConvertToInterlevedFormat2ch(T* pDestBuffer, size_t destBufferSize, const T* pSrcBuffer, size_t srcBufferSize, int blockSampleCount) NN_NOEXCEPT
{
    const auto channelCount = 2;
    const auto copySize = std::min(destBufferSize, srcBufferSize);
    const auto sampleCount = copySize / sizeof(T) / channelCount;
    const auto blockCount = sampleCount / blockSampleCount;

    auto blockBuffer0 = &pSrcBuffer[blockSampleCount * 0];
    auto blockBuffer1 = &pSrcBuffer[blockSampleCount * 1];
    const auto strideBlockSampleCount = blockSampleCount * (channelCount - 1);

    for (size_t i = 0; i < blockCount; ++i)
    {
        for (int j = 0; j < blockSampleCount; ++j)
        {
            *pDestBuffer++ = *blockBuffer0++;
            *pDestBuffer++ = *blockBuffer1++;
        }

        blockBuffer0 += strideBlockSampleCount;
        blockBuffer1 += strideBlockSampleCount;
    }

    return copySize;
}

template<typename T>
size_t ConvertToInterlevedFormat6ch(T* pDestBuffer, size_t destBufferSize, const T* pSrcBuffer, size_t srcBufferSize, int blockSampleCount) NN_NOEXCEPT
{
    const auto channelCount = 6;
    const auto copySize = std::min(destBufferSize, srcBufferSize);
    const auto sampleCount = copySize / sizeof(T) / channelCount;
    const auto blockCount = sampleCount / blockSampleCount;

    auto blockBuffer0 = &pSrcBuffer[blockSampleCount * 0];
    auto blockBuffer1 = &pSrcBuffer[blockSampleCount * 1];
    auto blockBuffer2 = &pSrcBuffer[blockSampleCount * 2];
    auto blockBuffer3 = &pSrcBuffer[blockSampleCount * 3];
    auto blockBuffer4 = &pSrcBuffer[blockSampleCount * 4];
    auto blockBuffer5 = &pSrcBuffer[blockSampleCount * 5];
    const auto strideBlockSampleCount = blockSampleCount * (channelCount - 1);

    for (int i = 0; i < blockCount; ++i)
    {
        for (int j = 0; j < blockSampleCount; ++j)
        {
            *pDestBuffer++ = *blockBuffer0++;
            *pDestBuffer++ = *blockBuffer1++;
            *pDestBuffer++ = *blockBuffer2++;
            *pDestBuffer++ = *blockBuffer3++;
            *pDestBuffer++ = *blockBuffer4++;
            *pDestBuffer++ = *blockBuffer5++;
        }

        blockBuffer0 += strideBlockSampleCount;
        blockBuffer1 += strideBlockSampleCount;
        blockBuffer2 += strideBlockSampleCount;
        blockBuffer3 += strideBlockSampleCount;
        blockBuffer4 += strideBlockSampleCount;
        blockBuffer5 += strideBlockSampleCount;
    }

    return copySize;
}


extern "C" void nnMain()
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    g_CircularBufferAllocator.Initialize(g_CircularBufferPoolMemory, sizeof(g_CircularBufferPoolMemory));

    InitializeFileSystem();
    InitializeHidDevices();

    // ============================================================================================
    // #1 オーディオ出力の初期化
    //
    // レンダリング結果を再生するためのオーディオ出力を準備します。
    // ============================================================================================
    int sampleRate;
    const int channelCount = 2;
    nn::audio::AudioOutParameter audioOutParameter;
    nn::audio::AudioOut audioOut;
    nn::os::SystemEvent audioOutSystemEvent;
    {
        nn::audio::InitializeAudioOutParameter(&audioOutParameter);
        audioOutParameter.channelCount = channelCount;
        const auto result = nn::audio::OpenDefaultAudioOut(&audioOut, &audioOutSystemEvent, audioOutParameter);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        sampleRate = nn::audio::GetAudioOutSampleRate(&audioOut);

        if(!(sampleRate == 32000 || sampleRate == 48000))
        {
            NN_ABORT("DefaultAudioOut does not support 32000/48000 Hz as sample rate. This sample only supports 32000/48000 Hz.");
        }
    }


    // =========================================================================================
    // #2. CPU レンダリング指定して AudioRenderer を初期化
    //
    // CPU レンダリングを行うにはレンダリング処理を実行するデバイスに
    // AudioRendererRenderingDevice_Cpu を指定してオーディオレンダラを初期化します。
    // また、本サンプルでは ExecutionAudioRendererRendering() 呼び出しによるレンダリング処理を
    // 行うために、レンダリング処理の実行モードに AudioRendererExecutionMode_ManualExecution を
    // 指定しています。
    // =========================================================================================
    nn::audio::AudioRendererParameter audioRendererParameter;
    nn::audio::InitializeAudioRendererParameter(&audioRendererParameter);
    audioRendererParameter.sampleRate = sampleRate;
    audioRendererParameter.sampleCount = sampleRate / 200;
    audioRendererParameter.mixBufferCount = 6; // FinalMix(6)
    audioRendererParameter.voiceCount = 24;
    audioRendererParameter.subMixCount = 0;
    audioRendererParameter.sinkCount = 1;
    audioRendererParameter.effectCount = 0;
    audioRendererParameter.performanceFrameCount = 0;
    audioRendererParameter.executionMode = nn::audio::AudioRendererExecutionMode_ManualExecution;
    audioRendererParameter.renderingDevice = nn::audio::AudioRendererRenderingDevice_Cpu;

    // ミックスバッファとオーディオバスの関係を定義します。
    int8_t mainBus[2];
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 4;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 5;

    // パラメータがシステムでサポートされているかどうかを確認します。
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(audioRendererParameter),
        "Invalid AudioRendererParameter specified."
    );

    // レンダラのワークバッファを準備します。
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(audioRendererParameter);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::os::SystemEvent audioRendererSystemEvent;

    // レンダラを初期化します。
    nn::audio::AudioRendererHandle handle;
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(&handle, &audioRendererSystemEvent, audioRendererParameter, workBuffer, workBufferSize).IsSuccess(),
        "Failed to open AudioRenderer"
    );

    // AudioRendererConfig を初期化します。
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(audioRendererParameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, audioRendererParameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 6);

    // =======================================================================================
    // #3-1 オーディオデバイスへのレンダリング結果の転送 | CircularBufferSink の準備
    //
    // オーディオレンダラのレンダリング結果を取得するために、CircularBufferSink を準備します。
    // =======================================================================================
    const auto bufferingFrameCount = 16;
    size_t circularBufferSinkBufferSize = nn::audio::GetRequiredBufferSizeForCircularBufferSink(&audioRendererParameter, channelCount, bufferingFrameCount, nn::audio::SampleFormat_PcmInt16);
    void* circularBufferSinkBuffer = g_CircularBufferAllocator.Allocate(nn::util::align_up(circularBufferSinkBufferSize, nn::audio::MemoryPoolType::SizeGranularity), nn::audio::MemoryPoolType::AddressAlignment);
    NN_ABORT_UNLESS_NOT_NULL(circularBufferSinkBuffer);
    nn::audio::MemoryPoolType circularBufferMemoryPool;
    auto rt = nn::audio::AcquireMemoryPool(&config, &circularBufferMemoryPool, circularBufferSinkBuffer, nn::util::align_up(circularBufferSinkBufferSize, nn::audio::MemoryPoolType::SizeGranularity));
    NN_ABORT_UNLESS(rt);
    rt = nn::audio::RequestAttachMemoryPool(&circularBufferMemoryPool);
    NN_ABORT_UNLESS(rt);
    nn::audio::CircularBufferSinkType circularBufferSink;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddCircularBufferSink(&config, &circularBufferSink, &finalMix, mainBus, channelCount, circularBufferSinkBuffer, circularBufferSinkBufferSize, nn::audio::SampleFormat_PcmInt16));

    // 設定したパラメータをレンダラに反映させます。
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handle, &config));

    // レンダリングを開始します。
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handle));

    // Voice を用意します。
    nn::audio::VoiceType voiceSine;
    nn::audio::WaveBuffer waveBufferSine;
    void* dataSine;

    // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
    nn::audio::MemoryPoolType waveBufferMemoryPool;
    rt = nn::audio::AcquireMemoryPool(&config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = nn::audio::RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(rt);

    // Voice で再生する波形を準備します。
    {
        const int sineSampleRate = 32000;
        const int sineFrequency = 440;
        const int sineSampleCount = 32000;
        std::size_t size = GenerateSineWave(&dataSine, sineSampleRate, sineFrequency, sineSampleCount);

        nn::audio::AcquireVoiceSlot(&config, &voiceSine, sineSampleRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
        nn::audio::SetVoiceDestination(&config, &voiceSine, &finalMix);

        waveBufferSine.buffer = dataSine;
        waveBufferSine.size = size;
        waveBufferSine.startSampleOffset = 0;
        waveBufferSine.endSampleOffset = sineSampleCount;
        waveBufferSine.loop = 0;
        waveBufferSine.isEndOfStream = false;

        // 波形を Voice に追加します。
        nn::audio::AppendWaveBuffer(&voiceSine, &waveBufferSine);
        nn::audio::AppendWaveBuffer(&voiceSine, &waveBufferSine);
        nn::audio::AppendWaveBuffer(&voiceSine, &waveBufferSine);
        nn::audio::AppendWaveBuffer(&voiceSine, &waveBufferSine);
        nn::audio::SetVoicePlayState(&voiceSine, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceSine, &finalMix, 0.707f / 2, 0, mainBus[0]);
        nn::audio::SetVoiceMixVolume(&voiceSine, &finalMix, 0.707f / 2, 0, mainBus[1]);
    }


    // SE
    nn::audio::VoiceType voiceSe[SeCount];
    nn::audio::WaveBuffer waveBufferSe[SeCount];
    nn::audio::AdpcmHeaderInfo* header[SeCount];
    void* dataSe[SeCount];

    for (int i = 0; i < SeCount; ++i)
    {
        header[i] = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(g_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
        std::size_t dataSeSize = ReadAdpcmFile(header[i], &dataSe[i], g_SeFileNames[i]);
        nn::audio::AcquireVoiceSlot(&config, &voiceSe[i], header[i]->sampleRate, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &header[i]->parameter, sizeof(nn::audio::AdpcmParameter));
        nn::audio::SetVoiceDestination(&config, &voiceSe[i], &finalMix);

        waveBufferSe[i].buffer = dataSe[i];
        waveBufferSe[i].size = dataSeSize;
        waveBufferSe[i].startSampleOffset = 0;
        waveBufferSe[i].endSampleOffset = header[i]->sampleCount;
        waveBufferSe[i].loop = false;
        waveBufferSe[i].isEndOfStream = false;
        waveBufferSe[i].pContext = &header[i]->loopContext;
        waveBufferSe[i].contextSize = sizeof(nn::audio::AdpcmContext);

        nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
        nn::audio::SetVoicePlayState(&voiceSe[i], nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 0.707f / 2, 0, mainBus[0]);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 0.707f / 2, 0, mainBus[1]);
    }

    // ============================================================================================
    // #3-2 オーディオデバイスへのレンダリング結果の転送 | オーディオ出力の準備
    //
    // CircularBufferSink から取得したレンダリング結果を再生するためのオーディオ出力を準備します。
    // ============================================================================================
    const auto bufferCount = 2;
    const auto resultDataSize = audioRendererParameter.sampleCount * channelCount * sizeof(int16_t) * bufferingFrameCount;
    const auto bufferSize = nn::util::align_up(resultDataSize, nn::audio::AudioOutBuffer::SizeGranularity);
    nn::audio::AudioOutBuffer audioOutBuffer[bufferCount];
    size_t currentResultDataOffset = 0;
    int16_t* pResultData = nullptr;
    {
        // AudioOutBuffer を準備します。
        for (int i = 0; i < bufferCount; ++i)
        {
            // 先に空のバッファを追加しておきます。
            auto buffer = reinterpret_cast<int16_t*>(g_Allocator.Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment));
            memset(buffer, 0, bufferSize);
            NN_ASSERT_NOT_NULL(buffer);

            nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], buffer, bufferSize, resultDataSize);
            nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
        }

        // CircularBufferSink からレンダリング結果を受け取るためのバッファを準備します。
        pResultData = reinterpret_cast<int16_t*>(g_Allocator.Allocate(resultDataSize, nn::audio::AudioOutBuffer::AddressAlignment));
        NN_ASSERT_NOT_NULL(pResultData);

        // オーディオ出力を開始します。
        NN_ABORT_UNLESS(nn::audio::StartAudioOut(&audioOut).IsSuccess());
    }

    PrintUsage();

    for (;;)
    {
        nn::hid::NpadButtonSet npadButtonCurrent = {};
        nn::hid::NpadButtonSet npadButtonDown = {};
        nn::hid::AnalogStickState analogStickStateL = {};
        nn::hid::AnalogStickState analogStickStateR = {};

        // Npad の入力を取得します。
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
        {
            static nn::hid::NpadFullKeyState npadFullKeyState = {};
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadFullKeyState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            npadFullKeyState = state;
        }
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
        {
            static nn::hid::NpadHandheldState npadHandheldState = {};
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadHandheldState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            npadHandheldState = state;
        }

        // DebugPad の入力を取得します。
        {
            static nn::hid::DebugPadState debugPadState = {};
            nn::hid::DebugPadButtonSet debugPadButtonCurrent = {};
            nn::hid::DebugPadButtonSet debugPadButtonDown = {};
            nn::hid::DebugPadState state;
            nn::hid::GetDebugPadState(&state);
            debugPadButtonCurrent |= state.buttons;
            debugPadButtonDown |= state.buttons & ~debugPadState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            debugPadState = state;
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonCurrent, debugPadButtonCurrent);
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonDown, debugPadButtonDown);
        }

        //SE を再生します（同じ SE を多重再生しません）。
        for (int i = 0; i < SeCount; ++i)
        {
            // SE番号 {0, 1, 2, 3, 4, 5} が、ボタン {A, B, X, Y, L, R} に対応します。
            if(npadButtonDown.Test(i))
            {
                if (nn::audio::GetReleasedWaveBuffer(&voiceSe[i]))
                {
                    nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
                }
            }
        }

        //サイン波の音量を操作します。
        float sineVolume = nn::audio::GetVoiceVolume(&voiceSine) + 0.01f * analogStickStateL.x / (::nn::hid::AnalogStickMax + 1);

        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::Right >())
        {
            sineVolume += 0.01f;
        }
        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::Left >())
        {
            sineVolume -= 0.01f;
        }

        // Voice のボリューム・ミックスボリュームは nn::audio::VoiceType::GetVolumeMin() 以上 nn::audio::VoiceType::GetVolumeMax() 以下の値で指定できます。
        // このサンプルではノイズにならぬよう (0.0f, 2.0f) もしくは (0.0f, 1.0f) の範囲に制限しています。
        if (sineVolume < 2.0f && sineVolume > 0.0f)
        {
            nn::audio::SetVoiceVolume(&voiceSine, sineVolume);
        }

        //サイン波のピッチを操作します。
        float sinePitch = nn::audio::GetVoicePitch(&voiceSine) + 0.01f * analogStickStateR.x / (::nn::hid::AnalogStickMax + 1);
        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::Up >())
        {
            sinePitch += 0.01f;
        }
        if(npadButtonCurrent.Test< ::nn::hid::NpadButton::Down >())
        {
            sinePitch -= 0.01f;
        }

        if (sinePitch < nn::audio::VoiceType::GetPitchMax() && sinePitch > nn::audio::VoiceType::GetPitchMin())
        {
            nn::audio::SetVoicePitch(&voiceSine, sinePitch);
        }

        while (const nn::audio::WaveBuffer* pWaveBuffer = nn::audio::GetReleasedWaveBuffer(&voiceSine))
        {
            nn::audio::AppendWaveBuffer(&voiceSine, pWaveBuffer);
        }

        if(npadButtonDown.Test< ::nn::hid::NpadButton::Plus >())
        {
            break;
        }

        // CircularBufferSink からレンダリング結果を取得します。
        currentResultDataOffset += nn::audio::ReadCircularBufferSink(&circularBufferSink,
            pResultData + (currentResultDataOffset / sizeof(int16_t)),
            resultDataSize - currentResultDataOffset);

        // ========================================================================
        // 3. レンダリング処理の開始タイミングの指定
        // ========================================================================

        // AudioRendererConfig の内容をオーディオレンダラに反映します。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handle, &config));

        // オーディオレンダリング処理を実行します。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ExecuteAudioRendererRendering(handle));

        if (currentResultDataOffset < resultDataSize)
        {
            continue;
        }
        NN_ASSERT(currentResultDataOffset == resultDataSize);

        nn::audio::AudioOutBuffer* pReleasedBuffer = nullptr;

        // レンダリング結果を AudioOut に流します。
        for(;;)
        {
            pReleasedBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);

            if(pReleasedBuffer != nullptr)
            {
                const auto releasedBufferBufferSize = nn::audio::GetAudioOutBufferBufferSize(pReleasedBuffer);
                NN_ASSERT(releasedBufferBufferSize >= resultDataSize);

                int16_t* pInterleavedBuffer = reinterpret_cast<int16_t*>(nn::audio::GetAudioOutBufferDataPointer(pReleasedBuffer));

                const auto writtenSize = ConvertToInterlevedFormat2ch(pInterleavedBuffer, releasedBufferBufferSize, pResultData, resultDataSize, audioRendererParameter.sampleCount);
                NN_ASSERT(writtenSize == resultDataSize);
                nn::audio::SetAudioOutBufferInfo(pReleasedBuffer, pInterleavedBuffer, releasedBufferBufferSize, writtenSize);
                nn::audio::AppendAudioOutBuffer(&audioOut, pReleasedBuffer);

                memset(pResultData, 0, resultDataSize);
                currentResultDataOffset = 0;

                break;
            }
            else
            {
                audioOutSystemEvent.Wait();
            }
        }
    }

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(audioRendererSystemEvent.GetBase());

    // オーディオ出力を終了します。
    nn::audio::StopAudioOut(&audioOut);
    nn::audio::CloseAudioOut(&audioOut);
    nn::os::DestroySystemEvent(audioOutSystemEvent.GetBase());

    // メモリを解放します。
    if (pResultData)
    {
        g_Allocator.Free(pResultData);
        pResultData = nullptr;
    }
    if (dataSine)
    {
        g_WaveBufferAllocator.Free(dataSine);
        dataSine = nullptr;
    }
    for (int i = 0; i < SeCount; ++i)
    {
        if (dataSe[i])
        {
            g_WaveBufferAllocator.Free(dataSe[i]);
            dataSe[i] = nullptr;
            g_WaveBufferAllocator.Free(header[i]);
            header[i] = nullptr;
        }
    }
    if(circularBufferSinkBuffer)
    {
        g_CircularBufferAllocator.Free(circularBufferSinkBuffer);
        circularBufferSinkBuffer = nullptr;
    }
    if (configBuffer)
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if (workBuffer)
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }

    FinalizeFileSystem();
}  // NOLINT(readability/fn_size)
