﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AtkStreamSound.cpp,PageSampleAtkStreamSound}
 *
 * @brief
 *  nn::atk ライブラリでストリームサウンドを使用して再生するサンプルプログラム
 */

/**
 * @page PageSampleAtkStreamSound ストリームサウンドの再生
 * @tableofcontents
 *
 * @brief
 *  nn::atk ライブラリでストリームサウンドを使用して再生するサンプルプログラムの解説です。
 *
 * @section PageSampleAtkStreamSound_SectionBrief 概要
 *  nn::atk ライブラリでストリームサウンドを使用して再生するサンプルです。
 *
 * @section PageSampleAtkStreamSound_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/AtkStreamSound Samples/Sources/Applications/AtkStreamSound @endlink 以下にあります。
 *
 * @section PageSampleAtkStreamSound_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleAtkStreamSound_SectionHowToOperate 操作方法
 *  コンソールに操作方法が表示されます。
 *
 * @section PageSampleAtkStreamSound_SectionPrecaution 注意事項
 *  実行中のログが、画面上とコンソールに表示され、
 *  起動直後のログとして操作方法が表示されます。
 *
 *  PC 版ではウィンドウは立ち上がらず、コンソール表示のみになる点にご注意ください。
 *
 * @section PageSampleAtkStreamSound_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAtkStreamSound_SectionDetail 解説
 *
 * @subsection PageSampleAtkStreamSound_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  AtkStreamSound.cpp
 *  @includelineno AtkStreamSound.cpp
 *
 * @subsection PageSampleAtkStreamSound_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの Atk に関連している処理の流れは以下の通りです(AtkSimple と違う部分には追加要素と書かれています)。
 *
 *  - Atk ライブラリの初期化 (InitializeAtk())
 *    - OpusDecoder の初期化(追加要素)
 *    - HardwareOpusDecoder の初期化(追加要素)
 *    - ファイルリードキャッシュに用いるバッファの設定(追加要素)
 *    - AtkSimple で実行している初期化処理
 *  - SoundHeap にウェーブサウンドとシーケンスサウンドの再生に必要な情報をロード (LoadData())
 *  - SoundHeap にプリフェッチデータを用いたストリームサウンドの再生に必要な情報をロード (LoadData()、追加要素)
 *  - メインループでの処理
 *    - キー操作による処理
 *      - サウンドの再生・停止処理を実行
 *    - SoundArchivePlayer の更新
 *  - Atk ライブラリの終了処理 (FinalizeAtk())
 *    - AtkSimple で実行している終了処理
 *    - HardwareOpusDecoder の終了処理(追加要素)
 *    - OpusDecoder の終了処理(追加要素)
 *
 * @subsection PageSampleAtkStreamSound_SectionSampleDetailForOpus Opus 再生に関する補足
 *
 *  通常の nn::atk ライブラリの初期化に加えて、nn::atk::InitializeOpusDecoder() を追加で実行することにより、
 *  あらかじめ登録しておいた Opus フォーマットのファイルを使用するストリームサウンドを再生することができるようになります。
 *  また、Opus デコーダの終了処理は AtkSimple で実行している終了処理の後に実行する必要があります。
 *
 *  ループ再生をする場合には、データ側で事前の準備が必要になる点にご注意ください。
 *  詳しくは、NintendoWare オーディオの圧縮オーディオのドキュメントをご確認ください。
 *
 * @subsection PageSampleAtkStreamSound_SectionSampleDetailForHardwareOpus ハードウェアアクセラレータを利用した Opus の再生に関する補足
 *
 *  通常の nn::atk ライブラリの初期化に加えて、nn::atk::InitializeHardwareOpusDecoder() を追加で実行し、
 *  再生時に nn::atk::SoundStartable::StartInfo によりハードウェアアクセラレータの利用を設定することで、
 *  Opus フォーマットのファイルをハードウェアアクセラレータを利用して、ストリーム再生することができるようになります。
 *
 *  その他の注意点は、 Opus 再生に関する補足の項目と同様になります。
 *  詳しくは、NintendoWare オーディオの圧縮オーディオのページ、およびプログラミングガイドのストリームサウンドのページをご確認ください。
 *
 * @subsection PageSampleAtkStreamSound_SectionSampleDetailForPrefetch プリフェッチ機能に関する補足
 *
 *  ストリームサウンドのプリフェッチ機能を利用する場合、SoundMaker においてプリフェッチを使用する設定にしておく必要があります。
 *  また、プログラム上では SoundHeap にあらかじめプリフェッチデータをロードしておく必要があります。
 *  プリフェッチデータは、ウェーブサウンドやシーケンスサウンドのロードと同様に、nn::atk::SoundDataManager::LoadData()
 *  を使用することでロードすることができます。
 *
 *  SoundMaker 上での設定方法については、SoundMaker のドキュメントをご確認ください。
 *
 * @subsection PageSampleAtkStreamSound_SectionSampleDetailForStreamJump ストリームジャンプ機能に関する補足
 *
 *  ストリームジャンプ機能を利用するためには、リージョン情報を埋め込んだ波形を使ったストリームサウンドを使用する必要があります。
 *  リージョン情報が埋め込まれているストリームサウンドを再生する際に、nn::atk::SoundStartable::StartInfo に、コールバック関数を
 *  設定しておき、コールバックでリージョンの制御を行うことで任意のリージョンを移動しながら再生することができます。
 *
 *  コールバックへの書き方については、StreamRegionCallbackSample1（リージョンをインデックスで指定するサンプル）、
 *  StreamRegionCallbackSample2（リージョンを文字列で指定するサンプル）をご確認ください。
 *  また、プログラミングガイドのストリームサウンドのページの解説も併せてご確認ください。
 *
 *  リージョン情報の埋め込み方については、ガイドのストリームサウンドのページをご確認ください。
 *
 * @subsection PageSampleAtkStreamSound_SectionSampleDetailForFileReadCache ファイルリードキャッシュ機能に関する補足
 *
 * ストリームサウンドでは一定量サウンドデータを読み込み配置する、という処理を繰り返しますが
 * 環境によっては読み込みを細かく繰り返すよりも、一度に大きく読み込む方が読み込みの効率が良くなる場合があります。
 *
 * ファイルリードキャッシュ機能では、別途用意した大きめのバッファにサウンドデータを一度に大きく読み込み、
 * 一定量の再生が終わるとそのバッファから配置を行う処理を繰り返す事で、ファイルアクセスの効率の向上を狙います。
 *
 * ファイルリードキャッシュ機能は、サウンドアーカイブプレイヤーの初期化パラメータに
 * キャッシュ用のバッファの設定を行う事で有効になります。
 *
 * ファイルリードキャッシュの詳細な利用方法については、プログラミングガイドのストリームサウンドのページをご確認ください。
 *
 * @subsection PageSampleAtkStreamSound_SectionSampleDetailForMultiTrack マルチトラック機能に関する補足
 *
 *  ストリームサウンドは、波形を複数のトラックに設定して、同時に再生するといったことができます。
 *
 *  ストリームサウンドのパラメータはストリームサウンド単位で設定できるものと、トラック単位で設定できるものがあり、
 *  音量などのパラメータはトラック単位で設定することができます。
 *  そのため、例えば、ゲームシーンに応じてトラックの音量を調整することで、
 *  あるシーンでは特定のトラックが追加で再生されるようにし、別のシーンでは特定のトラックだけが再生されないようにするといったことが実現できます。
 *
 *  マルチトラックの設定になっているストリームサウンドであっても、トラックが１つしかないストリームサウンドと同じように再生することができます。
 *  ただし、マルチトラックのストリームサウンドの場合には、必要とするストリームチャンネルの数が増える点にご注意ください。
 *  ストリームチャンネルの数の設定はサウンドアーカイブに記録されているため、SoundMaker での変更が必要になります。
 *
 *  詳細はガイドのストリームサウンドのページをご確認ください。
 *
 */

#include "Common.fsid"

#include <nn/atk.h>

#include <nns/atk/atk_SampleCommon.h>

#include <nns/nns_Log.h>

namespace
{
    const char Title[] = "AtkStreamSound";
    const char ArchiveRelativePath[] = "Common.bfsar";

    const int SoundHeapSize = 4 * 1024 * 1024;

    nn::atk::SoundHeap          g_SoundHeap;
    nn::atk::FsSoundArchive     g_SoundArchive;
    nn::atk::SoundArchivePlayer g_SoundArchivePlayer;
    nn::atk::SoundDataManager   g_SoundDataManager;

    nn::audio::MemoryPoolType   g_MemoryPool;

    void* g_pMemoryForSoundSystem;
    void* g_pMemoryForSoundHeap;
    void* g_pMemoryForInfoBlock;
    void* g_pMemoryForSoundDataManager;
    void* g_pMemoryForSoundArchivePlayer;
    void* g_pMemoryForStreamBuffer;
    void* g_pMemoryForStreamCacheBuffer;
    void* g_pMemoryForOpusDecoder;
    void* g_pMemoryForHardwareOpusDecoder;

    nn::atk::SoundHandle        g_SoundHandle;
    nn::atk::SoundHandle        g_SoundHandleHold;

    enum StreamRegion {
        StreamRegionIntro = 0,
        StreamRegionA = 1,
        StreamRegionB = 2
    };
    const int RegionList[] = {
        StreamRegionIntro,
        StreamRegionA,
        StreamRegionA,
        StreamRegionB,
        StreamRegionA,
        StreamRegionB
    };
    int g_RegionListIndex = 0;

    const char* RegionNameList[] =
    {
        "Intro",
        "RegionA",
        "RegionB",
        "RegionA",
        "RegionA",
        "RegionB"
    };
    int g_RegionNameListIndex = 0;
}

// ジャンプ先をインデックスで指定するコールバック処理のサンプル
nn::atk::StreamRegionCallbackResult StreamRegionCallbackSample1( nn::atk::StreamRegionCallbackParam* param, void* args )
{
    NN_UNUSED(args);
    NN_ABORT_UNLESS(g_RegionListIndex >= 0);

    if ( g_RegionListIndex >= sizeof(RegionList) / sizeof(RegionList[0]) )
    {
        NNS_LOG("[%s] g_RegionListIndex(%d) => FINISH\n", __FUNCTION__, g_RegionListIndex);
        return nn::atk::StreamRegionCallbackResult_Finish;
    }

    param->regionNo = RegionList[g_RegionListIndex];
    NN_ABORT_UNLESS_LESS(param->regionNo, param->regionCount);
    NNS_LOG("[%s] g_RegionListIndex(%d) => CONTINUE\n", __FUNCTION__, g_RegionListIndex);

    g_RegionListIndex++;

    return nn::atk::StreamRegionCallbackResult_Continue;
}

// ジャンプ先を文字列で指定するコールバック処理のサンプル
nn::atk::StreamRegionCallbackResult StreamRegionCallbackSample2( nn::atk::StreamRegionCallbackParam* param, void* args)
{
    NN_UNUSED(args);
    NN_ABORT_UNLESS(g_RegionNameListIndex >= 0);

    if (g_RegionNameListIndex >= sizeof(RegionNameList) / sizeof(RegionNameList[0]))
    {
        NNS_LOG("[%s] %s => FINISH\n", __FUNCTION__, param->regionName);
        return nn::atk::StreamRegionCallbackResult_Finish;
    }

    param->isRegionNameEnabled = true;
    std::strncpy(param->regionName, RegionNameList[g_RegionNameListIndex], nn::atk::RegionNameLengthMax + 1);
    NNS_LOG("[%s] %s => CONTINUE\n", __FUNCTION__, param->regionName);

    g_RegionNameListIndex++;

    return nn::atk::StreamRegionCallbackResult_Continue;
}

void SetupOpusDecoder()
{
    int decoderCount = 2;
    std::size_t memSizeForOpusDecoder = nn::atk::GetRequiredOpusDecoderBufferSize(decoderCount);
    g_pMemoryForOpusDecoder = nns::atk::Allocate(memSizeForOpusDecoder);
    nn::atk::InitializeOpusDecoder(g_pMemoryForOpusDecoder, memSizeForOpusDecoder, decoderCount);
}

void UnsetupOpusDecoder()
{
    nn::atk::FinalizeOpusDecoder();
    nns::atk::Free(g_pMemoryForOpusDecoder);
}

void SetupHardwareOpusDecoder()
{
    int decoderCount = 2;
    std::size_t memSizeForHardwareOpusDecoder = nn::atk::GetRequiredHardwareOpusDecoderBufferSize(decoderCount);
    g_pMemoryForHardwareOpusDecoder = nns::atk::Allocate(memSizeForHardwareOpusDecoder);
    nn::atk::InitializeHardwareOpusDecoder(g_pMemoryForHardwareOpusDecoder, memSizeForHardwareOpusDecoder, decoderCount);
}

void UnsetupHardwareOpusDecoder()
{
    nn::atk::FinalizeHardwareOpusDecoder();
    nns::atk::Free(g_pMemoryForHardwareOpusDecoder);
}

void Initialize()
{
    nns::atk::InitializeHeap();
    nns::atk::InitializeFileSystem();
    nns::atk::InitializeHidDevices();
}

void Finalize()
{
    nns::atk::FinalizeHidDevices();
    nns::atk::FinalizeFileSystem();
    nns::atk::FinalizeHeap();
}

void InitializeAtk()
{
    bool isSuccess = true;

    // OpusDecoder の初期化
    SetupOpusDecoder();

    // HardwareOpusDecoder の初期化
    SetupHardwareOpusDecoder();

    // SoundSystem の初期化
    nn::atk::SoundSystem::SoundSystemParam param;
    std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );
    g_pMemoryForSoundSystem = nns::atk::Allocate( memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );
    isSuccess = nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>( g_pMemoryForSoundSystem ),
        memSizeForSoundSystem );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundSystem" );

    // SoundHeap の初期化
    g_pMemoryForSoundHeap = nns::atk::Allocate( SoundHeapSize );
    isSuccess = g_SoundHeap.Create( g_pMemoryForSoundHeap, SoundHeapSize );
    NN_ABORT_UNLESS( isSuccess, "cannot create SoundHeap" );

    // SoundArchive の初期化
    const char* archiveAbsolutePath = nns::atk::GetAbsolutePath(ArchiveRelativePath);
    isSuccess = g_SoundArchive.Open(archiveAbsolutePath);
    NN_ABORT_UNLESS( isSuccess, "cannot open SoundArchive(%s)\n", archiveAbsolutePath );

    // SoundArchive のパラメータ情報をメモリにロード
    std::size_t infoBlockSize = g_SoundArchive.GetHeaderSize();
    g_pMemoryForInfoBlock = nns::atk::Allocate(  infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
    isSuccess = g_SoundArchive.LoadHeader( g_pMemoryForInfoBlock, infoBlockSize );
    NN_ABORT_UNLESS( isSuccess, "cannot load InfoBlock" );

    // SoundDataManager の初期化
    std::size_t memSizeForSoundDataManager = g_SoundDataManager.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundDataManager = nns::atk::Allocate( memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize );
    isSuccess = g_SoundDataManager.Initialize(
        &g_SoundArchive,
        g_pMemoryForSoundDataManager,
        memSizeForSoundDataManager );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundDataManager" );

    // SoundArchivePlayer で用いるストリームバッファの初期化
    // ストリームバッファはメモリプール管理されているヒープから確保する必要があります。
    std::size_t memSizeForStreamBuffer = g_SoundArchivePlayer.GetRequiredStreamBufferSize( &g_SoundArchive );
    g_pMemoryForStreamBuffer = nns::atk::AllocateForMemoryPool(memSizeForStreamBuffer);

    // 専用のヒープをメモリプールにアタッチ
    nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPool, nns::atk::GetPoolHeapAddress(), nns::atk::GetPoolHeapSize());

    // ファイルリードキャッシュに用いるバッファの設定
    const int CacheBufferSize = 128 * 1024;
    std::size_t memSizeForStreamCacheBuffer = g_SoundArchivePlayer.GetRequiredStreamCacheSize( &g_SoundArchive, CacheBufferSize );
    g_pMemoryForStreamCacheBuffer = nns::atk::Allocate( memSizeForStreamCacheBuffer, nn::atk::FsSoundArchive::BufferAlignSize );

    // SoundArchivePlayer の初期化
    std::size_t memSizeForSoundArchivePlayer = g_SoundArchivePlayer.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundArchivePlayer = nns::atk::Allocate( memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize);
    nn::atk::SoundArchivePlayer::InitializeParam initializeParam;
    initializeParam.pSoundArchive = &g_SoundArchive;
    initializeParam.pSoundDataManager = &g_SoundDataManager;
    initializeParam.pSetupBuffer = g_pMemoryForSoundArchivePlayer;
    initializeParam.setupBufferSize = memSizeForSoundArchivePlayer;
    initializeParam.pStreamBuffer = g_pMemoryForStreamBuffer;
    initializeParam.streamBufferSize = memSizeForStreamBuffer;
    initializeParam.pStreamCacheBuffer = g_pMemoryForStreamCacheBuffer;
    initializeParam.streamCacheSize = memSizeForStreamCacheBuffer;
    isSuccess = g_SoundArchivePlayer.Initialize(initializeParam);
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundArchivePlayer" );

    // STRM のマーカー情報の取得とダンプのサンプルコード
    const int MarkerInfoCountMax = 16;     // 大きめのサイズを指定
    nn::atk::StreamSoundMarkerInfo markerInfos[MarkerInfoCountMax];
    int actualMarkerInfoCount = 0;
    g_SoundArchivePlayer.ReadMarkerInfoArray(markerInfos, &actualMarkerInfoCount, MarkerInfoCountMax, STRM_MARIOKART_STREAMJUMP);
    NN_ABORT_UNLESS(actualMarkerInfoCount <= MarkerInfoCountMax);

    for (auto i = 0; i < actualMarkerInfoCount; ++i)
    {
        NN_LOG("[Marker%d]\n", i);
        NN_LOG("   name = %s\n", markerInfos[i].name);
        NN_LOG("   position = %d\n", markerInfos[i].position);
    }
}

void FinalizeAtk()
{
    g_SoundArchivePlayer.Finalize();

    // 専用のヒープをメモリプールからデタッチ
    nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPool);

    g_SoundDataManager.Finalize();
    g_SoundArchive.Close();
    g_SoundHeap.Destroy();
    nn::atk::SoundSystem::Finalize();

    nns::atk::FreeForMemoryPool(g_pMemoryForStreamBuffer);
    nns::atk::Free(g_pMemoryForSoundArchivePlayer);
    nns::atk::Free(g_pMemoryForSoundDataManager);
    nns::atk::Free(g_pMemoryForInfoBlock);
    nns::atk::Free(g_pMemoryForSoundHeap);
    nns::atk::Free(g_pMemoryForSoundSystem);

    UnsetupHardwareOpusDecoder();
    UnsetupOpusDecoder();
}

void LoadData()
{
    bool isSuccess = true;

    isSuccess = g_SoundDataManager.LoadData( SEQ_MARIOKART, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SEQ_MARIOKART) failed." );

    isSuccess = g_SoundDataManager.LoadData( SE_YOSHI, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_YOSHI) failed." );

    // プリフェッチストリームデータのロード
    // 通常のストリームサウンドにはこの手順は必要ありません
    isSuccess = g_SoundDataManager.LoadData( STRM_MARIOKART_PREFETCH, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(STRM_MARIOKART_PREFETCH) failed." );
}

void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, nn::atk::SoundStartable::StartInfo* pStartInfo, const char* debugLabelName)
{
    g_SoundHandle.Stop( 0 );

    bool result = false;
    if (pStartInfo == NULL)
    {
        result = g_SoundArchivePlayer.StartSound( &g_SoundHandle, soundId ).IsSuccess();
    }
    else
    {
        result = g_SoundArchivePlayer.StartSound( &g_SoundHandle, soundId, pStartInfo ).IsSuccess();
    }

    NNS_LOG("StartSound(%s) ... (%d)\n", debugLabelName, result);
}

void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName)
{
    PlayWithStartSound(soundId, NULL, debugLabelName);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("[A]              StartSound STRM (STRM_MARIOKART_OPUS)[OpusDecoder]\n");
    NNS_LOG("[R+A]            StartSound STRM (STRM_MARIOKART_OPUS)[HardwareOpusDecoder]\n");
    NNS_LOG("[X]              StartSound STRM (STRM_MARIOKART_STREAMJUMP[String])\n");
    NNS_LOG("[R+X]            StartSound STRM (STRM_MARIOKART_STREAMJUMP[Index])\n");
    NNS_LOG("[Y]              StartSound STRM (STRM_MARIOKART)\n");
    NNS_LOG("[R+Y]            StartSound STRM (STRM_MARIOKART_PREFETCH)\n");
    NNS_LOG("[DOWN]           StartSound STRM (STRM_MULTITRACK)\n");
    NNS_LOG("[B]              Stop Sound\n");
    NNS_LOG("[RIGHT]          StartSound SEQ  (SEQ_MARIOKART)\n");
    NNS_LOG("[UP]             StartSound WSD  (SE_YOSHI)\n");
    NNS_LOG("[L]              Print Usage\n");
    NNS_LOG("[+/Start][Space] Exit Application\n");
    NNS_LOG("--------------------------------------------------\n");
}

bool UpdateAtk()
{
    // StartSound / Stop
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >() )
    {
        if (nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
        {
            nn::atk::SoundStartable::StartInfo info;
            info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundMetaInfo;
            const nn::atk::SoundArchive& archive = g_SoundArchivePlayer.GetSoundArchive();
            // ハードウェアアクセラレータを利用した Opus 再生をしたいラベルのメタ情報をコピーします
            archive.ReadStreamSoundInfo(&info.streamSoundMetaInfo, STRM_MARIOKART_OPUS);
            // デコード形式を Opus (ハードウェアアクセラレータによる再生) に変更します
            info.streamSoundMetaInfo.decodeMode = nn::atk::SoundArchive::DecodeMode_Accelerator;

            PlayWithStartSound(STRM_MARIOKART_OPUS, &info, "STRM_MARIOKART_OPUS[HardwareOpusDecoder]");
        }
        else
        {
            PlayWithStartSound(STRM_MARIOKART_OPUS, "STRM_MARIOKART_OPUS[OpusDecoder]");
        }
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >() )
    {
        if (nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
        {
            g_RegionListIndex = 0;

            nn::atk::SoundStartable::StartInfo info;
            info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;
            info.streamSoundInfo.regionCallback = StreamRegionCallbackSample1;
            info.streamSoundInfo.regionCallbackArg = nullptr;

            PlayWithStartSound(STRM_MARIOKART_STREAMJUMP, &info, "STRM_MARIOKART_STREAMJUMP[Index]");
        }
        else
        {
            g_RegionNameListIndex = 0;

            nn::atk::SoundStartable::StartInfo info;
            info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;
            info.streamSoundInfo.regionCallback = StreamRegionCallbackSample2;
            info.streamSoundInfo.regionCallbackArg = nullptr;

            PlayWithStartSound(STRM_MARIOKART_STREAMJUMP, &info, "STRM_MARIOKART_STREAMJUMP[String]");
        }
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >() )
    {
        if (nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
        {
            PlayWithStartSound(STRM_MARIOKART_PREFETCH, "STRM_MARIOKART_PREFETCH");
        }
        else
        {
            PlayWithStartSound(STRM_MARIOKART, "STRM_MARIOKART");
        }
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >() )
    {
        PlayWithStartSound(STRM_MULTITRACK, "STRM_MULTITRACK");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B >() )
    {
        g_SoundHandle.Stop( 0 );
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
    {
        PlayWithStartSound(SEQ_MARIOKART, "SEQ_MARIOKART");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >() )
    {
        PlayWithStartSound(SE_YOSHI, "SE_YOSHI");
    }

    // Print Usage
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::L >() )
    {
        PrintUsage();
    }

    // Exit
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Start >() )
    {
        return false;
    }

    g_SoundArchivePlayer.Update();

    return true;
}

extern "C" void nnMain()
{
    Initialize();
    InitializeAtk();

    LoadData();

    PrintUsage();

    for ( ;; )
    {
        nns::atk::UpdateHidDevices();

        if ( !UpdateAtk() )
        {
            break;
        }

        // Vsync の代わり
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    FinalizeAtk();
    Finalize();
}

