﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ModuleSelector.h"

#include <cstdio>
#include <nn/nn_Log.h>
#include <nn/os/os_Thread.h>

#include "nns/atk/atk_SampleCommon.h"
#include "CheckCode/ICheckModule.h"

#include "FlagList.h"
#include "GfxCode/DebugViewer.h"

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/atk/fnd/string/atkfnd_String.h>

namespace
{
    // 操作説明のデフォルトの文字列描画位置
    const int DefaultPrintUsagePosX = 800;
    const int DefaultPrintUsagePosY = 0;
    // モジュールリストのデフォルトの文字列描画位置
    const int DefaultPrintModuleListPosX = 650;
    const int DefaultPrintModuleListPosY = 0;
}
#endif

ModuleSelector::ModuleSelector() NN_NOEXCEPT
{
}

ModuleSelector::~ModuleSelector() NN_NOEXCEPT
{
}

ICheckModule* ModuleSelector::Select(ICheckModule* pCheckModuleList[], int moduleNum) NN_NOEXCEPT
{
    PrintMenu(pCheckModuleList, moduleNum);
    NN_LOG("------------------------------------\n");

    int selectedNo = 0;
    int flagIndex = 0;
    NN_LOG("[Up/Down] Select, [A] Run, [Start/Space] Exit, [Left/Right] SelectFlag, [X] ChangeFlag\n");
    PrintSelectedItem(pCheckModuleList, selectedNo, flagIndex);
    for (;;)
    {
        DebugViewer::GetInstance().Reset();
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
        UpdateDraw(pCheckModuleList, moduleNum, selectedNo);
#endif
        DebugViewer::GetInstance().Update();
        DebugViewer::GetInstance().Present();
        nns::atk::UpdateHidDevices();

        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >())
        {
            --selectedNo;
            if (selectedNo < 0)
            {
                selectedNo = moduleNum - 1;
            }

            flagIndex = 0;
            ResetFlags(pCheckModuleList[selectedNo]);

            PrintSelectedItem(pCheckModuleList, selectedNo, flagIndex);
        }
        else if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >())
        {

            ++selectedNo;
            if (selectedNo >= moduleNum)
            {
                selectedNo = 0;
            }

            flagIndex = 0;
            ResetFlags(pCheckModuleList[selectedNo]);

            PrintSelectedItem(pCheckModuleList, selectedNo, flagIndex);
        }
        else if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left>())
        {
            if (pCheckModuleList[selectedNo]->GetLocalFlagList().GetFlagCount() > 0)
            {
                --flagIndex;
                if (flagIndex < 0)
                {
                    flagIndex = pCheckModuleList[selectedNo]->GetLocalFlagList().GetFlagCount() - 1;
                }
                PrintSelectedItem(pCheckModuleList, selectedNo, flagIndex);
            }
        }
        else if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right>())
        {
            if (pCheckModuleList[selectedNo]->GetLocalFlagList().GetFlagCount() > 0)
            {
                ++flagIndex;
                if (flagIndex >= pCheckModuleList[selectedNo]->GetLocalFlagList().GetFlagCount())
                {
                    flagIndex = 0;
                }
                PrintSelectedItem(pCheckModuleList, selectedNo, flagIndex);
            }
        }
        else if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >())
        {
            break;
        }
        else if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >())
        {
            if (pCheckModuleList[selectedNo]->GetLocalFlagList().GetFlagCount() > 0)
            {
                pCheckModuleList[selectedNo]->GetLocalFlagList().SetFlagEnabled(
                    flagIndex,
                    !pCheckModuleList[selectedNo]->GetLocalFlagList().IsFlagEnabled(flagIndex));
                PrintSelectedItem(pCheckModuleList, selectedNo, flagIndex);
            }
        }
        else if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Start >())
        {
            return nullptr;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    return pCheckModuleList[selectedNo];
}

void ModuleSelector::ResetFlags(ICheckModule* pModule) NN_NOEXCEPT
{
    for (auto i = 0; i < pModule->GetLocalFlagList().GetFlagCount(); ++i)
    {
        pModule->GetLocalFlagList().SetFlagEnabled(i, false);
    }
}

void ModuleSelector::PrintItem(ICheckModule* pModule, int index, int flagIndex) NN_NOEXCEPT
{
    NN_LOG(" %2d: %s\n", index, pModule->GetModuleName());
    if (flagIndex >= 0 && pModule->GetLocalFlagList().GetFlagCount() > 0)
    {
        pModule->GetLocalFlagList().ShowFlags(flagIndex, true);
    }
}

void ModuleSelector::PrintMenu(ICheckModule* pCheckModuleList[], int moduleNum) NN_NOEXCEPT
{
    NN_LOG("[Menu]\n");
    for (auto i = 0; i < moduleNum; ++i)
    {
        PrintItem(pCheckModuleList[i], i, -1);
    }
}

void ModuleSelector::PrintSelectedItem(ICheckModule* pCheckModuleList[], int selectedNo, int flagIndex) NN_NOEXCEPT
{
    NN_LOG(">");
    PrintItem(pCheckModuleList[selectedNo], selectedNo, flagIndex);
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void ModuleSelector::UpdateDraw(ICheckModule* pCheckModuleList[], int moduleNum, int selectedNo) NN_NOEXCEPT
{
    // 操作説明の表示
    DebugPrint("Selector_Usage", DefaultPrintUsagePosX, DefaultPrintUsagePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY,
        "\n"
        "[Up/Down]    :Select\n"
        "[A]          :Run\n"
        "[Start/Space]:Exit\n"
        "[Left/Right] :SelectFlag\n"
        "[X]          :ChangeFlag"
    );

    // モジュール一覧表示
    {
        const int StrModuleListLen = 256;
        char strModuleList[StrModuleListLen] = "\n";
        for (auto i = 0; i < moduleNum; ++i)
        {
            auto moduleName = pCheckModuleList[i]->GetModuleName();
            auto moduleNameLen = nn::util::Strnlen( moduleName, StrModuleListLen );
            nn::atk::detail::fnd::strncat( strModuleList, StrModuleListLen, i == selectedNo ? "> " : "  ", 2 );
            nn::atk::detail::fnd::strncat( strModuleList, StrModuleListLen, moduleName, moduleNameLen + 2 );
            if ( i != moduleNum - 1 )
            {
                nn::atk::detail::fnd::strncat( strModuleList, StrModuleListLen, "\n", 2 );
            }
        }
        DebugPrint("ModuleList", DefaultPrintModuleListPosX, DefaultPrintModuleListPosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY, strModuleList);
    }
}
#endif
