﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../AtkSandboxCommon.h"
#include "ICheckModule.h"

namespace
{
    void OnSoundStop()
    {
        NN_LOG("SoundStopped!!\n");
    }
}

class StartInfoCheckModule : public ICheckModule
{
public:
    // ---------------------------------------------------------------------------------------------------------------
    class IStartInfoChecker
    {
    public:
        virtual ~IStartInfoChecker() NN_NOEXCEPT {}

        virtual void Initialize() NN_NOEXCEPT = 0;
        virtual void Finalize() NN_NOEXCEPT = 0;
        virtual void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT = 0;
        virtual void ResetValue() NN_NOEXCEPT = 0;
        virtual void IncrementValue() NN_NOEXCEPT = 0;
        virtual void DecrementValue() NN_NOEXCEPT = 0;
        virtual void PrintValue() NN_NOEXCEPT = 0;
        virtual void PrintCheckerType() NN_NOEXCEPT = 0;

    public:
        void SetCommonObjectPointer(CommonObject* pCommonObject) NN_NOEXCEPT
        {
            m_pCommonObject = pCommonObject;
        }

        CommonObject* GetCommonObjectPointer() const NN_NOEXCEPT
        {
            return m_pCommonObject;
        }

    private:
        CommonObject* m_pCommonObject;
    };

    // ---------------------------------------------------------------------------------------------------------------
    class StartInfoCheckerForStartOffset : public IStartInfoChecker
    {
    public:
        StartInfoCheckerForStartOffset() {}
        virtual ~StartInfoCheckerForStartOffset() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE
        {
            ResetValue();
        }

        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT NN_OVERRIDE
        {
            nn::atk::SoundStartable::StartInfo startInfo;
            startInfo.enableFlag = nn::atk::SoundStartable::StartInfo::EnableFlagBit_StartOffset;
            startInfo.startOffset = m_StartOffset;
            startInfo.startOffsetType = m_StartOffsetType;

            CommonObject::StartParam startParam;
            startParam.pStartInfo = &startInfo;

            GetCommonObjectPointer()->PlayWithStartSound(soundId, debugLabelName, startParam);
        }

        virtual void ResetValue() NN_NOEXCEPT NN_OVERRIDE
        {
            m_StartOffsetType = nn::atk::SoundStartable::StartInfo::StartOffsetType_MilliSeconds;
            m_StartOffset = StartOffsetForMilliSeconds;
        }

        virtual void IncrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
            switch(m_StartOffsetType)
            {
            case nn::atk::SoundStartable::StartInfo::StartOffsetType_MilliSeconds:
                m_StartOffsetType = nn::atk::SoundStartable::StartInfo::StartOffsetType_Tick;
                m_StartOffset = StartOffsetForTick;
                break;
            case nn::atk::SoundStartable::StartInfo::StartOffsetType_Tick:
                m_StartOffsetType = nn::atk::SoundStartable::StartInfo::StartOffsetType_Sample;
                m_StartOffset = StartOffsetForSample;
                break;
            case nn::atk::SoundStartable::StartInfo::StartOffsetType_Sample:
                m_StartOffsetType = nn::atk::SoundStartable::StartInfo::StartOffsetType_MilliSeconds;
                m_StartOffset = StartOffsetForMilliSeconds;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
                break;
            }
        }

        virtual void DecrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
            switch(m_StartOffsetType)
            {
            case nn::atk::SoundStartable::StartInfo::StartOffsetType_MilliSeconds:
                m_StartOffsetType = nn::atk::SoundStartable::StartInfo::StartOffsetType_Sample;
                m_StartOffset = StartOffsetForSample;
                break;
            case nn::atk::SoundStartable::StartInfo::StartOffsetType_Tick:
                m_StartOffsetType = nn::atk::SoundStartable::StartInfo::StartOffsetType_MilliSeconds;
                m_StartOffset = StartOffsetForMilliSeconds;
                break;
            case nn::atk::SoundStartable::StartInfo::StartOffsetType_Sample:
                m_StartOffsetType = nn::atk::SoundStartable::StartInfo::StartOffsetType_Tick;
                m_StartOffset = StartOffsetForTick;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
                break;
            }
        }

        virtual void PrintValue() NN_NOEXCEPT NN_OVERRIDE
        {
            const char* StartOffsetTypeStr[] =
            {
                "MilliSeconds(Always)",
                "Tick(Sequence only)",
                "Sample(Stream and Wave only)"
            };

            NN_LOG("[StartOffsetCheck] startOffsetType = %s, startOffset = %d\n", StartOffsetTypeStr[m_StartOffsetType], m_StartOffset);
        }

        virtual void PrintCheckerType() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("--- StartOffset Check ---\n");
        }

    private:
        static const int StartOffsetForMilliSeconds = 200;
        static const int StartOffsetForTick = 500;
        static const int StartOffsetForSample = 5000;

        nn::atk::SoundStartable::StartInfo::StartOffsetType m_StartOffsetType;
        int m_StartOffset;
    };

    // ---------------------------------------------------------------------------------------------------------------
    // TODO: 実装
    class StartInfoCheckerForPlayerId : public IStartInfoChecker
    {
    public:
        StartInfoCheckerForPlayerId() {}
        virtual ~StartInfoCheckerForPlayerId() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE
        {
            ResetValue();
        }

        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT NN_OVERRIDE
        {
            nn::atk::SoundStartable::StartInfo startInfo;
            startInfo.enableFlag = nn::atk::SoundStartable::StartInfo::EnableFlagBit_PlayerId;
            startInfo.playerId = m_PlayerId;

            CommonObject::StartParam startParam;
            startParam.pStartInfo = &startInfo;

            GetCommonObjectPointer()->PlayWithStartSound(soundId, debugLabelName, startParam);
        }

        virtual void ResetValue() NN_NOEXCEPT NN_OVERRIDE
        {
            m_PlayerId = 0;
        }

        virtual void IncrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
            // TODO: 使い方を調べる
        }

        virtual void DecrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
            // TODO: 使い方を調べる
        }

        virtual void PrintValue() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("[PlayerIdCheck] playerId = %d\n", m_PlayerId);
        }

        virtual void PrintCheckerType() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("--- PlayerId Check ---\n");
        }

    private:

        int m_PlayerId;
    };

    // ---------------------------------------------------------------------------------------------------------------
    // TODO: 実装
    class StartInfoCheckerForPlayerPriority : public IStartInfoChecker
    {
    public:
        StartInfoCheckerForPlayerPriority() {}
        virtual ~StartInfoCheckerForPlayerPriority() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE
        {
            ResetValue();
        }

        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT NN_OVERRIDE
        {
            nn::atk::SoundStartable::StartInfo startInfo;
            startInfo.enableFlag = nn::atk::SoundStartable::StartInfo::EnableFlagBit_PlayerPriority;
            startInfo.playerPriority = m_PlayerPriority;

            CommonObject::StartParam startParam;
            startParam.pStartInfo = &startInfo;

            GetCommonObjectPointer()->PlayWithStartSound(soundId, debugLabelName, startParam);
        }

        virtual void ResetValue() NN_NOEXCEPT NN_OVERRIDE
        {
            m_PlayerPriority = 0;
        }

        virtual void IncrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
            // TODO: 使い方を調べる
        }

        virtual void DecrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
            // TODO: 使い方を調べる
        }

        virtual void PrintValue() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("[PlayerPriorityCheck] playerPriority = %d\n", m_PlayerPriority);
        }

        virtual void PrintCheckerType() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("--- PlayerPriority Check ---\n");
        }

    private:

        int m_PlayerPriority;
    };

    // ---------------------------------------------------------------------------------------------------------------
    // TODO: 実装
    class StartInfoCheckerForActorPlayerId : public IStartInfoChecker
    {
    public:
        StartInfoCheckerForActorPlayerId() {}
        virtual ~StartInfoCheckerForActorPlayerId() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE
        {
            ResetValue();
        }

        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT NN_OVERRIDE
        {
            nn::atk::SoundStartable::StartInfo startInfo;
            startInfo.enableFlag = nn::atk::SoundStartable::StartInfo::EnableFlagBit_ActorPlayerId;
            startInfo.actorPlayerId = m_ActorPlayerId;

            CommonObject::StartParam startParam;
            startParam.pStartInfo = &startInfo;

            GetCommonObjectPointer()->PlayWithStartSound(soundId, debugLabelName, startParam);
        }

        virtual void ResetValue() NN_NOEXCEPT NN_OVERRIDE
        {
            m_ActorPlayerId = 0;
        }

        virtual void IncrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
            // TODO: 使い方を調べる
        }

        virtual void DecrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
            // TODO: 使い方を調べる
        }

        virtual void PrintValue() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("[ActorPlayerIdCheck] actorPlayerId = %d\n", m_ActorPlayerId);
        }

        virtual void PrintCheckerType() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("--- ActorPlayerId Check ---\n");
        }

    private:

        int m_ActorPlayerId;
    };

    // ---------------------------------------------------------------------------------------------------------------
    // SequenceSoundInfo, StreamSoundInfo, WaveSoundInfo, StreamSoundMetaInfo は ExternalFile サンプルに任せる

    // ---------------------------------------------------------------------------------------------------------------
    // TODO: VoiceRendereType のチェックの追加

    // ---------------------------------------------------------------------------------------------------------------
    // FadeFrame は現在無効？？

    // ---------------------------------------------------------------------------------------------------------------
    class StartInfoCheckerForSoundStopCallback : public IStartInfoChecker
    {
    public:
        StartInfoCheckerForSoundStopCallback() {}
        virtual ~StartInfoCheckerForSoundStopCallback() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE
        {
            m_SoundStopCallback = OnSoundStop;
        }

        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT NN_OVERRIDE
        {
            nn::atk::SoundStartable::StartInfo startInfo;
            startInfo.enableFlag = nn::atk::SoundStartable::StartInfo::EnableFlagBit_SoundStopCallback;
            startInfo.soundStopCallback = m_SoundStopCallback;

            CommonObject::StartParam startParam;
            startParam.pStartInfo = &startInfo;

            GetCommonObjectPointer()->PlayWithStartSound(soundId, debugLabelName, startParam);
        }

        virtual void ResetValue() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        virtual void IncrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        virtual void DecrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        virtual void PrintValue() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        virtual void PrintCheckerType() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("--- SoundStopCallback Check ---\n");
        }

    private:

        nn::atk::SoundStopCallback m_SoundStopCallback;
    };

    // ---------------------------------------------------------------------------------------------------------------
    class StartInfoCheckerForDelayTime : public IStartInfoChecker
    {
    public:
        StartInfoCheckerForDelayTime() {}
        virtual ~StartInfoCheckerForDelayTime() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE
        {
            ResetValue();
        }

        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT NN_OVERRIDE
        {
            nn::atk::SoundStartable::StartInfo startInfo;
            startInfo.enableFlag = nn::atk::SoundStartable::StartInfo::EnableFlagBit_DelayTime;
            startInfo.delayTime = m_DelayTime;

            CommonObject::StartParam startParam;
            startParam.pStartInfo = &startInfo;

            GetCommonObjectPointer()->PlayWithStartSound(soundId, debugLabelName, startParam);
        }

        virtual void ResetValue() NN_NOEXCEPT NN_OVERRIDE
        {
            m_DelayTime = DefaultDelayTime;
        }

        virtual void IncrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
            m_DelayTime += 30;
            m_DelayTime = std::min(DelayTimeMax, m_DelayTime);
        }

        virtual void DecrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
            m_DelayTime -= 30;
            m_DelayTime = std::max(DelayTimeMin, m_DelayTime);
        }

        virtual void PrintValue() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("[DelayTimeCheck] delayTime = %d\n", m_DelayTime);
        }

        virtual void PrintCheckerType() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("--- DelayTime Check ---\n");
        }

    private:
        static const int DefaultDelayTime = 300;
        static const int DelayTimeMin = 0;
        static const int DelayTimeMax = 3000;

        int m_DelayTime;
    };

    // ---------------------------------------------------------------------------------------------------------------
    class StartInfoCheckerForDelayCount : public IStartInfoChecker
    {
    public:
        StartInfoCheckerForDelayCount() {}
        virtual ~StartInfoCheckerForDelayCount() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE
        {
            ResetValue();
        }

        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT NN_OVERRIDE
        {
            nn::atk::SoundStartable::StartInfo startInfo;
            startInfo.enableFlag = nn::atk::SoundStartable::StartInfo::EnableFlagBit_DelayCount;
            startInfo.delayCount = m_DelayCount;

            CommonObject::StartParam startParam;
            startParam.pStartInfo = &startInfo;

            GetCommonObjectPointer()->PlayWithStartSound(soundId, debugLabelName, startParam);
        }

        virtual void ResetValue() NN_NOEXCEPT NN_OVERRIDE
        {
            m_DelayCount = DefaultDelayCount;
        }

        virtual void IncrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
            m_DelayCount += 5;
            m_DelayCount = std::min(DelayCountMax, m_DelayCount);
        }

        virtual void DecrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
            m_DelayCount -= 5;
            m_DelayCount = std::max(DelayCountMin, m_DelayCount);
        }

        virtual void PrintValue() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("[DelayCountCheck] delayCount = %d\n", m_DelayCount);
        }

        virtual void PrintCheckerType() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("--- DelayCount Check ---\n");
        }

    private:
        static const int DefaultDelayCount = 10;
        static const int DelayCountMin = 0;
        static const int DelayCountMax = 500;

        int m_DelayCount;
    };

    // ---------------------------------------------------------------------------------------------------------------
    class StartInfoCheckerForUpdateType : public IStartInfoChecker
    {
    public:
        StartInfoCheckerForUpdateType() {}
        virtual ~StartInfoCheckerForUpdateType() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE
        {
            ResetValue();
        }

        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE {}

        virtual void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT NN_OVERRIDE
        {
            nn::atk::SoundStartable::StartInfo startInfo;
            startInfo.enableFlag = nn::atk::SoundStartable::StartInfo::EnableFlagBit_UpdateType;
            startInfo.updateType = m_UpdateType;

            CommonObject::StartParam startParam;
            startParam.pStartInfo = &startInfo;

            GetCommonObjectPointer()->PlayWithStartSound(soundId, debugLabelName, startParam);
        }

        virtual void ResetValue() NN_NOEXCEPT NN_OVERRIDE
        {
            m_UpdateType = nn::atk::UpdateType_AudioFrame;
        }

        virtual void IncrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
            if (m_UpdateType == nn::atk::UpdateType_AudioFrame)
            {
                m_UpdateType = nn::atk::UpdateType_GameFrame;
            }
            else
            {
                m_UpdateType = nn::atk::UpdateType_AudioFrame;
            }
        }

        virtual void DecrementValue() NN_NOEXCEPT NN_OVERRIDE
        {
            if (m_UpdateType == nn::atk::UpdateType_AudioFrame)
            {
                m_UpdateType = nn::atk::UpdateType_GameFrame;
            }
            else
            {
                m_UpdateType = nn::atk::UpdateType_AudioFrame;
            }
        }

        virtual void PrintValue() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("[UpdateTypeCheck] updateType = %s\n", m_UpdateType == nn::atk::UpdateType_AudioFrame ? "AudioFrame" : "GameFrame");
        }

        virtual void PrintCheckerType() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("--- UpdateType Check ---\n");
        }

    private:

        nn::atk::UpdateType m_UpdateType;
    };

public:
    virtual void OnInitializeAtk() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnFinalizeAtk() NN_NOEXCEPT NN_OVERRIDE;

    virtual void OnLoadData() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnPrintUsage() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnUpdateInput() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnUpdateAtk() NN_NOEXCEPT NN_OVERRIDE;
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
    virtual void OnUpdateDraw() NN_NOEXCEPT NN_OVERRIDE;
#endif

    virtual const char* GetModuleName() const NN_NOEXCEPT NN_OVERRIDE
    {
        return "StartInfo";
    }

    virtual FlagList& GetLocalFlagList() NN_NOEXCEPT NN_OVERRIDE;

private:
    IStartInfoChecker* GetCurrentStartInfoChecker() NN_NOEXCEPT;
    void NextChecker() NN_NOEXCEPT;
    void PrevChecker() NN_NOEXCEPT;

private:
    CommonObject m_CommonObject;

    int m_StartInfoCheckerIndex;

    bool m_IsPause;
};
